<?php
defined('BASEPATH') or exit('No direct script access allowed');

class PerformanceMonitor {
    public $CI;

    public function __construct() {
        $this->CI =& get_instance();
        $this->CI->load->model('Cluster_model');
    }

    /**
     * 监控告警处理性能
     */
    public function monitorAlarmProcessing($alarm_id) {
        try {
            $metrics = $this->getAlarmMetrics($alarm_id);
            if ($metrics) {
                $this->recordMetrics('alarm_processing', $metrics);
                $this->checkPerformanceThreshold('alarm_processing', $metrics);
            }
        } catch (Exception $e) {
            log_message('error', 'Failed to monitor alarm processing: ' . $e->getMessage());
        }
    }

    /**
     * 获取告警处理指标
     */
    private function getAlarmMetrics($alarm_id) {
        $sql = "SELECT id, occur_timestamp, COALESCE(handle_time, NOW()) as end_time,
                       TIMESTAMPDIFF(SECOND, occur_timestamp, COALESCE(handle_time, NOW())) as duration
                FROM cluster_alarm_info WHERE id = ?";
        $result = $this->CI->Cluster_model->getList($sql, [$alarm_id]);
        return !empty($result) ? [
            'alarm_id' => $alarm_id,
            'duration' => $result[0]['duration'],
            'timestamp' => time()
        ] : null;
    }

    /**
     * 监控推送性能
     */
    public function monitorPushPerformance($push_type) {
        try {
            $metrics = $this->getPushMetrics($push_type);
            if ($metrics) {
                $this->recordMetrics('push_performance', $metrics);
                $this->checkPerformanceThreshold('push_performance', $metrics);
            }
        } catch (Exception $e) {
            log_message('error', 'Failed to monitor push performance: ' . $e->getMessage());
        }
    }

    /**
     * 获取推送性能指标
     */
    private function getPushMetrics($push_type) {
        $sql = "SELECT COUNT(*) as total,
                       SUM(CASE WHEN content_res = 'success' THEN 1 ELSE 0 END) as success
                FROM cluster_alarm_push_log WHERE push_type = ?
                AND create_at > UNIX_TIMESTAMP(DATE_SUB(NOW(), INTERVAL 5 MINUTE))";
        $result = $this->CI->Cluster_model->getList($sql, [$push_type]);
        if (!empty($result)) {
            return [
                'push_type' => $push_type,
                'total' => $result[0]['total'],
                'success' => $result[0]['success'],
                'success_rate' => $result[0]['total'] > 0 ? ($result[0]['success'] / $result[0]['total']) * 100 : 100,
                'timestamp' => time()
            ];
        }
        return null;
    }

    /**
     * 监控系统资源
     */
    public function monitorSystemResources() {
        try {
            $metrics = $this->getSystemMetrics();
            if ($metrics) {
                $this->recordMetrics('system_resources', $metrics);
                $this->checkPerformanceThreshold('system_resources', $metrics);
            }
        } catch (Exception $e) {
            log_message('error', 'Failed to monitor system resources: ' . $e->getMessage());
        }
    }

    /**
     * 获取系统资源指标
     */
    private function getSystemMetrics() {
        return [
            'cpu_usage' => $this->getCpuUsage(),
            'memory_usage' => $this->getMemoryUsage(),
            'disk_usage' => $this->getDiskUsage(),
            'timestamp' => time()
        ];
    }

    /**
     * 获取 CPU 使用率
     */
    private function getCpuUsage() {
        if (function_exists('sys_getloadavg')) {
            $load = sys_getloadavg();
            return $load[0] * 100 / $this->getCpuCores();
        }
        return 0;
    }

    /**
     * 获取 CPU 核心数
     */
    private function getCpuCores() {
        $cores = 1;
        if (is_file('/proc/cpuinfo')) {
            $cpuinfo = file_get_contents('/proc/cpuinfo');
            preg_match_all('/^processor/m', $cpuinfo, $matches);
            $cores = count($matches[0]);
        }
        return $cores > 0 ? $cores : 1;
    }

    /**
     * 获取内存使用率
     */
    private function getMemoryUsage() {
        if (is_file('/proc/meminfo')) {
            $meminfo = file_get_contents('/proc/meminfo');
            preg_match('/MemTotal:\s+(\d+) kB/i', $meminfo, $matches_total);
            preg_match('/MemFree:\s+(\d+) kB/i', $meminfo, $matches_free);
            
            if (isset($matches_total[1]) && isset($matches_free[1])) {
                $total = $matches_total[1];
                $free = $matches_free[1];
                return ($total - $free) / $total * 100;
            }
        }
        return 0;
    }

    /**
     * 获取磁盘使用率
     */
    private function getDiskUsage() {
        $path = '/';
        if (function_exists('disk_free_space') && function_exists('disk_total_space')) {
            $free = disk_free_space($path);
            $total = disk_total_space($path);
            return $total > 0 ? (($total - $free) / $total) * 100 : 0;
        }
        return 0;
    }

    /**
     * 记录性能指标
     */
    private function recordMetrics($type, $data) {
        $sql = "INSERT INTO cluster_performance_metrics (metric_type, metric_data, create_at) VALUES (?, ?, ?)";
        $this->CI->Cluster_model->updateList($sql, [
            $type,
            json_encode($data),
            time()
        ]);
    }

    /**
     * 检查性能阈值
     */
    private function checkPerformanceThreshold($type, $metrics) {
        $thresholds = $this->getPerformanceThresholds();
        if (isset($thresholds[$type])) {
            $threshold = $thresholds[$type];
            
            switch ($type) {
                case 'alarm_processing':
                    if ($metrics['duration'] > $threshold['max_duration']) {
                        $this->triggerPerformanceAlert($type, '告警处理时间过长', $metrics);
                    }
                    break;
                    
                case 'push_performance':
                    if ($metrics['success_rate'] < $threshold['min_success_rate']) {
                        $this->triggerPerformanceAlert($type, '推送成功率过低', $metrics);
                    }
                    break;
                    
                case 'system_resources':
                    if ($metrics['cpu_usage'] > $threshold['max_cpu_usage']) {
                        $this->triggerPerformanceAlert($type, 'CPU使用率过高', $metrics);
                    }
                    if ($metrics['memory_usage'] > $threshold['max_memory_usage']) {
                        $this->triggerPerformanceAlert($type, '内存使用率过高', $metrics);
                    }
                    if ($metrics['disk_usage'] > $threshold['max_disk_usage']) {
                        $this->triggerPerformanceAlert($type, '磁盘使用率过高', $metrics);
                    }
                    break;
            }
        }
    }

    /**
     * 获取性能阈值配置
     */
    private function getPerformanceThresholds() {
        $sql = "SELECT message FROM cluster_alarm_message_config WHERE type = 'performance'";
        $result = $this->CI->Cluster_model->getList($sql);
        
        if (!empty($result)) {
            $thresholds = json_decode($result[0]['message'], true);
        } else {
            // 默认阈值
            $thresholds = [
                'alarm_processing' => [
                    'max_duration' => 300 // 5分钟
                ],
                'push_performance' => [
                    'min_success_rate' => 80 // 80%成功率
                ],
                'system_resources' => [
                    'max_cpu_usage' => 90, // 90% CPU使用率
                    'max_memory_usage' => 90, // 90% 内存使用率
                    'max_disk_usage' => 90 // 90% 磁盘使用率
                ]
            ];
        }
        
        return $thresholds;
    }

    /**
     * 触发性能告警
     */
    private function triggerPerformanceAlert($type, $message, $metrics) {
        $sql = "INSERT INTO cluster_alarm_info (alarm_type, job_info, occur_timestamp, alarm_level) 
                VALUES (?, ?, NOW(), ?)";
        $this->CI->Cluster_model->updateList($sql, [
            'performance_' . $type,
            json_encode([
                'message' => $message,
                'metrics' => $metrics
            ]),
            'WARNING'
        ]);
    }
} 