# MongoDB Change Stream 调试指南

## 问题描述
测试人员发现 MongoDB changeStream 的 update 监听流没有生效。

## 🐛 发现的关键 Bug

### 1. **数据类型映射错误** (已修复)
**位置**: `sendEditResult` 和 `sendDeleteResult` 函数
**问题**:
- 第1198行：update 操作使用了错误的数据类型 `int8(13)`，应该是 `int8(14)`
- 第1340行：delete 操作使用了错误的数据类型 `int8(13)`，应该是 `int8(15)`

**影响**: 这可能导致接收端无法正确识别操作类型，从而忽略 update 和 delete 事件。

**修复**:
```go
// 修复前
dataType := int8(13) // update - 错误！
dataType := int8(13) // delete - 错误！

// 修复后
dataType := int8(14) // update - 正确
dataType := int8(15) // delete - 正确
```

### 2. **未处理的操作类型** (已修复)
**问题**: 代码中没有处理未知的 operationType，可能导致程序静默跳过某些事件。

**修复**: 添加了未知操作类型的警告日志。

## 常见原因分析

### 1. MongoDB 配置问题
- **副本集要求**: Change Stream 只能在副本集模式下工作，单机模式不支持
- **版本要求**: MongoDB 3.6+ 才支持 Change Stream
- **Oplog 要求**: 需要有可用的 oplog（副本集的操作日志）

### 2. 权限问题
- 用户需要有读取 oplog 的权限
- 需要对目标数据库和集合有读取权限

### 3. 网络连接问题
- Change Stream 是长连接，网络中断会导致监听失效
- 防火墙可能阻止连接

## 调试工具

### 1. MongoDB 诊断工具
```bash
cd KunlunMonitor/go/consistent_sql
go run diagnose_mongodb.go
```

这个工具会检查：
- MongoDB 版本
- 副本集状态
- Oplog 可用性
- 用户权限
- Change Stream 创建能力

### 2. Change Stream 测试工具
```bash
cd KunlunMonitor/go/consistent_sql
go run test_changestream.go
```

这个工具会：
- 创建测试数据
- 启动 Change Stream 监听
- 模拟 update/delete 操作
- 显示接收到的事件

## 日志增强

已在代码中添加了详细的 Change Stream 调试日志：

### 启动阶段日志
- Change Stream 配置信息
- 重试机制日志
- 连接成功/失败日志

### 事件处理日志
- 原始事件内容
- 操作类型识别
- 文档ID提取
- 数据处理过程

### 错误恢复日志
- 错误类型和原因
- Resume Token 保存
- 重试过程
- 恢复成功/失败

## 解决方案

### 1. 检查 MongoDB 配置

#### 检查是否为副本集
```javascript
// 在 mongo shell 中执行
db.isMaster()
```

如果没有 `setName` 字段，说明不是副本集模式。

#### 初始化副本集
```bash
# 1. 停止 MongoDB
sudo systemctl stop mongod

# 2. 修改配置文件 /etc/mongod.conf
replication:
  replSetName: "rs0"

# 3. 启动 MongoDB
sudo systemctl start mongod

# 4. 初始化副本集
mongo --eval "rs.initiate()"

# 5. 检查状态
mongo --eval "rs.status()"
```

### 2. 检查权限

确保用户有以下权限：
```javascript
// 创建具有必要权限的用户
db.createUser({
  user: "changestream_user",
  pwd: "password",
  roles: [
    { role: "read", db: "your_database" },
    { role: "read", db: "local" }  // 读取 oplog 需要
  ]
})
```

### 3. 代码优化建议

#### 添加连接池配置
```go
clientOptions := options.Client().ApplyURI(mongoURL)
clientOptions.SetMaxPoolSize(10)
clientOptions.SetMinPoolSize(2)
clientOptions.SetMaxConnIdleTime(30 * time.Second)
```

#### 添加心跳检测
```go
clientOptions.SetHeartbeatInterval(10 * time.Second)
clientOptions.SetServerSelectionTimeout(5 * time.Second)
```

#### 优化 Change Stream 选项
```go
changeStreamOptions := options.ChangeStream()
changeStreamOptions.SetFullDocument(options.UpdateLookup)
changeStreamOptions.SetBatchSize(100)  // 调整批次大小
changeStreamOptions.SetMaxAwaitTime(1 * time.Second)  // 设置等待时间
```

## 监控和日志

### 查看详细日志
设置日志级别为 Debug 来查看更多信息：
```go
logger.SetLevel(logrus.DebugLevel)
```

### 关键日志字段
- `job_id`: 任务ID
- `database`: 数据库名
- `collection`: 集合名
- `operation`: 操作类型 (update/delete)
- `document_id`: 文档ID
- `resume_token`: 恢复令牌
- `error`: 错误信息

### 日志示例
```json
{
  "level": "info",
  "time": "2024-01-01T10:00:00Z",
  "msg": "检测到MongoDB变更事件",
  "job_id": "job123",
  "operation": "update",
  "database": "test_db",
  "collection": "test_collection",
  "document_id": "507f1f77bcf86cd799439011",
  "resume_token": "..."
}
```

## 故障排除步骤

1. **运行诊断工具**
   ```bash
   go run diagnose_mongodb.go
   ```

2. **检查 MongoDB 日志**
   ```bash
   tail -f /var/log/mongodb/mongod.log
   ```

3. **测试 Change Stream**
   ```bash
   go run test_changestream.go
   ```

4. **查看应用日志**
   ```bash
   tail -f ./consistent_sql.log | grep -i changestream
   ```

5. **手动测试操作**
   ```javascript
   // 在 mongo shell 中手动执行更新
   db.test_collection.updateOne(
     {_id: ObjectId("...")}, 
     {$set: {test_field: "new_value"}}
   )
   ```

## 性能优化

### 1. 减少监听范围
```go
// 只监听特定操作
pipeline := mongo.Pipeline{
  {{"$match", bson.D{
    {"operationType", bson.D{{"$in", bson.A{"update", "delete"}}}},
    {"ns.db", dbName},
    {"ns.coll", collectionName},
  }}},
}
```

### 2. 使用 Resume Token
```go
// 保存和恢复 Resume Token
if resumeToken := changeStream.ResumeToken(); resumeToken != nil {
  // 保存到持久化存储
  saveResumeToken(jobId, resumeToken)
}
```

### 3. 批量处理
```go
changeStreamOptions.SetBatchSize(100)  // 增加批次大小
```

## 联系支持

如果问题仍然存在，请提供：
1. MongoDB 版本信息
2. 副本集配置
3. 完整的错误日志
4. 诊断工具的输出结果
