package task

import (
	"net"
	"sync"
	"time"
)

// MockUnixConnection 模拟Unix socket连接
type MockUnixConnection struct {
	receivedData []byte
	mutex        sync.Mutex
	socketPath   string
}

// NewMockUnixConnection 创建一个新的Unix socket连接模拟器
func NewMockUnixConnection(socketPath string) *MockUnixConnection {
	return &MockUnixConnection{
		receivedData: make([]byte, 0),
		socketPath:   socketPath,
	}
}

// Write 实现io.Writer接口，记录写入的数据
func (m *MockUnixConnection) Write(data []byte) (n int, err error) {
	m.mutex.Lock()
	defer m.mutex.Unlock()
	m.receivedData = append(m.receivedData, data...)
	return len(data), nil
}

// Read 实现io.Reader接口，始终返回EOF
func (m *MockUnixConnection) Read(p []byte) (n int, err error) {
	return 0, nil
}

// Close 不执行实际操作
func (m *MockUnixConnection) Close() error {
	return nil
}

// LocalAddr 返回本地Unix socket地址
func (m *MockUnixConnection) LocalAddr() net.Addr {
	return &net.UnixAddr{Name: m.socketPath, Net: "unix"}
}

// RemoteAddr 返回远程Unix socket地址
func (m *MockUnixConnection) RemoteAddr() net.Addr {
	return &net.UnixAddr{Name: m.socketPath + "-client", Net: "unix"}
}

// SetDeadline 设置超时（未实际使用）
func (m *MockUnixConnection) SetDeadline(t time.Time) error {
	return nil
}

// SetReadDeadline 设置读取超时（未实际使用）
func (m *MockUnixConnection) SetReadDeadline(t time.Time) error {
	return nil
}

// SetWriteDeadline 设置写入超时（未实际使用）
func (m *MockUnixConnection) SetWriteDeadline(t time.Time) error {
	return nil
}

// GetReceivedData 获取通过此连接发送的所有数据
func (m *MockUnixConnection) GetReceivedData() []byte {
	m.mutex.Lock()
	defer m.mutex.Unlock()
	return m.receivedData
}
