package task

import (
	"context"
	"encoding/json"
	"fmt"
	"math"
	"net"
	"strings"
	"sync"
	"time"

	"github.com/sirupsen/logrus"
	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/bson/primitive"
	"go.mongodb.org/mongo-driver/mongo"
	"go.mongodb.org/mongo-driver/mongo/options"

	"consistent_sql/logger"
	"consistent_sql/model"
	"consistent_sql/protocol"
)

// 添加常量定义
const (
	defaultRetryInterval = 100 * time.Millisecond
	defaultReadTimeout   = 86400 * time.Second //5 * time.Second
	maxRetryAttempts     = 3
	retryWaitInterval    = 2 * time.Second
)

// MongoTaskStatus 结构体记录任务状态
type MongoTaskStatus struct {
	IsRunning   bool
	LastID      string
	Error       error
	FirstSend   bool     // 标记是否是第一次发送
	WaitingAck  bool     // 是否正在等待确认
	ResumeToken bson.Raw // 用于变更流断点续传
}

// MongoTaskManager 结构体管理多个MongoDB任务
type MongoTaskManager struct {
	Tasks    map[string]context.CancelFunc
	Statuses map[string]*MongoTaskStatus
	Mu       sync.Mutex

	// 连接池
	clients   map[string]*mongo.Client
	clientsMu sync.Mutex

	// 添加新的字段：消息处理回调
	ackHandlers   map[string]func(byte, []byte)
	ackHandlersMu sync.RWMutex
}

// NewMongoTaskManager 创建新的MongoDB任务管理器
func NewMongoTaskManager() *MongoTaskManager {
	return &MongoTaskManager{
		Tasks:       make(map[string]context.CancelFunc),
		Statuses:    make(map[string]*MongoTaskStatus),
		clients:     make(map[string]*mongo.Client),
		ackHandlers: make(map[string]func(byte, []byte)),
	}
}

// createMongoClientOptions 创建MongoDB客户端选项，支持认证
func (tm *MongoTaskManager) createMongoClientOptions(mongos model.RecvData) *options.ClientOptions {
	// 处理URL格式，确保包含mongodb://前缀
	mongoURL := mongos.URL
	if !strings.HasPrefix(mongoURL, "mongodb://") && !strings.HasPrefix(mongoURL, "mongodb+srv://") {
		mongoURL = "mongodb://" + mongoURL
		logger.Info("任务ID=%s：自动添加mongodb://前缀，完整URL=%s", mongos.JobId, mongoURL)
	}

	clientOptions := options.Client().ApplyURI(mongoURL)

	// 如果提供了用户名和密码，设置认证信息
	if mongos.DbUser != "" && mongos.DbPasswd != "" {
		credential := options.Credential{
			Username: mongos.DbUser,
			Password: mongos.DbPasswd,
		}
		clientOptions.SetAuth(credential)
		logger.Info("任务ID=%s：配置MongoDB认证，用户名=%s", mongos.JobId, mongos.DbUser)
	} else {
		logger.Info("任务ID=%s：配置MongoDB无认证连接", mongos.JobId)
	}

	return clientOptions
}

// StartTaskMongo 启动一个新MongoDB任务，如果任务已存在，则不会重复创建
func (tm *MongoTaskManager) StartTaskMongo(mongos model.RecvData, conn net.Conn) {
	tm.Mu.Lock()
	if _, exists := tm.Tasks[mongos.JobId]; exists {
		logger.Info("任务已存在: %s", mongos.JobId)
		tm.Mu.Unlock()
		protocol.SendError(conn, "1", "job id "+mongos.JobId+" exists", mongos.JobId)
		return
	}

	logger.Info("创建MongoDB任务，ID=%s", mongos.JobId)
	tm.Statuses[mongos.JobId] = &MongoTaskStatus{
		IsRunning:  true,
		LastID:     mongos.Gtid,
		Error:      nil,
		FirstSend:  true,  // 初始设置为第一次发送
		WaitingAck: false, // 初始设置为未等待确认
	}
	logger.Info("任务ID=%s：初始化状态，FirstSend=true（首次发送不需等待确认），WaitingAck=false（第二次及后续发送会永久等待确认，无超时限制）", mongos.JobId)

	ctx, cancel := context.WithCancel(context.Background())
	tm.Tasks[mongos.JobId] = cancel
	tm.Mu.Unlock()

	// 启动同步任务
	go tm.syncMongoTask(ctx, mongos, conn)
}

// 添加消息处理器注册方法
// registerAckHandler 注册一个确认消息处理函数
func (tm *MongoTaskManager) registerAckHandler(jobId string, handler func(byte, []byte)) {
	tm.ackHandlersMu.Lock()
	defer tm.ackHandlersMu.Unlock()
	tm.ackHandlers[jobId] = handler
}

// unregisterAckHandler 注销一个确认消息处理函数
func (tm *MongoTaskManager) unregisterAckHandler(jobId string) {
	tm.ackHandlersMu.Lock()
	defer tm.ackHandlersMu.Unlock()
	delete(tm.ackHandlers, jobId)
}

// HandleAckMessage 处理来自main.go分发的确认消息
func (tm *MongoTaskManager) HandleAckMessage(jobId string, msgType byte, msgData []byte) {
	tm.ackHandlersMu.RLock()
	handler, exists := tm.ackHandlers[jobId]
	tm.ackHandlersMu.RUnlock()

	if exists && handler != nil {
		handler(msgType, msgData)
	}
}

// syncMongoTask 处理MongoDB同步任务
func (tm *MongoTaskManager) syncMongoTask(ctx context.Context, mongos model.RecvData, conn net.Conn) {
	logger.Info("开始MongoDB同步任务，任务ID=%s", mongos.JobId)

	// 创建上下文 (取消使用syncCtx变量)
	cancelSync := func() {}
	defer cancelSync()

	// 连接MongoDB
	clientOptions := tm.createMongoClientOptions(mongos)
	client, err := mongo.Connect(ctx, clientOptions)
	if err != nil {
		logger.Error("任务ID=%s：MongoDB客户端连接失败: %v", mongos.JobId, err)
		tm.updateTaskStatus(mongos.JobId, "", err, false, false)
		protocol.SendError(conn, "1", "客户端链接失败", mongos.JobId)
		return
	}
	defer client.Disconnect(ctx)

	// 获取集合
	collection := client.Database(mongos.DbDatabase).Collection(mongos.DbTable)

	// 添加 MongoDB 配置检测
	tm.checkMongoDBConfiguration(ctx, client, mongos.JobId)

	// 设置 Change Stream 监听集合变更
	// 创建增量同步的上下文，当主同步取消时可以单独取消增量同步
	changeStreamCtx, cancelChangeStream := context.WithCancel(ctx)
	defer cancelChangeStream()

	// 尝试从状态中获取上次的resumeToken
	var resumeToken bson.Raw
	tm.Mu.Lock()
	if status, exists := tm.Statuses[mongos.JobId]; exists && status.ResumeToken != nil {
		resumeToken = status.ResumeToken
		logger.Info("任务ID=%s: 使用保存的resumeToken恢复增量监听", mongos.JobId)
	}
	tm.Mu.Unlock()

	// 配置Change Stream选项
	changeStreamOptions := options.ChangeStream()
	// 设置批大小和全文档
	changeStreamOptions.SetBatchSize(10)
	changeStreamOptions.SetFullDocument(options.UpdateLookup)
	// 如果有恢复令牌，设置从该令牌之后恢复
	if resumeToken != nil {
		changeStreamOptions.SetResumeAfter(resumeToken)
	}

	// 设置Change Stream监听update和delete操作
	pipeline := mongo.Pipeline{
		{{
			"$match", bson.D{{
				"operationType", bson.D{{
					"$in", bson.A{"update", "delete"},
				}},
			}},
		}},
	}

	// 启动Change Stream
	// 添加重试机制
	var changeStream *mongo.ChangeStream
	maxRetries := 5

	// 添加 changeStream 调试日志
	logger.WithFields(logrus.Fields{
		"job_id":       mongos.JobId,
		"database":     mongos.DbDatabase,
		"collection":   mongos.DbTable,
		"pipeline":     pipeline,
		"options":      changeStreamOptions,
		"resume_token": resumeToken,
	}).Info("开始启动 MongoDB Change Stream")

	for i := 0; i < maxRetries; i++ {
		logger.WithFields(logrus.Fields{
			"job_id":    mongos.JobId,
			"attempt":   i + 1,
			"max_retry": maxRetries,
		}).Info("尝试启动 MongoDB Change Stream")

		changeStream, err = collection.Watch(changeStreamCtx, pipeline, changeStreamOptions)
		if err == nil {
			logger.WithFields(logrus.Fields{
				"job_id":     mongos.JobId,
				"attempt":    i + 1,
				"database":   mongos.DbDatabase,
				"collection": mongos.DbTable,
			}).Info("MongoDB Change Stream 启动成功")
			break
		}

		logger.WithFields(logrus.Fields{
			"job_id":     mongos.JobId,
			"attempt":    i + 1,
			"max_retry":  maxRetries,
			"error":      err.Error(),
			"database":   mongos.DbDatabase,
			"collection": mongos.DbTable,
		}).Error("启动MongoDB Change Stream失败")

		// 指数退避策略
		backoff := time.Duration(math.Pow(2, float64(i))) * time.Second
		if backoff > 30*time.Second {
			backoff = 30 * time.Second
		}

		logger.WithFields(logrus.Fields{
			"job_id":       mongos.JobId,
			"backoff_time": backoff.String(),
		}).Info("等待重试 MongoDB Change Stream")

		time.Sleep(backoff)
	}

	if err != nil {
		logger.WithFields(logrus.Fields{
			"job_id":     mongos.JobId,
			"database":   mongos.DbDatabase,
			"collection": mongos.DbTable,
			"error":      err.Error(),
		}).Error("启动MongoDB Change Stream失败: 所有尝试均失败")

		tm.updateTaskStatus(mongos.JobId, "", err, false, false)
		protocol.SendError(conn, "1", "启动MongoDB Change Stream失败: "+err.Error(), mongos.JobId)
		return
	}
	defer changeStream.Close(changeStreamCtx)

	// 创建共享的MongoDB客户端连接池
	// 这个客户端将由监听器使用
	sharedClientOptions := tm.createMongoClientOptions(mongos)
	// 设置连接池选项
	sharedClientOptions.SetMaxPoolSize(5)
	sharedClientOptions.SetMinPoolSize(1)
	sharedClient, err := mongo.Connect(changeStreamCtx, sharedClientOptions)
	if err != nil {
		logger.Error("任务ID=%s：创建MongoDB共享客户端失败: %v", mongos.JobId, err)
		tm.updateTaskStatus(mongos.JobId, "", err, false, false)
		protocol.SendError(conn, "1", "客户端链接失败", mongos.JobId)
		return
	}
	defer sharedClient.Disconnect(changeStreamCtx)

	// 启动单独的goroutine监听Change Stream
	go tm.watchMongoChanges(changeStreamCtx, changeStream, conn, mongos.JobId, mongos.DbDatabase, mongos.DbTable, mongos.URL, sharedClient)

	// 获取起始ID
	lastID := mongos.Gtid
	if lastID != "" {
		logger.Info("任务ID=%s：使用指定的起始ID=%s开始同步", mongos.JobId, lastID)
	} else {
		logger.Info("任务ID=%s：未指定起始ID，将从头开始同步", mongos.JobId)
	}

	// 发送开始消息
	protocol.SendError(conn, "0", "job "+mongos.JobId+" start", mongos.JobId)

	// 创建接收确认消息的通道
	ackChan := make(chan bool, 1)

	// 设置TCP保活选项
	if tcpConn, ok := conn.(*net.TCPConn); ok {
		tcpConn.SetKeepAlive(true)
		tcpConn.SetKeepAlivePeriod(30 * time.Second)
	}

	// 重要修改：不再直接从conn读取，而是通过共享的消息通道接收消息
	// 在source.go的handleSourceConnection中应当解析消息后分发到相应的通道
	// 这里我们仅处理从main.go中分发到该任务的消息
	//
	// 为了保持代码结构不变，改为通过MongoDB任务管理器添加消息处理回调函数
	tm.registerAckHandler(mongos.JobId, func(msgType byte, msgData []byte) {
		if msgType == byte(protocol.MRsprowmsg) {
			logger.Info("任务ID=%s：收到 MRsprowmsg 确认消息", mongos.JobId)
			// 更新任务状态
			tm.Mu.Lock()
			if status, exists := tm.Statuses[mongos.JobId]; exists {
				status.WaitingAck = false
			}
			tm.Mu.Unlock()

			// 通知主循环
			select {
			case ackChan <- true:
			default:
				// 如果通道已满，先清空，再发送
				select {
				case <-ackChan:
				default:
				}
				ackChan <- true
			}
		} else if len(msgData) > 0 {
			// 兼容旧消息格式（文本格式）
			rawMsg := string(msgData)
			if strings.Contains(rawMsg, "\"type\":4") || strings.Contains(rawMsg, "M_RspRowMsg") {
				logger.Info("任务ID=%s：收到旧格式 MRsprowmsg 确认消息", mongos.JobId)
				// 更新任务状态
				tm.Mu.Lock()
				if status, exists := tm.Statuses[mongos.JobId]; exists {
					status.WaitingAck = false
				}
				tm.Mu.Unlock()

				// 通知主循环
				select {
				case ackChan <- true:
				default:
					// 如果通道已满，先清空，再发送
					select {
					case <-ackChan:
					default:
					}
					ackChan <- true
				}
			}
		}
	})

	// 清理函数：在任务结束时取消注册
	defer tm.unregisterAckHandler(mongos.JobId)

	// 设置增量同步的时间间隔，当没有新数据时使用
	incremental_sync_interval := 5 * time.Second
	var lastSyncTime time.Time = time.Now()
	var noDataCount int = 0

	for {
		select {
		case <-ctx.Done():
			logger.Info("任务已取消: %s", mongos.JobId)
			tm.updateTaskStatus(mongos.JobId, lastID, nil, false, false)
			return
		default:
			// 检查是否在等待确认
			tm.Mu.Lock()
			status, exists := tm.Statuses[mongos.JobId]
			isWaiting := exists && status.WaitingAck
			isFirstSend := exists && status.FirstSend
			tm.Mu.Unlock()

			if isWaiting {
				// 等待确认，但同时也要监听取消信号
				select {
				case <-ackChan:
					// 收到确认，继续处理
					logger.Info("任务ID=%s：收到MRsprowmsg确认，继续处理下一批数据", mongos.JobId)
				case <-ctx.Done():
					// 收到取消信号
					logger.Info("任务已取消: %s", mongos.JobId)
					tm.updateTaskStatus(mongos.JobId, lastID, nil, false, false)
					return
				}
				// 移除超时处理，如果没收到确认就等待，直到收到确认或取消信号
			}

			// 查询并同步数据
			result, err := tm.SyncMongoTask(ctx, collection, lastID)
			if err != nil {
				logger.Error("同步失败: %s, 错误: %v", mongos.JobId, err)
				time.Sleep(defaultRetryInterval)
				continue
			}

			if len(result.Data) > 0 {
				// 检查是否是首次发送
				tm.Mu.Lock()
				status, exists := tm.Statuses[mongos.JobId]
				isFirstSendFlag := exists && status.FirstSend
				tm.Mu.Unlock()

				if isFirstSendFlag {
					logger.Info("任务ID=%s：处理首包数据，首包数据不进行去重处理，当前LastID=%s", mongos.JobId, lastID)
				} else {
					logger.Info("任务ID=%s：处理后续包数据，将过滤掉与LastID=%s相同的记录", mongos.JobId, lastID)
				}

				// 过滤掉与上次同步ID相同的记录，避免重复处理
				filteredData := result.Data
				if lastID != "" {
					// 检查是否是第一包数据
					tm.Mu.Lock()
					status, exists := tm.Statuses[mongos.JobId]
					isFirstSend := exists && status.FirstSend
					tm.Mu.Unlock()

					// 如果不是第一包数据，才进行过滤去重
					if !isFirstSend {
						filteredData = make([]model.MongoSourceDBData, 0, len(result.Data))
						for _, data := range result.Data {
							// 数据ID与上次ID不同，则保留
							if data.Id != lastID {
								filteredData = append(filteredData, data)
							} else {
								logger.Info("任务ID=%s：过滤掉重复记录，ID=%s", mongos.JobId, data.Id)
							}
						}
						logger.Info("任务ID=%s：过滤结果 - 过滤前%d条记录，过滤后%d条记录，过滤掉%d条重复记录",
							mongos.JobId, len(result.Data), len(filteredData), len(result.Data)-len(filteredData))
					} else {
						logger.Info("任务ID=%s：首包数据不进行去重，保留所有%d条记录", mongos.JobId, len(result.Data))
					}
				}

				// 如果过滤后没有数据，则继续下一轮查询
				if len(filteredData) == 0 {
					logger.Info("任务ID=%s：过滤重复记录后无新数据，继续下一轮查询", mongos.JobId)
					time.Sleep(1 * time.Second)
					continue
				}

				// 更新处理后的数据
				result.Data = filteredData

				// 有新数据，重置计数器
				noDataCount = 0
				lastSyncTime = time.Now()

				// 发送数据
				if err := tm.sendSyncResult(conn, mongos.JobId, &result, mongos.DbDatabase, mongos.DbTable); err != nil {
					logger.Error("发送数据失败: %v", err)
					time.Sleep(defaultRetryInterval)
					continue
				}

				// 更新等待状态
				if !isFirstSend {
					tm.Mu.Lock()
					if status, exists := tm.Statuses[mongos.JobId]; exists {
						status.WaitingAck = true
					}
					tm.Mu.Unlock()
				} else {
					tm.Mu.Lock()
					if status, exists := tm.Statuses[mongos.JobId]; exists {
						status.FirstSend = false
					}
					tm.Mu.Unlock()
				}

				// 更新最后ID
				if result.LastID != primitive.NilObjectID {
					lastID = result.LastID.Hex()
					var waitingFlag bool = false
					if !isFirstSend {
						waitingFlag = true
					}
					tm.updateTaskStatus(mongos.JobId, lastID, nil, false, waitingFlag)
				}
			} else {
				// 没有新数据，增加计数器
				noDataCount++

				// 如果没有搜索到新数据，并且距离上次同步已经超过间隔时间
				if time.Since(lastSyncTime) > incremental_sync_interval {
					logger.Info("任务ID=%s：没有新数据，等待增量同步间隔(%v)后继续，上次ID=%s",
						mongos.JobId, incremental_sync_interval, lastID)
					time.Sleep(incremental_sync_interval)
					lastSyncTime = time.Now()
				} else {
					// 短暂等待后继续
					time.Sleep(1 * time.Second)
				}
			}
		}
	}
}

// updateTaskStatus 更新任务状态
func (tm *MongoTaskManager) updateTaskStatus(jobId, lastID string, err error, isFirstSend bool, waitingAck bool) {
	tm.Mu.Lock()
	defer tm.Mu.Unlock()

	if status, exists := tm.Statuses[jobId]; exists {
		if lastID != "" {
			status.LastID = lastID
		}
		status.Error = err
		if err != nil {
			status.IsRunning = false
		}
		status.FirstSend = isFirstSend
		status.WaitingAck = waitingAck
	}
}

// sendSyncResult 发送同步结果
func (tm *MongoTaskManager) sendSyncResult(conn net.Conn, jobId string, result *model.SyncResult, dbName, tableName string) error {
	if len(result.Data) == 0 {
		return nil
	}

	// 准备行数据
	rowCount := uint32(len(result.Data))

	// 获取当前任务状态
	tm.Mu.Lock()
	isFirstSend := false
	if status, exists := tm.Statuses[jobId]; exists {
		isFirstSend = status.FirstSend
	}
	tm.Mu.Unlock()

	if isFirstSend {
		logger.Info("发送数据包(jobId=%s)：首次发送批量数据，不需要等待确认，本批次包含 %d 条记录", jobId, rowCount)
	} else {
		logger.Info("发送数据包(jobId=%s)：非首次发送批量数据，发送后需等待确认，本批次包含 %d 条记录", jobId, rowCount)
	}

	// 初始化消息buffer
	var message []byte

	// 1. 类型 (MReqrowmsg)
	messageType := byte(protocol.MReqrowmsg)
	message = append(message, messageType)

	// 2. 多少行 (uint32)
	message = append(message, byte(rowCount), byte(rowCount>>8), byte(rowCount>>16), byte(rowCount>>24))

	// 使用第一条数据的ID作为gtid
	firstData := result.Data[0]
	gtidBytes := []byte(fmt.Sprintf("'%s'", firstData.Id))
	gtidLen := uint32(len(gtidBytes))

	// 3. gtid (len+str)
	message = append(message, byte(gtidLen), byte(gtidLen>>8), byte(gtidLen>>16), byte(gtidLen>>24))
	message = append(message, gtidBytes...)

	// 4. binlog_file (int32)，MongoDB无binlog，直接写入0
	binlogFile := int32(0)
	message = append(message, byte(binlogFile), byte(binlogFile>>8), byte(binlogFile>>16), byte(binlogFile>>24))

	// 5. 数据类型 (insert=13/delete=15/update=14) - int8类型
	// 使用第一条数据的操作类型
	var dataType int8
	if strings.Contains(firstData.Optype, "insert") {
		dataType = 13
	} else if strings.Contains(firstData.Optype, "update") {
		dataType = 14
	} else if strings.Contains(firstData.Optype, "delete") {
		dataType = 15
	} else {
		dataType = 13 // 默认为插入
	}
	message = append(message, byte(dataType))

	// 6. db_name (len+str)
	dbNameBytes := []byte(dbName)
	dbNameLen := uint32(len(dbNameBytes))
	message = append(message, byte(dbNameLen), byte(dbNameLen>>8), byte(dbNameLen>>16), byte(dbNameLen>>24))
	message = append(message, dbNameBytes...)

	// 7. table_name (len+str)
	tbNameBytes := []byte(tableName)
	tbNameLen := uint32(len(tbNameBytes))
	message = append(message, byte(tbNameLen), byte(tbNameLen>>8), byte(tbNameLen>>16), byte(tbNameLen>>24))
	message = append(message, tbNameBytes...)

	// 8. gtid 再次写入 (len+str)
	message = append(message, byte(gtidLen), byte(gtidLen>>8), byte(gtidLen>>16), byte(gtidLen>>24))
	message = append(message, gtidBytes...)

	// 9. sql (int32) - MongoDB没有SQL，写入0
	sql := int32(0)
	message = append(message, byte(sql), byte(sql>>8), byte(sql>>16), byte(sql>>24))

	// 10. 列数量 (uint32) - MongoDB使用3列: _id, data, optype
	columnNum := uint32(3)
	message = append(message, byte(columnNum), byte(columnNum>>8), byte(columnNum>>16), byte(columnNum>>24))

	// 11. 列名数组 (每个列名: len+str)
	columnNames := []string{"__sys_obj_id__", "doc", "optype"}
	for _, colName := range columnNames {
		colNameBytes := []byte(colName)
		colNameLen := uint32(len(colNameBytes))
		message = append(message, byte(colNameLen), byte(colNameLen>>8), byte(colNameLen>>16), byte(colNameLen>>24))
		message = append(message, colNameBytes...)
	}

	// 12. 列属性数组 (每个属性: uint32, 都写0)
	for i := 0; i < 3; i++ {
		attr := uint32(0)
		message = append(message, byte(attr), byte(attr>>8), byte(attr>>16), byte(attr>>24))
	}

	// 13. 行数 (uint32, 发送所有行)
	message = append(message, byte(rowCount), byte(rowCount>>8), byte(rowCount>>16), byte(rowCount>>24))

	// 14. 所有行的内容
	for _, mongoData := range result.Data {
		// _id字段
		idBytes := []byte(fmt.Sprintf("'%s'", mongoData.Id))
		idLen := uint32(len(idBytes))
		message = append(message, byte(idLen), byte(idLen>>8), byte(idLen>>16), byte(idLen>>24))
		message = append(message, idBytes...)

		// data字段
		dataBytes := []byte(fmt.Sprintf("'%s'", mongoData.Addr))
		dataLen := uint32(len(dataBytes))
		message = append(message, byte(dataLen), byte(dataLen>>8), byte(dataLen>>16), byte(dataLen>>24))
		message = append(message, dataBytes...)

		// optype字段
		optypeBytes := []byte(fmt.Sprintf("'%s'", mongoData.Optype))
		optypeLen := uint32(len(optypeBytes))
		message = append(message, byte(optypeLen), byte(optypeLen>>8), byte(optypeLen>>16), byte(optypeLen>>24))
		message = append(message, optypeBytes...)

		// 如果是更新操作，还需要添加旧值信息
		if dataType == 14 { // update
			// 旧的_id值
			oldIdBytes := []byte(mongoData.Id)
			oldIdLen := uint32(len(oldIdBytes))
			message = append(message, byte(oldIdLen), byte(oldIdLen>>8), byte(oldIdLen>>16), byte(oldIdLen>>24))
			message = append(message, oldIdBytes...)

			// 旧的data值 (简化处理，使用空对象)
			oldDataBytes := []byte("{}")
			oldDataLen := uint32(len(oldDataBytes))
			message = append(message, byte(oldDataLen), byte(oldDataLen>>8), byte(oldDataLen>>16), byte(oldDataLen>>24))
			message = append(message, oldDataBytes...)

			// 旧的optype值
			oldOptypeBytes := []byte("'old'")
			oldOptypeLen := uint32(len(oldOptypeBytes))
			message = append(message, byte(oldOptypeLen), byte(oldOptypeLen>>8), byte(oldOptypeLen>>16), byte(oldOptypeLen>>24))
			message = append(message, oldOptypeBytes...)
		}

		// logger.Info("正在打包第 %d/%d 条数据: ID=%s", i+1, rowCount, mongoData.Id)
	}

	// 发送完整的消息包
	logger.Info("发送完整的MongoDB数据包(jobId=%s): 包含 %d 条记录，数据包大小=%d字节", jobId, rowCount, len(message))

	// 引入重试机制发送数据包
	maxSendRetries := 3
	for retry := 0; retry < maxSendRetries; retry++ {
		// 在发送消息时捕获错误
		err := protocol.SendMsg(conn, message)
		if err == nil {
			logger.Info("MongoDB数据包发送完成(jobId=%s): 成功发送 %d 条记录", jobId, rowCount)
			break
		}

		// 连接错误，尝试重试
		if retry < maxSendRetries-1 {
			logger.Warn("数据包发送失败(尝试 %d/%d): %v，将在500ms后重试", retry+1, maxSendRetries, err)
			time.Sleep(500 * time.Millisecond)
			continue
		}

		// 最后一次尝试也失败
		return fmt.Errorf("发送消息失败(已尝试%d次): %v", maxSendRetries, err)
	}

	// 添加发送完成总结日志
	tm.Mu.Lock()
	var statusFirstSend bool = false
	if status, exists := tm.Statuses[jobId]; exists {
		statusFirstSend = status.FirstSend
	}
	tm.Mu.Unlock()

	if statusFirstSend {
		logger.Info("任务ID=%s：首次批量数据发送完成，本批次共发送%d条记录，将继续处理下一批数据，无需等待确认",
			jobId, rowCount)
	} else {
		logger.Info("任务ID=%s：非首次批量数据发送完成，本批次共发送%d条记录，进入永久等待确认状态，收到MRsprowmsg后才能继续，无超时限制",
			jobId, rowCount)
	}
	return nil
}

// formatBytesHex 将字节数组格式化为十六进制表示，便于日志打印
func formatBytesHex(data []byte) string {
	var hexStr strings.Builder
	for i := 0; i < len(data); i++ {
		hexStr.WriteString(fmt.Sprintf("%02X", data[i]))
		if i < len(data)-1 {
			hexStr.WriteString(" ")
		}
	}
	return hexStr.String()
}

// watchMongoChanges 监听MongoDB的变更操作
func (tm *MongoTaskManager) watchMongoChanges(ctx context.Context, changeStream *mongo.ChangeStream, conn net.Conn, jobId, dbName, tableName, mongoURL string, sharedClient *mongo.Client) {
	logger.WithFields(logrus.Fields{
		"job_id":     jobId,
		"database":   dbName,
		"collection": tableName,
		"mongo_url":  mongoURL,
	}).Info("开始监听MongoDB变更操作")

	// 创建接收确认消息的通道
	ackChan := make(chan bool, 1)

	// 为该监听任务注册确认消息处理函数
	ackHandlerId := jobId + "_watch"
	tm.registerAckHandler(ackHandlerId, func(msgType byte, msgData []byte) {
		if msgType == byte(protocol.MRsprowmsg) {
			logger.Info("监听任务ID=%s：收到 MRsprowmsg 确认消息", ackHandlerId)
			// 更新任务状态
			tm.Mu.Lock()
			if status, exists := tm.Statuses[jobId]; exists {
				status.WaitingAck = false
			}
			tm.Mu.Unlock()

			// 通知主循环
			select {
			case ackChan <- true:
			default:
				// 如果通道已满，先清空，再发送
				select {
				case <-ackChan:
				default:
				}
				ackChan <- true
			}
		}
	})

	// 清理函数：在任务结束时取消注册
	defer tm.unregisterAckHandler(ackHandlerId)

	for {
		select {
		case <-ctx.Done():
			logger.WithFields(logrus.Fields{
				"job_id": jobId,
			}).Info("监听任务已取消")
			return
		default:
			// 等待下一个变更事件
			logger.WithFields(logrus.Fields{
				"job_id":     jobId,
				"database":   dbName,
				"collection": tableName,
			}).Debug("等待 MongoDB Change Stream 事件")

			if changeStream.Next(ctx) {
				// 解析变更事件
				var changeDoc bson.M
				if err := changeStream.Decode(&changeDoc); err != nil {
					logger.WithFields(logrus.Fields{
						"job_id":     jobId,
						"database":   dbName,
						"collection": tableName,
						"error":      err.Error(),
					}).Error("解析MongoDB变更事件失败")
					time.Sleep(1 * time.Second)
					continue
				}

				// 记录原始变更事件
				logger.WithFields(logrus.Fields{
					"job_id":     jobId,
					"database":   dbName,
					"collection": tableName,
					"change_doc": changeDoc,
				}).Debug("收到 MongoDB Change Stream 原始事件")

				// 处理变更事件
				operationType, _ := changeDoc["operationType"].(string)
				ns, _ := changeDoc["ns"].(bson.M)
				documentKey, _ := changeDoc["documentKey"].(bson.M)

				// 获取文档ID
				var objectId string
				if id, ok := documentKey["_id"].(primitive.ObjectID); ok {
					objectId = id.Hex()
				} else if id, ok := documentKey["_id"].(string); ok {
					objectId = id
				} else {
					logger.WithFields(logrus.Fields{
						"job_id":       jobId,
						"database":     dbName,
						"collection":   tableName,
						"document_key": documentKey,
					}).Error("无法获取变更文档的ID")
					continue
				}

				logger.WithFields(logrus.Fields{
					"job_id":       jobId,
					"operation":    operationType,
					"database":     ns["db"],
					"collection":   ns["coll"],
					"document_id":  objectId,
					"resume_token": changeStream.ResumeToken(),
				}).Info("检测到MongoDB变更事件")

				// 根据操作类型处理
				if operationType == "update" {
					logger.WithFields(logrus.Fields{
						"job_id":      jobId,
						"operation":   operationType,
						"document_id": objectId,
					}).Info("开始处理 MongoDB update 事件")

					// 对于更新操作，获取更新后的完整文档
					var result model.SyncResult

					// 使用共享的客户端连接池
					// 获取集合
					collection := sharedClient.Database(dbName).Collection(tableName)

					// 构建ObjectID
					docObjectID, err := primitive.ObjectIDFromHex(objectId)
					if err != nil {
						logger.WithFields(logrus.Fields{
							"job_id":      jobId,
							"operation":   operationType,
							"document_id": objectId,
							"error":       err.Error(),
						}).Error("无效的ObjectID")
						continue
					}

					// 查询更新后的文档
					var updatedDoc bson.M
					err = collection.FindOne(ctx, bson.M{"_id": docObjectID}).Decode(&updatedDoc)
					if err != nil {
						logger.WithFields(logrus.Fields{
							"job_id":      jobId,
							"operation":   operationType,
							"document_id": objectId,
							"error":       err.Error(),
						}).Error("查询更新后的文档失败")
						continue
					}

					logger.WithFields(logrus.Fields{
						"job_id":      jobId,
						"operation":   operationType,
						"document_id": objectId,
						"updated_doc": updatedDoc,
					}).Debug("成功查询到更新后的文档")

					// 构建同步结果
					dataStr := tm.documentToJsonString(updatedDoc)
					data := model.MongoSourceDBData{
						Id:     objectId,
						Addr:   dataStr,
						Optype: "update", // 编辑操作使用update作为操作类型
					}

					result.Data = append(result.Data, data)
					result.LastID = docObjectID

					// 发送编辑结果
					if err := tm.sendEditResult(conn, jobId, &result, dbName, tableName); err != nil {
						logger.Error("发送编辑数据失败: %v", err)
						continue
					}

					// 等待确认
					tm.Mu.Lock()
					if status, exists := tm.Statuses[jobId]; exists {
						status.WaitingAck = true
					}
					tm.Mu.Unlock()

					// 等待确认消息，添加超时机制避免永久阻塞
					select {
					case <-ackChan:
						logger.Info("编辑操作确认接收完成，继续监听下一个变更")
					case <-time.After(5 * time.Second):
						logger.Warn("编辑操作确认超时，继续监听下一个变更")
					case <-ctx.Done():
						return
					}

					// 重置等待状态
					tm.Mu.Lock()
					if status, exists := tm.Statuses[jobId]; exists {
						status.WaitingAck = false
					}
					tm.Mu.Unlock()
				} else if operationType == "delete" {
					logger.WithFields(logrus.Fields{
						"job_id":      jobId,
						"operation":   operationType,
						"document_id": objectId,
					}).Info("开始处理 MongoDB delete 事件")

					// 对于删除操作，只发送文档ID
					var result model.SyncResult

					// 构建ObjectID
					docObjectID, err := primitive.ObjectIDFromHex(objectId)
					if err != nil {
						logger.WithFields(logrus.Fields{
							"job_id":      jobId,
							"operation":   operationType,
							"document_id": objectId,
							"error":       err.Error(),
						}).Error("无效的ObjectID")
						continue
					}

					// 构建同步结果，对于删除操作，Addr包含__sys_obj_id__
					deleteDataStr := fmt.Sprintf("{\"__sys_obj_id__\":\"%s\"}", objectId)
					data := model.MongoSourceDBData{
						Id:     objectId,
						Addr:   deleteDataStr, // 删除操作包含__sys_obj_id__字段
						Optype: "delete",      // 删除操作使用delete作为操作类型
					}

					logger.WithFields(logrus.Fields{
						"job_id":      jobId,
						"operation":   operationType,
						"document_id": objectId,
						"delete_data": deleteDataStr,
					}).Debug("构建删除操作数据")

					result.Data = append(result.Data, data)
					result.LastID = docObjectID

					// 发送删除结果
					if err := tm.sendDeleteResult(conn, jobId, &result, dbName, tableName); err != nil {
						logger.WithFields(logrus.Fields{
							"job_id":      jobId,
							"operation":   operationType,
							"document_id": objectId,
							"error":       err.Error(),
						}).Error("发送删除数据失败")
						continue
					}

					// 等待确认
					tm.Mu.Lock()
					if status, exists := tm.Statuses[jobId]; exists {
						status.WaitingAck = true
					}
					tm.Mu.Unlock()

					// 等待确认消息，添加超时机制避免永久阻塞
					select {
					case <-ackChan:
						logger.Info("删除操作确认接收完成，继续监听下一个变更")
					case <-time.After(5 * time.Second):
						logger.Warn("删除操作确认超时，继续监听下一个变更")
					case <-ctx.Done():
						return
					}

					// 重置等待状态
					tm.Mu.Lock()
					if status, exists := tm.Statuses[jobId]; exists {
						status.WaitingAck = false
					}
					tm.Mu.Unlock()
				} else {
					// 处理未知的操作类型
					logger.WithFields(logrus.Fields{
						"job_id":      jobId,
						"operation":   operationType,
						"document_id": objectId,
						"database":    dbName,
						"collection":  tableName,
					}).Warn("收到未处理的 MongoDB 操作类型，跳过处理")
				}
			} else if err := changeStream.Err(); err != nil {
				if err == context.Canceled || err == context.DeadlineExceeded {
					logger.WithFields(logrus.Fields{
						"job_id": jobId,
						"error":  err.Error(),
					}).Info("MongoDB变更流已关闭")
					return
				}

				logger.WithFields(logrus.Fields{
					"job_id":       jobId,
					"database":     dbName,
					"collection":   tableName,
					"error":        err.Error(),
					"resume_token": changeStream.ResumeToken(),
				}).Error("MongoDB Change Stream 发生错误，开始重试恢复")

				// 对于非致命错误，采用指数退避重试
				retryCount := 0
				maxRetries := 10
				for retryCount < maxRetries {
					retryCount++
					// 计算退避时间
					backoff := time.Duration(math.Pow(2, float64(retryCount))) * time.Second
					if backoff > 60*time.Second {
						backoff = 60 * time.Second // 最长退避1分钟
					}

					logger.WithFields(logrus.Fields{
						"job_id":       jobId,
						"database":     dbName,
						"collection":   tableName,
						"error":        err.Error(),
						"retry_count":  retryCount,
						"max_retries":  maxRetries,
						"backoff_time": backoff.String(),
					}).Warn("MongoDB Change Stream错误，准备重试")

					// 等待退避时间，同时监听取消信号
					select {
					case <-time.After(backoff):
						// 继续重试
					case <-ctx.Done():
						// 上下文取消，退出
						return
					}

					// 保存ResumeToken，如果有的话
					if resumeToken := changeStream.ResumeToken(); resumeToken != nil {
						tm.Mu.Lock()
						if status, exists := tm.Statuses[jobId]; exists {
							status.ResumeToken = resumeToken
							logger.WithFields(logrus.Fields{
								"job_id":       jobId,
								"resume_token": resumeToken,
							}).Info("保存resumeToken供将来恢复使用")
						}
						tm.Mu.Unlock()
					}

					// 尝试创建新的变更流
					// 获取当前的resumeToken
					currentResumeToken := changeStream.ResumeToken()

					logger.WithFields(logrus.Fields{
						"job_id":               jobId,
						"database":             dbName,
						"collection":           tableName,
						"current_resume_token": currentResumeToken,
						"retry_count":          retryCount,
					}).Info("尝试使用resumeToken恢复 MongoDB Change Stream")

					options := options.ChangeStream().SetResumeAfter(currentResumeToken)
					pipeline := mongo.Pipeline{
						{{
							"$match", bson.D{{
								"operationType", bson.D{{
									"$in", bson.A{"update", "delete"},
								}},
							}},
						}},
					}

					newStream, err := sharedClient.Database(dbName).Collection(tableName).Watch(ctx, pipeline, options)
					if err == nil {
						// 关闭旧流
						changeStream.Close(ctx)
						// 使用新流
						changeStream = newStream

						logger.WithFields(logrus.Fields{
							"job_id":       jobId,
							"database":     dbName,
							"collection":   tableName,
							"retry_count":  retryCount,
							"resume_token": currentResumeToken,
						}).Info("成功恢复 MongoDB Change Stream")
						break // 重试成功，退出重试循环
					} else {
						logger.WithFields(logrus.Fields{
							"job_id":       jobId,
							"database":     dbName,
							"collection":   tableName,
							"retry_count":  retryCount,
							"resume_token": currentResumeToken,
							"error":        err.Error(),
						}).Error("恢复 MongoDB Change Stream失败")
					}
				}

				// 如果超出重试次数仍然失败，记录日志并继续循环
				if retryCount >= maxRetries {
					logger.WithFields(logrus.Fields{
						"job_id":      jobId,
						"database":    dbName,
						"collection":  tableName,
						"max_retries": maxRetries,
						"final_error": err.Error(),
					}).Error("所有重试均失败，将重新开始监听")
				}
				continue
			} else {
				// 没有事件且没有错误，可能是正常的空闲状态
				logger.WithFields(logrus.Fields{
					"job_id":     jobId,
					"database":   dbName,
					"collection": tableName,
				}).Debug("MongoDB Change Stream 空闲，等待新事件")
			}
		}
	}
}

// documentToJsonString 将bson.M文档转换为JSON字符串
func (tm *MongoTaskManager) documentToJsonString(document bson.M) string {
	// 构建字段值的字符串表示
	fields := make([]string, 0, len(document))
	for key, value := range document {
		// 将 _id 字段重命名为 __sys_obj_id__
		fieldName := key
		if key == "_id" {
			fieldName = "__sys_obj_id__"
		}

		// 简单地将值转换为字符串表示
		var valueStr string
		switch v := value.(type) {
		case string:
			valueStr = "\"" + v + "\""
		case int, int32, int64, float32, float64:
			valueStr = fmt.Sprintf("%v", v)
		case bool:
			valueStr = fmt.Sprintf("%v", v)
		case primitive.ObjectID:
			valueStr = "\"" + v.Hex() + "\""
		case primitive.DateTime:
			valueStr = "\"" + time.Unix(int64(v)/1000, int64(v)%1000*1000000).Format(time.RFC3339Nano) + "\""
		case nil:
			valueStr = "null"
		case bson.M, bson.D, map[string]interface{}, []interface{}:
			bytes, _ := json.Marshal(v)
			valueStr = string(bytes)
		default:
			valueStr = fmt.Sprintf("%v", v)
		}
		fields = append(fields, "\""+fieldName+"\":"+valueStr)
	}

	return "{" + strings.Join(fields, ",") + "}"
}

// sendEditResult 发送编辑结果
func (tm *MongoTaskManager) sendEditResult(conn net.Conn, jobId string, result *model.SyncResult, dbName, tableName string) error {
	if len(result.Data) == 0 {
		return nil
	}

	// 准备行数据
	rowCount := uint32(len(result.Data))

	// 获取当前任务状态
	tm.Mu.Lock()
	if _, exists := tm.Statuses[jobId]; !exists {
		logger.Warn("任务状态不存在: %s", jobId)
	}
	tm.Mu.Unlock()

	logger.Info("发送MongoDB编辑数据包(jobId=%s)：包含 %d 条记录", jobId, rowCount)

	// 初始化消息buffer
	var message []byte

	// 1. 类型 (MReqrowmsg)
	messageType := byte(protocol.MReqrowmsg)
	message = append(message, messageType)

	// 2. 多少行 (uint32)
	message = append(message, byte(rowCount), byte(rowCount>>8), byte(rowCount>>16), byte(rowCount>>24))

	// 使用第一条数据的ID作为gtid
	firstData := result.Data[0]
	gtidBytes := []byte(fmt.Sprintf("'%s'", firstData.Id))
	gtidLen := uint32(len(gtidBytes))

	// 3. gtid (len+str)
	message = append(message, byte(gtidLen), byte(gtidLen>>8), byte(gtidLen>>16), byte(gtidLen>>24))
	message = append(message, gtidBytes...)

	// 4. binlog_file (int32)，MongoDB无binlog，直接写入0
	binlogFile := int32(0)
	message = append(message, byte(binlogFile), byte(binlogFile>>8), byte(binlogFile>>16), byte(binlogFile>>24))

	// 5. 数据类型 (insert=13/delete=15/update=14) - int8类型
	// 客户端要求默认使用13
	dataType := int8(13) // 客户端要求默认为13
	message = append(message, byte(dataType))

	// 6. db_name (len+str)
	dbNameBytes := []byte(dbName)
	dbNameLen := uint32(len(dbNameBytes))
	message = append(message, byte(dbNameLen), byte(dbNameLen>>8), byte(dbNameLen>>16), byte(dbNameLen>>24))
	message = append(message, dbNameBytes...)

	// 7. table_name (len+str)
	tbNameBytes := []byte(tableName)
	tbNameLen := uint32(len(tbNameBytes))
	message = append(message, byte(tbNameLen), byte(tbNameLen>>8), byte(tbNameLen>>16), byte(tbNameLen>>24))
	message = append(message, tbNameBytes...)

	// 8. gtid 再次写入 (len+str)
	message = append(message, byte(gtidLen), byte(gtidLen>>8), byte(gtidLen>>16), byte(gtidLen>>24))
	message = append(message, gtidBytes...)

	// 9. sql (int32) - MongoDB没有SQL，写入0
	sql := int32(0)
	message = append(message, byte(sql), byte(sql>>8), byte(sql>>16), byte(sql>>24))

	// 10. 列数量 (uint32) - MongoDB使用3列: _id, data, optype
	columnNum := uint32(3)
	message = append(message, byte(columnNum), byte(columnNum>>8), byte(columnNum>>16), byte(columnNum>>24))

	// 11. 列名数组 (每个列名: len+str)
	columnNames := []string{"__sys_obj_id__", "doc", "optype"}
	for _, colName := range columnNames {
		colNameBytes := []byte(colName)
		colNameLen := uint32(len(colNameBytes))
		message = append(message, byte(colNameLen), byte(colNameLen>>8), byte(colNameLen>>16), byte(colNameLen>>24))
		message = append(message, colNameBytes...)
	}

	// 12. 列属性数组 (每个属性: uint32, 都写0)
	for i := 0; i < 3; i++ {
		attr := uint32(0)
		message = append(message, byte(attr), byte(attr>>8), byte(attr>>16), byte(attr>>24))
	}

	// 13. 行数 (uint32, 发送所有行)
	message = append(message, byte(rowCount), byte(rowCount>>8), byte(rowCount>>16), byte(rowCount>>24))

	// 14. 所有行的内容
	for _, mongoData := range result.Data {
		// _id字段
		idBytes := []byte(fmt.Sprintf("'%s'", mongoData.Id))
		idLen := uint32(len(idBytes))
		message = append(message, byte(idLen), byte(idLen>>8), byte(idLen>>16), byte(idLen>>24))
		message = append(message, idBytes...)

		// data字段
		dataBytes := []byte(fmt.Sprintf("'%s'", mongoData.Addr))
		dataLen := uint32(len(dataBytes))
		message = append(message, byte(dataLen), byte(dataLen>>8), byte(dataLen>>16), byte(dataLen>>24))
		message = append(message, dataBytes...)

		// optype字段
		optypeBytes := []byte(fmt.Sprintf("'%s'", mongoData.Optype))
		optypeLen := uint32(len(optypeBytes))
		message = append(message, byte(optypeLen), byte(optypeLen>>8), byte(optypeLen>>16), byte(optypeLen>>24))
		message = append(message, optypeBytes...)

		// 记录文档ID和操作类型
		logger.Info("处理编辑数据：ID=%s, 操作类型=%s", mongoData.Id, mongoData.Optype)
	}

	// 发送完整的消息包
	logger.Info("发送完整的MongoDB编辑数据包(jobId=%s): 包含 %d 条记录，数据包大小=%d字节", jobId, rowCount, len(message))

	// 引入重试机制发送数据包
	maxSendRetries := 3
	for retry := 0; retry < maxSendRetries; retry++ {
		// 在发送消息时捕获错误
		err := protocol.SendMsg(conn, message)
		if err == nil {
			logger.Info("MongoDB编辑数据包发送完成(jobId=%s): 成功发送 %d 条记录", jobId, rowCount)
			break
		}

		// 连接错误，尝试重试
		if retry < maxSendRetries-1 {
			logger.Warn("编辑数据包发送失败(尝试 %d/%d): %v，将在500ms后重试", retry+1, maxSendRetries, err)
			time.Sleep(500 * time.Millisecond)
			continue
		}

		// 最后一次尝试也失败
		return fmt.Errorf("发送编辑消息失败(已尝试%d次): %v", maxSendRetries, err)
	}

	// 添加发送完成总结日志
	logger.Info("任务ID=%s：编辑数据发送完成，本批次共发送%d条记录，等待确认", jobId, rowCount)

	return nil
}

// sendDeleteResult 发送删除结果
func (tm *MongoTaskManager) sendDeleteResult(conn net.Conn, jobId string, result *model.SyncResult, dbName, tableName string) error {
	if len(result.Data) == 0 {
		return nil
	}

	// 准备行数据
	rowCount := uint32(len(result.Data))

	// 获取当前任务状态
	tm.Mu.Lock()
	if _, exists := tm.Statuses[jobId]; !exists {
		logger.Warn("任务状态不存在: %s", jobId)
	}
	tm.Mu.Unlock()

	logger.Info("发送MongoDB删除数据包(jobId=%s)：包含 %d 条记录", jobId, rowCount)

	// 初始化消息buffer
	var message []byte

	// 1. 类型 (MReqrowmsg)
	messageType := byte(protocol.MReqrowmsg)
	message = append(message, messageType)

	// 2. 多少行 (uint32)
	message = append(message, byte(rowCount), byte(rowCount>>8), byte(rowCount>>16), byte(rowCount>>24))

	// 使用第一条数据的ID作为gtid
	firstData := result.Data[0]
	gtidBytes := []byte(fmt.Sprintf("'%s'", firstData.Id))
	gtidLen := uint32(len(gtidBytes))

	// 3. gtid (len+str)
	message = append(message, byte(gtidLen), byte(gtidLen>>8), byte(gtidLen>>16), byte(gtidLen>>24))
	message = append(message, gtidBytes...)

	// 4. binlog_file (int32)，MongoDB无binlog，直接写入0
	binlogFile := int32(0)
	message = append(message, byte(binlogFile), byte(binlogFile>>8), byte(binlogFile>>16), byte(binlogFile>>24))

	// 5. 数据类型 (insert=13/delete=15/update=14) - int8类型
	// 客户端要求默认使用13
	dataType := int8(13) // 客户端要求默认为13
	message = append(message, byte(dataType))

	// 6. db_name (len+str)
	dbNameBytes := []byte(dbName)
	dbNameLen := uint32(len(dbNameBytes))
	message = append(message, byte(dbNameLen), byte(dbNameLen>>8), byte(dbNameLen>>16), byte(dbNameLen>>24))
	message = append(message, dbNameBytes...)

	// 7. table_name (len+str)
	tbNameBytes := []byte(tableName)
	tbNameLen := uint32(len(tbNameBytes))
	message = append(message, byte(tbNameLen), byte(tbNameLen>>8), byte(tbNameLen>>16), byte(tbNameLen>>24))
	message = append(message, tbNameBytes...)

	// 8. gtid 再次写入 (len+str)
	message = append(message, byte(gtidLen), byte(gtidLen>>8), byte(gtidLen>>16), byte(gtidLen>>24))
	message = append(message, gtidBytes...)

	// 9. sql (int32) - MongoDB没有SQL，写入0
	sql := int32(0)
	message = append(message, byte(sql), byte(sql>>8), byte(sql>>16), byte(sql>>24))

	// 10. 列数量 (uint32) - MongoDB使用3列: _id, data, optype
	columnNum := uint32(3)
	message = append(message, byte(columnNum), byte(columnNum>>8), byte(columnNum>>16), byte(columnNum>>24))

	// 11. 列名数组 (每个列名: len+str)
	columnNames := []string{"__sys_obj_id__", "doc", "optype"}
	for _, colName := range columnNames {
		colNameBytes := []byte(colName)
		colNameLen := uint32(len(colNameBytes))
		message = append(message, byte(colNameLen), byte(colNameLen>>8), byte(colNameLen>>16), byte(colNameLen>>24))
		message = append(message, colNameBytes...)
	}

	// 12. 列属性数组 (每个属性: uint32, 都写0)
	for i := 0; i < 3; i++ {
		attr := uint32(0)
		message = append(message, byte(attr), byte(attr>>8), byte(attr>>16), byte(attr>>24))
	}

	// 13. 行数 (uint32, 发送所有行)
	message = append(message, byte(rowCount), byte(rowCount>>8), byte(rowCount>>16), byte(rowCount>>24))

	// 14. 所有行的内容
	for _, mongoData := range result.Data {
		// _id字段
		idBytes := []byte(fmt.Sprintf("'%s'", mongoData.Id))
		idLen := uint32(len(idBytes))
		message = append(message, byte(idLen), byte(idLen>>8), byte(idLen>>16), byte(idLen>>24))
		message = append(message, idBytes...)

		// data字段 - 删除操作包含__sys_obj_id__
		dataBytes := []byte(fmt.Sprintf("'%s'", mongoData.Addr))
		dataLen := uint32(len(dataBytes))
		message = append(message, byte(dataLen), byte(dataLen>>8), byte(dataLen>>16), byte(dataLen>>24))
		message = append(message, dataBytes...)

		// optype字段
		optypeBytes := []byte(fmt.Sprintf("'%s'", mongoData.Optype))
		optypeLen := uint32(len(optypeBytes))
		message = append(message, byte(optypeLen), byte(optypeLen>>8), byte(optypeLen>>16), byte(optypeLen>>24))
		message = append(message, optypeBytes...)

		// 记录文档ID和操作类型
		logger.Info("处理删除数据：ID=%s, 操作类型=%s", mongoData.Id, mongoData.Optype)
	}

	// 发送完整的消息包
	logger.Info("发送完整的MongoDB删除数据包(jobId=%s): 包含 %d 条记录，数据包大小=%d字节", jobId, rowCount, len(message))

	// 引入重试机制发送数据包
	maxSendRetries := 3
	for retry := 0; retry < maxSendRetries; retry++ {
		// 在发送消息时捕获错误
		err := protocol.SendMsg(conn, message)
		if err == nil {
			logger.Info("MongoDB删除数据包发送完成(jobId=%s): 成功发送 %d 条记录", jobId, rowCount)
			break
		}

		// 连接错误，尝试重试
		if retry < maxSendRetries-1 {
			logger.Warn("删除数据包发送失败(尝试 %d/%d): %v，将在500ms后重试", retry+1, maxSendRetries, err)
			time.Sleep(500 * time.Millisecond)
			continue
		}

		// 最后一次尝试也失败
		return fmt.Errorf("发送删除消息失败(已尝试%d次): %v", maxSendRetries, err)
	}

	// 添加发送完成总结日志
	logger.Info("任务ID=%s：删除数据发送完成，本批次共发送%d条记录，等待确认", jobId, rowCount)

	return nil
}

// SyncMongoTask 查询并同步MongoDB数据
func (tm *MongoTaskManager) SyncMongoTask(ctx context.Context, collection *mongo.Collection, lastID string) (model.SyncResult, error) {
	queryStartTime := time.Now()
	logger.Info("开始MongoDB查询，时间：%v", queryStartTime.Format("2006-01-02 15:04:05.000"))

	// 构建查询条件
	var lastObjectID primitive.ObjectID
	var filter bson.M

	if lastID != "" {
		// 过滤掉可能存在的单引号
		lastID = strings.Trim(lastID, "'")

		var err error
		lastObjectID, err = primitive.ObjectIDFromHex(lastID)
		if err != nil {
			logger.Error("无效的 ObjectID: %s", err.Error())
			return model.SyncResult{}, err
		}
		// 注意: 使用 $gt 而不是 $gte，确保边界记录不会被重复处理
		filter = bson.M{"_id": bson.M{"$gt": lastObjectID}}
		logger.Info("MongoDB查询使用 $gt 条件，将排除ID=%s的边界记录", lastID)
	} else {
		filter = bson.M{}
		logger.Info("MongoDB查询没有设置ID过滤条件，将查询所有记录")
	}

	// 设置查询选项
	findOptions := options.Find().
		SetBatchSize(5).
		SetSort(bson.D{{Key: "_id", Value: 1}}).
		SetLimit(5) // 每次获取5条记录

	// 日志记录查询条件
	logger.Info("MongoDB查询条件: %v, 上次ID: %s, 批次大小: 5", filter, lastID)

	// 执行查询
	cursor, err := collection.Find(ctx, filter, findOptions)
	if err != nil {
		logger.Error("MongoDB查询失败: %s, 耗时: %v", err.Error(), time.Since(queryStartTime))
		return model.SyncResult{}, err
	}
	defer cursor.Close(ctx)

	// 准备结果
	var result model.SyncResult
	result.Data = make([]model.MongoSourceDBData, 0, 5) // 预分配容量
	var lastSyncedID primitive.ObjectID
	documentCount := 0

	// 遍历结果
	for cursor.Next(ctx) {
		documentCount++

		var document bson.M
		if err := cursor.Decode(&document); err != nil {
			logger.Error("解码文档失败: %s", err.Error())
			continue
		}

		// 处理文档
		lastSyncedID = document["_id"].(primitive.ObjectID)

		// 构建字段值的字符串表示
		var dataStr string
		fields := make([]string, 0, len(document))
		for key, value := range document {
			// 将 _id 字段重命名为 __sys_obj_id__
			fieldName := key
			if key == "_id" {
				fieldName = "__sys_obj_id__"
			}

			// 简单地将值转换为字符串表示
			var valueStr string
			switch v := value.(type) {
			case string:
				valueStr = "\"" + v + "\""
			case int, int32, int64, float32, float64:
				valueStr = fmt.Sprintf("%v", v)
			case bool:
				valueStr = fmt.Sprintf("%v", v)
			case primitive.ObjectID:
				valueStr = "\"" + v.Hex() + "\""
			case primitive.DateTime:
				valueStr = "\"" + time.Unix(int64(v)/1000, int64(v)%1000*1000000).Format(time.RFC3339Nano) + "\""
			case nil:
				valueStr = "null"
			case bson.M, bson.D, map[string]interface{}, []interface{}:
				bytes, _ := json.Marshal(v)
				valueStr = string(bytes)
			default:
				valueStr = fmt.Sprintf("%v", v)
			}
			fields = append(fields, "\""+fieldName+"\":"+valueStr)
		}
		dataStr = "{" + strings.Join(fields, ",") + "}"

		// 创建数据项
		data := model.MongoSourceDBData{
			Id:     lastSyncedID.Hex(),
			Addr:   dataStr,
			Optype: "insert",
		}

		result.Data = append(result.Data, data)
	}

	if err := cursor.Err(); err != nil {
		logger.Error("游标遍历错误: %s", err.Error())
		return result, err
	}

	result.LastID = lastSyncedID
	totalTime := time.Since(queryStartTime)

	if len(result.Data) > 0 {
		// 计算平均处理时间
		avgProcessTime := totalTime / time.Duration(len(result.Data))
		logger.Info("MongoDB同步查询完成: 获取%d条记录, 最后ID=%v, 总耗时: %v, 平均每条处理时间: %v",
			len(result.Data), result.LastID, totalTime, avgProcessTime)

		// 添加第一条和最后一条记录的ID信息，方便追踪
		if len(result.Data) > 0 {
			firstID := result.Data[0].Id
			lastID := result.Data[len(result.Data)-1].Id
			logger.Info("本次查询数据范围: 第一条ID=%s, 最后一条ID=%s, 总计%d条记录",
				firstID, lastID, len(result.Data))
		}
	} else {
		logger.Info("MongoDB同步查询完成: 未获取到新记录, 总耗时: %v", totalTime)
	}

	// 即使查询结果为空也返回，让上层逻辑决定如何处理
	return result, nil
}

// RemoveTask 从任务管理器中移除任务
func (tm *MongoTaskManager) RemoveTask(taskName string) {
	tm.Mu.Lock()
	defer tm.Mu.Unlock()

	if cancel, exists := tm.Tasks[taskName]; exists {
		cancel()
		delete(tm.Tasks, taskName)
		delete(tm.Statuses, taskName)

		// 清理消息处理回调
		tm.ackHandlersMu.Lock()
		delete(tm.ackHandlers, taskName)
		tm.ackHandlersMu.Unlock()

		logger.Info("任务已移除: %s", taskName)
	}
}

// StopTask 停止指定的任务
func (tm *MongoTaskManager) StopTask(jobId string) bool {
	tm.Mu.Lock()
	defer tm.Mu.Unlock()

	if cancel, exists := tm.Tasks[jobId]; exists {
		cancel()
		delete(tm.Tasks, jobId)

		if status, ok := tm.Statuses[jobId]; ok {
			status.IsRunning = false
		}

		// 取消注册消息处理回调，避免任务停止后仍然处理消息
		tm.ackHandlersMu.Lock()
		delete(tm.ackHandlers, jobId)
		tm.ackHandlersMu.Unlock()

		logger.Info("任务已停止: %s", jobId)
		return true
	}
	return false
}

// StopAllTasks 停止所有任务
func (tm *MongoTaskManager) StopAllTasks() {
	tm.Mu.Lock()
	defer tm.Mu.Unlock()

	for taskName, cancel := range tm.Tasks {
		cancel()

		if status, ok := tm.Statuses[taskName]; ok {
			status.IsRunning = false
		}

		logger.Info("任务已停止: %s", taskName)
	}
	tm.Tasks = make(map[string]context.CancelFunc)

	// 清理所有注册的消息处理回调
	tm.ackHandlersMu.Lock()
	tm.ackHandlers = make(map[string]func(byte, []byte))
	tm.ackHandlersMu.Unlock()
}

// GetAllTaskIDs 获取所有任务ID
func (tm *MongoTaskManager) GetAllTaskIDs() []string {
	tm.Mu.Lock()
	defer tm.Mu.Unlock()

	taskIDs := make([]string, 0, len(tm.Tasks))
	for taskID := range tm.Tasks {
		taskIDs = append(taskIDs, taskID)
	}
	return taskIDs
}

// GetTaskStatus 获取指定任务的状态
func (tm *MongoTaskManager) GetTaskStatus(jobId string) (*MongoTaskStatus, bool) {
	tm.Mu.Lock()
	defer tm.Mu.Unlock()

	status, exists := tm.Statuses[jobId]
	return status, exists
}

// checkMongoDBConfiguration 检查 MongoDB 配置，特别是 Change Stream 相关配置
func (tm *MongoTaskManager) checkMongoDBConfiguration(ctx context.Context, client *mongo.Client, jobId string) {
	logger.WithFields(logrus.Fields{
		"job_id": jobId,
	}).Info("开始检查 MongoDB 配置")

	// 1. 检查 MongoDB 版本
	result := client.Database("admin").RunCommand(ctx, bson.D{{"buildInfo", 1}})
	var buildInfo bson.M
	if err := result.Decode(&buildInfo); err != nil {
		logger.WithFields(logrus.Fields{
			"job_id": jobId,
			"error":  err.Error(),
		}).Error("获取 MongoDB 版本信息失败")
	} else {
		if version, ok := buildInfo["version"]; ok {
			logger.WithFields(logrus.Fields{
				"job_id":  jobId,
				"version": version,
			}).Info("MongoDB 版本检查")
		}
	}

	// 2. 检查副本集状态（最重要的检查）
	result = client.Database("admin").RunCommand(ctx, bson.D{{"isMaster", 1}})
	var isMasterResult bson.M
	if err := result.Decode(&isMasterResult); err != nil {
		logger.WithFields(logrus.Fields{
			"job_id": jobId,
			"error":  err.Error(),
		}).Error("检查副本集状态失败")
	} else {
		if setName, exists := isMasterResult["setName"]; exists {
			isPrimary := false
			if isPrimaryVal, ok := isMasterResult["ismaster"].(bool); ok {
				isPrimary = isPrimaryVal
			}

			logger.WithFields(logrus.Fields{
				"job_id":     jobId,
				"set_name":   setName,
				"is_primary": isPrimary,
			}).Info("✅ MongoDB 副本集配置正常")
		} else {
			logger.WithFields(logrus.Fields{
				"job_id": jobId,
			}).Error("❌ MongoDB 未配置副本集 - 这是 Change Stream 不工作的根本原因！")

			logger.WithFields(logrus.Fields{
				"job_id": jobId,
			}).Error("解决方案: 1) 停止MongoDB 2) 配置副本集 3) 重启MongoDB 4) 初始化副本集")
		}
	}

	// 3. 检查 oplog 状态
	oplogCollection := client.Database("local").Collection("oplog.rs")
	oplogCount, err := oplogCollection.CountDocuments(ctx, bson.M{})
	if err != nil {
		logger.WithFields(logrus.Fields{
			"job_id": jobId,
			"error":  err.Error(),
		}).Error("检查 oplog 失败")
	} else if oplogCount > 0 {
		logger.WithFields(logrus.Fields{
			"job_id":      jobId,
			"oplog_count": oplogCount,
		}).Info("✅ Oplog 可用")

		// 获取最新的 oplog 条目
		cursor, err := oplogCollection.Find(ctx, bson.M{}, options.Find().SetSort(bson.D{{"ts", -1}}).SetLimit(1))
		if err == nil {
			defer cursor.Close(ctx)
			if cursor.Next(ctx) {
				var latestOplog bson.M
				if err := cursor.Decode(&latestOplog); err == nil {
					logger.WithFields(logrus.Fields{
						"job_id":            jobId,
						"latest_oplog_time": latestOplog["ts"],
					}).Info("最新 oplog 时间戳")
				}
			}
		}
	} else {
		logger.WithFields(logrus.Fields{
			"job_id": jobId,
		}).Error("❌ Oplog 不可用或为空")
	}

	// 4. 检查用户权限
	result = client.Database("admin").RunCommand(ctx, bson.D{{"connectionStatus", 1}})
	var connStatus bson.M
	if err := result.Decode(&connStatus); err != nil {
		logger.WithFields(logrus.Fields{
			"job_id": jobId,
			"error":  err.Error(),
		}).Error("检查连接状态失败")
	} else {
		if authInfo, ok := connStatus["authInfo"].(bson.M); ok {
			if authenticatedUsers, ok := authInfo["authenticatedUsers"].(bson.A); ok && len(authenticatedUsers) > 0 {
				logger.WithFields(logrus.Fields{
					"job_id":              jobId,
					"authenticated_users": authenticatedUsers,
				}).Info("✅ 用户认证信息")
			} else {
				logger.WithFields(logrus.Fields{
					"job_id": jobId,
				}).Info("⚠️ 未检测到已认证用户（可能使用匿名连接）")
			}
		}
	}

	// 5. 测试 Change Stream 创建能力
	testCollection := client.Database("test_changestream_config").Collection("test")
	changeStreamOptions := options.ChangeStream()
	changeStreamOptions.SetFullDocument(options.UpdateLookup)

	pipeline := mongo.Pipeline{
		{{
			"$match", bson.D{{
				"operationType", bson.D{{
					"$in", bson.A{"insert", "update", "delete"},
				}},
			}},
		}},
	}

	changeStream, err := testCollection.Watch(ctx, pipeline, changeStreamOptions)
	if err != nil {
		logger.WithFields(logrus.Fields{
			"job_id": jobId,
			"error":  err.Error(),
		}).Error("❌ Change Stream 创建测试失败")

		logger.WithFields(logrus.Fields{
			"job_id": jobId,
		}).Error("可能原因: 1) MongoDB版本过低(<3.6) 2) 未运行在副本集模式 3) 权限不足")
	} else {
		logger.WithFields(logrus.Fields{
			"job_id": jobId,
		}).Info("✅ Change Stream 创建测试成功")
		changeStream.Close(ctx)
	}

	logger.WithFields(logrus.Fields{
		"job_id": jobId,
	}).Info("MongoDB 配置检查完成")
}
