package database

import (
	"context"
	"database/sql"
	"encoding/json"
	"errors"
	"fmt"
	_ "github.com/go-sql-driver/mysql" // MySQL 驱动
	_ "github.com/lib/pq"              // PostgreSQL 驱动
	"github.com/olivere/elastic/v7"
	"github.com/segmentio/kafka-go"
	"go.mongodb.org/mongo-driver/mongo" // MongoDB 包
	"go.mongodb.org/mongo-driver/mongo/options"
	"io/ioutil"
	"log"
	"net/http"
)

var MongoClient *mongo.Client
var PgDB *sql.DB
var MySQLDB *sql.DB
var KafkaReader *kafka.Reader
var ElasticsearchClient *elastic.Client

func fetchConfig(url string) (map[string]interface{}, error) {
	resp, err := http.Get(url)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()

	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return nil, err
	}

	var config map[string]interface{}
	if err := json.Unmarshal(body, &config); err != nil {
		return nil, err
	}

	return config, nil
}

func ConnectMongoDB(url string) error {

	uri := `mongodb://` + url + `/?replicaSet=rs0`
	println(uri)
	clientOptions := options.Client().ApplyURI(uri)
	client, err := mongo.Connect(context.TODO(), clientOptions)
	if err != nil {
		println("ConnectMongoDB-->" + err.Error())
		return err
	}

	if err := client.Ping(context.TODO(), nil); err != nil {
		return err
	}

	MongoClient = client
	log.Println("成功连接到MongoDB")
	return nil
}

func ConnectPostgres() error {
	postgresConfigURL := "http://localhost:8080/config/postgresql"
	config, err := fetchConfig(postgresConfigURL)
	if err != nil {
		return err
	}

	connStr := fmt.Sprintf("host=%s port=%s user=%s password=%s dbname=%s sslmode=disable",
		config["host"],
		config["port"],
		config["user"],
		config["password"],
		config["dbname"],
	)

	db, err := sql.Open("postgres", connStr)
	if err != nil {
		return err
	}

	if err := db.Ping(); err != nil {
		return err
	}

	PgDB = db
	log.Println("成功连接到PostgreSQL")
	return nil
}

func ConnectMysql(host, user, password, port, dbname string) error {

	connStr := fmt.Sprintf("%s:%s@tcp(%s:%s)/%s",
		user,
		password,
		host,
		port,
		dbname,
	)
	println(connStr)
	db, err := sql.Open("mysql", connStr)
	if err != nil {
		log.Println("ConnectMysql-->" + err.Error())
		return err
	}
	defer db.Close()
	if err := db.Ping(); err != nil {
		log.Println("ConnectMysql-->" + err.Error())
		return err
	}

	var dbCluster DBClusterSearch
	err = db.QueryRow("SELECT id, name, status FROM db_clusters WHERE status = ?", "inuse").Scan(&dbCluster.ID, &dbCluster.Name, &dbCluster.Status)
	if err != nil {
		if errors.Is(err, sql.ErrNoRows) {
			log.Println("No DBCluster found with status 'inuse'")
			return nil
		}
		log.Println("db_cluster->ConnectMysql-->" + err.Error())
		return err
	}

	var compNode CompNodeSearch
	err = db.QueryRow(`
    SELECT id, name, hostaddr, port, mysql_port, user_name, passwd, db_cluster_id, svr_node_id, status, max_conns, max_mem_mb, cpu_cores, cpu_limit_mode
    FROM comp_nodes
    WHERE db_cluster_id = ?`, dbCluster.ID).Scan(
		&compNode.ID,
		&compNode.Name,
		&compNode.HostAddr,
		&compNode.Port,
		&compNode.MySQLPort,
		&compNode.UserName,
		&compNode.Passwd,
		&compNode.DBClusterID,
		&compNode.SvrNodeID,
		&compNode.Status,
		&compNode.MaxConns,
		&compNode.MaxMemMB,
		&compNode.CPUCores,
		&compNode.CPULimitMode,
	)
	if err != nil {
		log.Println("comp_node->ConnectMysql-->" + err.Error())
		return err
	}

	pgConnstr := fmt.Sprintf("host=%s port=%s user=%s password=%s dbname=%s sslmode=disable",
		compNode.HostAddr,
		compNode.Port,
		compNode.UserName,
		compNode.Passwd,
		"postgres",
	)

	println(pgConnstr)

	db, err = sql.Open("postgres", pgConnstr)
	if err != nil {
		log.Println("ConnectMysql-->" + err.Error())
		return err
	}

	if err := db.Ping(); err != nil {
		log.Println("ConnectMysql-->" + err.Error())
		return err
	}

	PgDB = db
	log.Println("成功连接到PostgreSQL")
	return nil
}

func ConnectMySQL() error {
	mysqlConfigURL := "http://localhost:8080/config/mysql"
	config, err := fetchConfig(mysqlConfigURL)
	if err != nil {
		return err
	}

	connStr := fmt.Sprintf("%s:%s@tcp(%s:%s)/%s",
		config["user"],
		config["password"],
		config["host"],
		config["port"],
		config["dbname"],
	)

	db, err := sql.Open("mysql", connStr)
	if err != nil {
		return err
	}

	if err := db.Ping(); err != nil {
		return err
	}

	MySQLDB = db
	log.Println("成功连接到MySQL")
	return nil
}

func ConnectKafka() error {
	kafkaConfigURL := "http://localhost:8080/config/kafka"
	config, err := fetchConfig(kafkaConfigURL)
	if err != nil {
		return err
	}

	brokers := config["bootstrap_servers"].(string)
	topicPrefix := config["topic_prefix"].(string)

	KafkaReader = kafka.NewReader(kafka.ReaderConfig{
		Brokers: []string{brokers},
		GroupID: "example-group",
		Topic:   topicPrefix + "-example-topic",
	})

	log.Println("成功连接到Kafka")
	return nil
}

func ConnectElasticsearch() error {
	esConfigURL := "http://localhost:8080/config/elasticsearch"
	config, err := fetchConfig(esConfigURL)
	if err != nil {
		return err
	}

	client, err := elastic.NewClient(elastic.SetURL(config["host"].(string)))
	if err != nil {
		return err
	}

	ElasticsearchClient = client
	log.Println("成功连接到Elasticsearch")
	return nil
}
