<?php
defined('BASEPATH') or exit('No direct script access allowed');

class GrafanaAlerts extends MY_Controller
{
    public $grafana_url;
    public $grafana_api_key;
    public $prometheus_url;
    public $node_exporter_port;
    public $AlarmCategory_model;
    public $db;
    public $config;
    public $input;
    public $Cluster_model; // 添加Cluster_model公共属性，解决访问权限问题
    public $Login_model; // 添加Login_model公共属性，解决访问权限问题
    public $api_response; // 添加api_response公共属性，解决访问权限问题

    public function __construct()
    {
        parent::__construct();
        
        // 加载数据库
        $this->load->database();
        
        // 加载告警分类模型
        $this->load->model('AlarmCategory_model');
        
        // 加载集群模型
        $this->load->model('Cluster_model');
        
        // 加载用户模型 (如果需要)
        $this->load->model('Login_model');
        
        // 加载辅助函数
        $this->load->helper('url');
        
        // 获取配置项
        $this->load->config('myconfig');
        $grafana_url = $this->config->item('grafana_url');
        $this->grafana_url = ($grafana_url !== FALSE) ? $grafana_url : 'http://192.168.0.21:18080';
        
        $grafana_api_key = $this->config->item('grafana_api_key');
        $this->grafana_api_key = ($grafana_api_key !== FALSE) ? $grafana_api_key : '';
        
        // 从数据库中获取source节点的Prometheus信息
        $query = $this->db->query("SELECT hostaddr, prometheus_port FROM cluster_mgr_nodes WHERE member_state = 'source' LIMIT 1");
        if ($query->num_rows() > 0) {
            $row = $query->row();
            // 配置Prometheus服务器地址和节点导出器端口
            $this->prometheus_url = 'http://' . $row->hostaddr . ':' . $row->prometheus_port;
            $this->node_exporter_port = $row->prometheus_port - 1000; // 假设node_exporter_port比prometheus_port小1000
            log_message('debug', '从数据库获取Prometheus信息：' . $this->prometheus_url . '，节点导出器端口：' . $this->node_exporter_port);
        } else {
            // 使用默认值
            $this->prometheus_url = 'http://192.168.0.21:59024';
            $this->node_exporter_port = '58024';
            log_message('debug', '使用默认Prometheus信息：' . $this->prometheus_url . '，节点导出器端口：' . $this->node_exporter_port);
        }
    }

    /**
     * 获取 Grafana 告警列表
     * 通过 Grafana API 获取所有告警规则
     */
    public function getAlertRules()
    {
        // 记录请求信息
        log_message('debug', 'getAlertRules 方法被调用，IP：' . $this->input->ip_address());
        
        // 获取 Grafana 告警规则
        $url = $this->grafana_url . '/api/v1/provisioning/alert-rules';
        $headers = [
            'Authorization: Bearer ' . $this->grafana_api_key,
            'Content-Type: application/json',
            'Accept: application/json'
        ];

        log_message('debug', 'Grafana API 请求：' . $url);

        $response = $this->_curlRequest($url, 'GET', [], $headers);
        $alerts = json_decode($response, true);

        // 获取告警分类信息
        $alarmCategories = $this->AlarmCategory_model->getCategories();
        $alarmTypes = $this->AlarmCategory_model->getAlarmTypeDetails();

        // 将 Grafana 告警与系统告警类型进行匹配
        $result = [];
        if (!empty($alerts) && isset($alerts['data'])) {
            foreach ($alerts['data'] as $alert) {
                // 尝试匹配告警类型
                $matchedType = null;
                foreach ($alarmTypes as $type => $details) {
                    if (stripos($alert['title'], $details['label']) !== false) {
                        $matchedType = $type;
                        break;
                    }
                }

                // 获取分类信息
                $category = null;
                if ($matchedType) {
                    $category = $this->AlarmCategory_model->getCategoryByType($matchedType);
                }

                $result[] = [
                    'id' => $alert['id'] ?? '',
                    'uid' => $alert['uid'] ?? '',
                    'title' => $alert['title'] ?? '',
                    'condition' => $alert['condition'] ?? '',
                    'data' => $alert['data'] ?? [],
                    'state' => $alert['state'] ?? 'normal',
                    'type' => $matchedType,
                    'category' => $category,
                    'labels' => $alert['labels'] ?? [],
                    'annotations' => $alert['annotations'] ?? [],
                    'created' => $alert['created'] ?? '',
                    'updated' => $alert['updated'] ?? ''
                ];
            }
        }

        echo json_encode([
            'code' => 200,
            'message' => '获取 Grafana 告警规则成功',
            'data' => $result
        ]);
    }

    /**
     * 获取当前告警状态
     * 通过 Grafana API 获取当前正在告警的实例
     */
    public function getAlertInstances()
    {
        $url = $this->grafana_url . '/api/v1/alerts';
        $headers = [
            'Authorization: Bearer ' . $this->grafana_api_key,
            'Content-Type: application/json',
            'Accept: application/json'
        ];

        $response = $this->_curlRequest($url, 'GET', [], $headers);
        $alertInstances = json_decode($response, true);

        // 获取告警类型详细信息
        $alarmTypes = $this->AlarmCategory_model->getAlarmTypeDetails();

        // 处理告警实例数据
        $result = [];
        if (!empty($alertInstances) && isset($alertInstances['data'])) {
            foreach ($alertInstances['data'] as $instance) {
                // 尝试匹配告警类型
                $matchedType = null;
                foreach ($alarmTypes as $type => $details) {
                    if (isset($instance['labels']['alertname']) && 
                        stripos($instance['labels']['alertname'], $details['label']) !== false) {
                        $matchedType = $type;
                        break;
                    }
                }

                // 获取告警级别
                $level = 'WARNING';
                if ($matchedType && isset($alarmTypes[$matchedType]['level'])) {
                    $level = $alarmTypes[$matchedType]['level'];
                }

                // 获取告警分类
                $category = null;
                if ($matchedType) {
                    $category = $this->AlarmCategory_model->getCategoryByType($matchedType);
                }

                $result[] = [
                    'alertname' => $instance['labels']['alertname'] ?? '',
                    'status' => $instance['status']['state'] ?? '',
                    'active_at' => $instance['status']['startsAt'] ?? '',
                    'value' => $instance['value'] ?? '',
                    'labels' => $instance['labels'] ?? [],
                    'annotations' => $instance['annotations'] ?? [],
                    'type' => $matchedType,
                    'level' => $level,
                    'category' => $category
                ];
            }
        }

        echo json_encode([
            'code' => 200,
            'message' => '获取当前告警实例成功',
            'data' => $result
        ]);
    }

    /**
     * 获取系统监控告警配置
     * 返回系统监控相关的告警配置，包括告警类型和阈值
     */
    public function getSystemMonitorAlerts()
    {
        // 获取系统监控告警分类
        $categories = $this->AlarmCategory_model->getCategories();
        $systemCategory = null;
        
        foreach ($categories as $category) {
            if ($category['id'] === 'system_monitor') {
                $systemCategory = $category;
                break;
            }
        }
        
        if (!$systemCategory) {
            echo json_encode([
                'code' => 404,
                'message' => '系统监控告警分类不存在',
                'data' => null
            ]);
            return;
        }
        
        // 获取告警类型详情
        $alarmTypes = $this->AlarmCategory_model->getAlarmTypeDetails();
        $systemAlerts = [];
        
        foreach ($systemCategory['types'] as $type) {
            if (isset($alarmTypes[$type])) {
                $systemAlerts[] = [
                    'type' => $type,
                    'label' => $alarmTypes[$type]['label'],
                    'level' => $alarmTypes[$type]['level'],
                    'trigger_condition' => $alarmTypes[$type]['trigger_condition'],
                    'monitor_metric' => $alarmTypes[$type]['monitor_metric']
                ];
            }
        }
        
        echo json_encode([
            'code' => 200,
            'message' => '获取系统监控告警配置成功',
            'data' => [
                'category' => $systemCategory,
                'alerts' => $systemAlerts
            ]
        ]);
    }

   
    /**
     * 获取告警历史记录
     * 从 Grafana 获取历史告警记录
     */
    public function getAlertHistory()
    {
        $url = $this->grafana_url . '/api/v1/rules/history';
        $headers = [
            'Authorization: Bearer ' . $this->grafana_api_key,
            'Content-Type: application/json',
            'Accept: application/json'
        ];
        
        // 获取查询参数
        $limit = $this->input->get('limit') ?: 100;
        $from = $this->input->get('from') ?: '';
        $to = $this->input->get('to') ?: '';
        
        $params = [];
        if (!empty($limit)) {
            $params['limit'] = $limit;
        }
        if (!empty($from)) {
            $params['from'] = $from;
        }
        if (!empty($to)) {
            $params['to'] = $to;
        }
        
        $queryString = http_build_query($params);
        if (!empty($queryString)) {
            $url .= '?' . $queryString;
        }

        $response = $this->_curlRequest($url, 'GET', [], $headers);
        $history = json_decode($response, true);

        // 处理结果
        $result = [];
        if (!empty($history) && isset($history['data']) && isset($history['data']['results'])) {
            foreach ($history['data']['results'] as $item) {
                $result[] = [
                    'id' => $item['id'] ?? '',
                    'rule_uid' => $item['rule_uid'] ?? '',
                    'rule_name' => $item['rule_name'] ?? '',
                    'state' => $item['state'] ?? '',
                    'state_from' => $item['state_from'] ?? '',
                    'timestamp' => $item['timestamp'] ?? '',
                    'values' => $item['values'] ?? [],
                    'annotations' => $item['annotations'] ?? []
                ];
            }
        }

        echo json_encode([
            'code' => 200,
            'message' => '获取告警历史记录成功',
            'data' => $result
        ]);
    }

    /**
     * 获取集群节点告警
     * 获取与特定节点相关的告警，通过 IP 筛选
     * 支持多个 IP 地址，以逗号分隔
     */
    public function getNodeAlerts()
    {
        $ip_input = $this->input->get('ip');
        
        // 记录请求信息
        log_message('debug', 'getNodeAlerts 方法被调用，请求参数：ip=' . $ip_input . '，IP：' . $this->input->ip_address());
        log_message('debug', 'Prometheus 服务器地址：' . $this->prometheus_url . '，节点导出器端口：' . $this->node_exporter_port);
        
        if (empty($ip_input)) {
            log_message('error', 'getNodeAlerts 缺少参数：ip');
            echo json_encode([
                'code' => 400,
                'message' => '参数错误，请提供节点IP',
                'data' => null
            ]);
            return;
        }
        
        // 解析多个 IP，以逗号分隔
        $ip_list = array_map('trim', explode(',', $ip_input));
        log_message('debug', 'getNodeAlerts 解析的IP列表：' . json_encode($ip_list));
        
        // 存储所有节点的告警数据
        $all_alerts = [];
        $nodes_info = [];
        
        // 获取当前告警实例
        $url = $this->prometheus_url . '/api/v1/alerts';
        log_message('debug', 'Grafana 告警API请求：' . $url);
        
        $headers = [
            'Authorization: Bearer ' . $this->grafana_api_key,
            'Content-Type: application/json',
            'Accept: application/json'
        ];

        $response = $this->_curlRequest($url, 'GET', [], $headers);
        log_message('debug', 'Grafana 告警API响应数据长度：' . strlen($response));
        
        $alertInstances = json_decode($response, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            log_message('error', 'JSON解析错误：' . json_last_error_msg() . '，响应数据：' . substr($response, 0, 500));
        } else {
            log_message('debug', 'Grafana 告警实例数量：' . (isset($alertInstances['data']) ? count($alertInstances['data']) : 0));
        }
        
        // 遍历每个 IP，筛选相关告警
        foreach ($ip_list as $ip) {
            $instance_pattern = $ip . ':' . $this->node_exporter_port;
            $nodes_info[] = $instance_pattern;
            
            // 过滤该节点的告警
            $nodeAlerts = [];
            if (!empty($alertInstances) && isset($alertInstances['data'])) {
                foreach ($alertInstances['data'] as $instance) {
                    // 检查是否与指定节点相关
                    $isRelated = false;
                    if (isset($instance['labels']['instance']) && 
                        (strpos($instance['labels']['instance'], $ip) !== false)) {
                        $isRelated = true;
                    } elseif (isset($instance['labels']['node']) && 
                        (strpos($instance['labels']['node'], $ip) !== false)) {
                        $isRelated = true;
                    }
                    
                    if ($isRelated) {
                        $nodeAlerts[] = [
                            'node_ip' => $ip,
                            'instance' => $instance_pattern,
                            'alertname' => $instance['labels']['alertname'] ?? '',
                            'status' => $instance['status']['state'] ?? '',
                            'active_at' => $instance['status']['startsAt'] ?? '',
                            'value' => $instance['value'] ?? '',
                            'labels' => $instance['labels'] ?? [],
                            'annotations' => $instance['annotations'] ?? []
                        ];
                    }
                }
            }
            
            // 即使没有告警，也添加节点信息
            if (empty($nodeAlerts)) {
                $nodeInfo = [
                    'node_ip' => $ip,
                    'instance' => $instance_pattern,
                    'alert_count' => 0,
                    'has_alerts' => false
                ];
            } else {
                $nodeInfo = [
                    'node_ip' => $ip,
                    'instance' => $instance_pattern,
                    'alert_count' => count($nodeAlerts),
                    'has_alerts' => true,
                    'alerts' => $nodeAlerts
                ];
            }
            
            $all_alerts[$ip] = $nodeInfo;
        }
        
        echo json_encode([
            'code' => 200,
            'message' => '获取节点告警成功',
            'data' => $all_alerts,
            'node_count' => count($ip_list),
            'total_alert_count' => array_sum(array_column($all_alerts, 'alert_count')),
            'timestamp' => time()
        ]);
    }

    /**
     * 发送 cURL 请求的辅助方法
     */
    private function _curlRequest($url, $method = 'GET', $data = [], $headers = [])
    {
        log_message('debug', 'cURL 请求开始：' . $method . ' ' . $url);
        
        $ch = curl_init();
        
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 5);
        curl_setopt($ch, CURLOPT_TIMEOUT, 20);
        
        if ($method === 'POST') {
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        } elseif ($method !== 'GET') {
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $method);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        }
        
        if (!empty($headers)) {
            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
            log_message('debug', 'cURL 请求头：' . json_encode($headers));
        }
        
        $response = curl_exec($ch);
        
        if (curl_errno($ch)) {
            $error = curl_error($ch);
            log_message('error', 'cURL 错误：' . $error);
            $response = json_encode([
                'error' => 'cURL Error: ' . $error
            ]);
        } else {
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $requestTime = curl_getinfo($ch, CURLINFO_TOTAL_TIME);
            log_message('debug', 'cURL 请求完成：HTTP状态码=' . $httpCode . '，请求时间=' . $requestTime . '秒');
        }
        
        curl_close($ch);
        
        return $response;
    }

    /**
     * 从 Prometheus 获取指标数据
     * 基于 node.json 中定义的监控指标，直接从 Prometheus 获取数据
     * 支持多个 IP 地址，以逗号分隔
     */
    public function getPrometheusMetrics()
    {
        // 获取参数
        $ip_input = $this->input->get('ip');
        $metric = $this->input->get('metric') ?: 'cpu_usage';
        $time = $this->input->get('time') ?: '';
        
        if (empty($ip_input)) {
            echo json_encode([
                'code' => 400,
                'message' => '参数错误，请提供节点IP',
                'data' => null
            ]);
            return;
        }
        
        // 解析多个 IP，以逗号分隔
        $ip_list = array_map('trim', explode(',', $ip_input));
        
        // 存储所有节点的数据 - 按节点IP组织
        $nodes_data = [];
        
        // 格式化指标标签
        $metric_label = $this->_getMetricLabel($metric);
        
        // 遍历每个 IP 获取数据
        foreach ($ip_list as $ip) {
            // 根据实际 Prometheus 实例格式修改查询表达式
            $instance_pattern = $ip . ':' . $this->node_exporter_port;
            
            // 定义 node.json 中的指标对应的 PromQL 表达式
            $expressions = [
                // CPU 指标
                'cpu_usage' => '100 - (avg(irate(node_cpu_seconds_total{instance="' . $instance_pattern . '",mode="idle"}[5m])) * 100)',
                'cpu_system' => 'avg(irate(node_cpu_seconds_total{instance="' . $instance_pattern . '",mode="system"}[5m])) by (instance) * 100',
                'cpu_user' => 'avg(irate(node_cpu_seconds_total{instance="' . $instance_pattern . '",mode="user"}[5m])) by (instance) * 100',
                'cpu_iowait' => 'avg(irate(node_cpu_seconds_total{instance="' . $instance_pattern . '",mode="iowait"}[5m])) by (instance) * 100',
                'cpu_count' => 'count(node_cpu_seconds_total{instance="' . $instance_pattern . '", mode="system"}) by (cpu)',
                
                // 内存指标
                'memory_total' => 'node_memory_MemTotal_bytes{instance="' . $instance_pattern . '"}',
                'memory_used' => 'node_memory_MemTotal_bytes{instance="' . $instance_pattern . '"} - node_memory_MemAvailable_bytes{instance="' . $instance_pattern . '"}',
                'memory_free' => 'node_memory_MemFree_bytes{instance="' . $instance_pattern . '"}',
                'memory_available' => 'node_memory_MemAvailable_bytes{instance="' . $instance_pattern . '"}',
                'memory_cached' => 'node_memory_Cached_bytes{instance="' . $instance_pattern . '"}',
                'memory_buffers' => 'node_memory_Buffers_bytes{instance="' . $instance_pattern . '"}',
                'memory_usage' => '(1 - (node_memory_MemAvailable_bytes{instance="' . $instance_pattern . '"} / node_memory_MemTotal_bytes{instance="' . $instance_pattern . '"})) * 100',
                'swap_usage' => '(1 - (node_memory_SwapFree_bytes{instance="' . $instance_pattern . '"} / node_memory_SwapTotal_bytes{instance="' . $instance_pattern . '"})) * 100',
                
                // 磁盘指标
                'disk_total' => 'node_filesystem_size_bytes{instance="' . $instance_pattern . '",fstype=~"ext4|xfs"}',
                'disk_free' => 'node_filesystem_free_bytes{instance="' . $instance_pattern . '",fstype=~"ext4|xfs"}',
                'disk_usage' => '1-(node_filesystem_free_bytes{instance="' . $instance_pattern . '",fstype=~"ext4|xfs"} / node_filesystem_size_bytes{instance="' . $instance_pattern . '",fstype=~"ext4|xfs"})',
                'disk_io' => 'irate(node_disk_io_time_seconds_total{instance="' . $instance_pattern . '"}[5m])',
                'disk_read_iops' => 'irate(node_disk_reads_completed_total{instance="' . $instance_pattern . '"}[5m])',
                'disk_write_iops' => 'irate(node_disk_writes_completed_total{instance="' . $instance_pattern . '"}[5m])',
                'disk_read_bytes' => 'irate(node_disk_read_bytes_total{instance="' . $instance_pattern . '"}[5m])',
                'disk_write_bytes' => 'irate(node_disk_written_bytes_total{instance="' . $instance_pattern . '"}[5m])',
                'disk_read_time' => 'irate(node_disk_read_time_seconds_total{instance="' . $instance_pattern . '"}[5m]) / irate(node_disk_reads_completed_total{instance="' . $instance_pattern . '"}[5m])',
                'disk_write_time' => 'irate(node_disk_write_time_seconds_total{instance="' . $instance_pattern . '"}[5m]) / irate(node_disk_writes_completed_total{instance="' . $instance_pattern . '"}[5m])',
                
                // 网络指标
                'network_in' => 'irate(node_network_receive_bytes_total{instance="' . $instance_pattern . '",device!~"tap.*|veth.*|br.*|docker.*|virbr*|lo*"}[5m])*8',
                'network_out' => 'irate(node_network_transmit_bytes_total{instance="' . $instance_pattern . '",device!~"tap.*|veth.*|br.*|docker.*|virbr*|lo*"}[5m])*8',
                'tcp_connections' => 'node_netstat_Tcp_CurrEstab{instance="' . $instance_pattern . '"}',
                'tcp_timewaits' => 'node_sockstat_TCP_tw{instance="' . $instance_pattern . '"}',
                'sockets_used' => 'node_sockstat_sockets_used{instance="' . $instance_pattern . '"}',
                
                // 系统负载
                'load1' => 'node_load1{instance="' . $instance_pattern . '"}',
                'load5' => 'node_load5{instance="' . $instance_pattern . '"}',
                'load15' => 'node_load15{instance="' . $instance_pattern . '"}',
                'uptime' => 'time() - node_boot_time_seconds{instance="' . $instance_pattern . '"}',
                'file_descriptors' => 'node_filefd_allocated{instance="' . $instance_pattern . '"}',
                'context_switches' => 'irate(node_context_switches_total{instance="' . $instance_pattern . '"}[5m])',
                
                // 硬件温度
                'temperature' => 'node_hwmon_temp_celsius{instance="' . $instance_pattern . '"}'
            ];
            
            if (!isset($expressions[$metric])) {
                echo json_encode([
                    'code' => 400,
                    'message' => '不支持的指标：' . $metric,
                    'data' => null
                ]);
                return;
            }
            
            // 构建查询参数
            $query_params = ['query' => $expressions[$metric]];
            if (!empty($time)) {
                $query_params['time'] = $time;
            }
            
            // 构建 URL
            $url = $this->prometheus_url . '/api/v1/query?' . http_build_query($query_params);
            
            // 发送请求
            $headers = [
                'Accept: application/json'
            ];
            $response = $this->_curlRequest($url, 'GET', [], $headers);
            $result = json_decode($response, true);
            
            // 初始化该节点的数据结构
            $nodes_data[$ip] = [
                'node_ip' => $ip,
                'instance' => $instance_pattern,
                'metric_name' => $metric,
                'metric_label' => $metric_label,
                'query' => $expressions[$metric],
                'timestamp' => time(),
                'status' => 'success',
                'data' => []
            ];
            
            // 处理返回结果
            if (!empty($result) && isset($result['status']) && $result['status'] === 'success' && !empty($result['data']['result'])) {
                foreach ($result['data']['result'] as $item) {
                    $value = null;
                    if (isset($item['value']) && is_array($item['value']) && count($item['value']) > 1) {
                        $value = $item['value'][1];
                        // 如果是数字，尝试转换为浮点数
                        if (is_numeric($value)) {
                            $value = (float)$value;
                            // 保留两位小数
                            $value = round($value, 2);
                        }
                    }
                    
                    // 保存完整的指标元数据
                    $data_item = [
                        'metric_details' => isset($item['metric']) ? $item['metric'] : [],
                        'value' => $value,
                        'timestamp' => isset($item['value']) && is_array($item['value']) ? $item['value'][0] : null,
                        'formatted_value' => $this->_formatMetricValue($metric, $value),
                        'status' => $this->_getMetricStatus($metric, $value)
                    ];
                    
                    $nodes_data[$ip]['data'][] = $data_item;
                }
            }
            
            // 如果没有数据，添加错误信息
            if (empty($nodes_data[$ip]['data'])) {
                $nodes_data[$ip]['status'] = 'error';
                $nodes_data[$ip]['error_message'] = '节点未返回数据';
                
                $nodes_data[$ip]['data'][] = [
                    'metric_details' => [],
                    'value' => null,
                    'timestamp' => time(),
                    'formatted_value' => '未知',
                    'status' => 'unknown'
                ];
            }
        }
        
        echo json_encode([
            'code' => 200,
            'message' => '获取指标数据成功',
            'metric_name' => $metric,
            'metric_label' => $metric_label,
            'data' => $nodes_data,
            'node_count' => count($ip_list),
            'timestamp' => time()
        ]);
    }
    
    /**
     * 格式化指标值的显示
     */
    private function _formatMetricValue($metric, $value)
    {
        if ($value === null) {
            return '未知';
        }
        
        // 根据指标类型添加单位和格式化
        switch ($metric) {
            case 'cpu_usage':
            case 'cpu_system':
            case 'cpu_user':
            case 'cpu_iowait':
            case 'memory_usage':
            case 'swap_usage':
            case 'disk_usage':
                return round($value, 2) . '%';
                
            case 'memory_total':
            case 'memory_used':
            case 'memory_free':
            case 'memory_available':
            case 'memory_cached':
            case 'memory_buffers':
            case 'disk_total':
            case 'disk_free':
                return $this->_formatBytes($value);
                
            case 'network_in':
            case 'network_out':
                return $this->_formatBits($value);
                
            case 'disk_read_bytes':
            case 'disk_write_bytes':
                return $this->_formatBytes($value) . '/s';
                
            case 'uptime':
                return $this->_formatUptime($value);
                
            case 'disk_read_time':
            case 'disk_write_time':
                return round($value * 1000, 2) . 'ms';
                
            default:
                return $value;
        }
    }
    
    /**
     * 获取指标的显示名称
     */
    private function _getMetricLabel($metric)
    {
        $labels = [
            'cpu_usage' => 'CPU使用率',
            'cpu_system' => '系统CPU使用率',
            'cpu_user' => '用户CPU使用率',
            'cpu_iowait' => 'IO等待时间',
            'cpu_count' => 'CPU核心数',
            'memory_total' => '总内存',
            'memory_used' => '已用内存',
            'memory_free' => '空闲内存',
            'memory_available' => '可用内存',
            'memory_cached' => '缓存',
            'memory_buffers' => '缓冲区',
            'memory_usage' => '内存使用率',
            'swap_usage' => '交换分区使用率',
            'disk_total' => '磁盘总容量',
            'disk_free' => '磁盘可用空间',
            'disk_usage' => '磁盘使用率',
            'disk_io' => '磁盘IO时间',
            'disk_read_iops' => '磁盘读取IOPS',
            'disk_write_iops' => '磁盘写入IOPS',
            'disk_read_bytes' => '磁盘读取速度',
            'disk_write_bytes' => '磁盘写入速度',
            'disk_read_time' => '磁盘读取延迟',
            'disk_write_time' => '磁盘写入延迟',
            'network_in' => '网络接收速率',
            'network_out' => '网络发送速率',
            'tcp_connections' => 'TCP连接数',
            'tcp_timewaits' => 'TCP TIME_WAIT数',
            'sockets_used' => '已用套接字数',
            'load1' => '1分钟负载',
            'load5' => '5分钟负载',
            'load15' => '15分钟负载',
            'uptime' => '运行时间',
            'file_descriptors' => '文件描述符数',
            'context_switches' => '每秒上下文切换数',
            'temperature' => '硬件温度'
        ];
        
        return isset($labels[$metric]) ? $labels[$metric] : $metric;
    }
    
    /**
     * 格式化字节数
     */
    private function _formatBytes($bytes)
    {
        if ($bytes === null) {
            return '未知';
        }
        
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];
        $bytes = max($bytes, 0);
        $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
        $pow = min($pow, count($units) - 1);
        
        $bytes /= pow(1024, $pow);
        
        return round($bytes, 2) . ' ' . $units[$pow];
    }
    
    /**
     * 格式化比特数
     */
    private function _formatBits($bits)
    {
        if ($bits === null) {
            return '未知';
        }
        
        $units = ['bps', 'Kbps', 'Mbps', 'Gbps', 'Tbps'];
        $bits = max($bits, 0);
        $pow = floor(($bits ? log($bits) : 0) / log(1000));
        $pow = min($pow, count($units) - 1);
        
        $bits /= pow(1000, $pow);
        
        return round($bits, 2) . ' ' . $units[$pow];
    }
    
    /**
     * 格式化运行时间
     */
    private function _formatUptime($seconds)
    {
        if (!is_numeric($seconds)) {
            return '未知';
        }
        
        $seconds = (int)$seconds;
        $days = floor($seconds / 86400);
        $hours = floor(($seconds % 86400) / 3600);
        $minutes = floor(($seconds % 3600) / 60);
        
        $result = '';
        if ($days > 0) {
            $result .= $days . '天';
        }
        if ($hours > 0) {
            $result .= $hours . '小时';
        }
        if ($minutes > 0) {
            $result .= $minutes . '分钟';
        }
        
        return $result ?: '刚刚启动';
    }

    /**
     * 获取节点所有关键指标
     * 一次性获取节点的所有关键监控指标
     * 支持多个 IP 地址，以逗号分隔
     */
    public function getNodeMetrics()
    {
        // 获取参数
        $ip_input = $this->input->get('ip');
        
        // 记录请求信息
        log_message('debug', 'getNodeMetrics 方法被调用，请求参数：ip=' . $ip_input . '，IP：' . $this->input->ip_address());
        log_message('debug', 'Prometheus 服务器地址：' . $this->prometheus_url . '，节点导出器端口：' . $this->node_exporter_port);
        
        if (empty($ip_input)) {
            log_message('error', 'getNodeMetrics 缺少参数：ip');
            echo json_encode([
                'code' => 400,
                'message' => '参数错误，请提供节点IP',
                'data' => null
            ]);
            return;
        }
        
        // 解析多个 IP，以逗号分隔
        $ip_list = array_map('trim', explode(',', $ip_input));
        log_message('debug', 'getNodeMetrics 解析的IP列表：' . json_encode($ip_list));
        
        // 使用统一的 Prometheus 服务器地址配置
        $prometheus_url = $this->prometheus_url;
        log_message('debug', 'Prometheus 服务器地址：' . $prometheus_url);
        
        // 存储所有节点的数据
        $nodes_data = [];
        
        // 遍历每个 IP 获取数据
        foreach ($ip_list as $ip) {
            // 使用统一的节点导出器端口配置
            $instance_pattern = $ip . ':' . $this->node_exporter_port;
            log_message('debug', '处理节点：' . $ip . '，实例模式：' . $instance_pattern);
            
            // 定义要获取的关键指标
            $key_metrics = [
                'cpu_usage', 
                'memory_usage', 
                'disk_usage', 
                'load1', 
                'network_in', 
                'network_out',
                'uptime'
            ];
            
            $results = [];
            
            // 定义 node.json 中的指标对应的 PromQL 表达式
            $expressions = [
                'cpu_usage' => '100 - (avg(irate(node_cpu_seconds_total{instance="' . $instance_pattern . '",mode="idle"}[5m])) * 100)',
                'memory_usage' => '(1 - (node_memory_MemAvailable_bytes{instance="' . $instance_pattern . '"} / node_memory_MemTotal_bytes{instance="' . $instance_pattern . '"})) * 100',
                'disk_usage' => '1-(node_filesystem_free_bytes{instance="' . $instance_pattern . '",fstype=~"ext4|xfs"} / node_filesystem_size_bytes{instance="' . $instance_pattern . '",fstype=~"ext4|xfs"})',
                'load1' => 'node_load1{instance="' . $instance_pattern . '"}',
                'network_in' => 'sum(irate(node_network_receive_bytes_total{instance="' . $instance_pattern . '",device!~"tap.*|veth.*|br.*|docker.*|virbr*|lo*"}[5m]))',
                'network_out' => 'sum(irate(node_network_transmit_bytes_total{instance="' . $instance_pattern . '",device!~"tap.*|veth.*|br.*|docker.*|virbr*|lo*"}[5m]))',
                'uptime' => 'time() - node_boot_time_seconds{instance="' . $instance_pattern . '"}'
            ];
            
            // 发送请求获取每个指标
            foreach ($key_metrics as $metric) {
                if (!isset($expressions[$metric])) {
                    log_message('warning', '未知的指标：' . $metric);
                    continue;
                }
                
                $query_params = ['query' => $expressions[$metric]];
                $url = $prometheus_url . '/api/v1/query?' . http_build_query($query_params);
                
                log_message('debug', '请求 Prometheus 指标：' . $metric . '，URL：' . $url);
                
                $headers = [
                    'Accept: application/json'
                ];
                $response = $this->_curlRequest($url, 'GET', [], $headers);
                log_message('debug', 'Prometheus 响应数据长度：' . strlen($response));
                
                $result = json_decode($response, true);
                
                if (json_last_error() !== JSON_ERROR_NONE) {
                    log_message('error', 'JSON解析错误：' . json_last_error_msg() . '，响应数据：' . substr($response, 0, 500));
                }
                
                $value = null;
                if (!empty($result) && isset($result['status']) && $result['status'] === 'success' && !empty($result['data']['result'])) {
                    if (isset($result['data']['result'][0]['value']) && is_array($result['data']['result'][0]['value']) && count($result['data']['result'][0]['value']) > 1) {
                        $value = $result['data']['result'][0]['value'][1];
                        // 如果是数字，尝试转换为浮点数
                        if (is_numeric($value)) {
                            $value = (float)$value;
                            // 保留两位小数
                            $value = round($value, 2);
                        }
                        log_message('debug', '指标 ' . $metric . ' 的值：' . $value);
                    } else {
                        log_message('debug', '指标 ' . $metric . ' 无有效值，result结构：' . json_encode($result['data']['result']));
                    }
                } else {
                    log_message('warning', '指标 ' . $metric . ' 请求失败或无数据，状态：' . ($result['status'] ?? 'unknown'));
                }
                
                $results[$metric] = $value;
            }
            
            // 处理结果格式，添加标签和单位
            $formatted_results = [
                'cpu_usage' => [
                    'label' => 'CPU使用率',
                    'value' => $results['cpu_usage'],
                    'unit' => '%',
                    'formatted_value' => isset($results['cpu_usage']) ? $results['cpu_usage'] . '%' : '未知',
                    'status' => $this->_getMetricStatus('cpu_usage', $results['cpu_usage'])
                ],
                'memory_usage' => [
                    'label' => '内存使用率',
                    'value' => $results['memory_usage'],
                    'unit' => '%',
                    'formatted_value' => isset($results['memory_usage']) ? $results['memory_usage'] . '%' : '未知',
                    'status' => $this->_getMetricStatus('memory_usage', $results['memory_usage'])
                ],
                'disk_usage' => [
                    'label' => '磁盘使用率',
                    'value' => $results['disk_usage'],
                    'unit' => '%',
                    'formatted_value' => isset($results['disk_usage']) ? $results['disk_usage'] . '%' : '未知',
                    'status' => $this->_getMetricStatus('disk_usage', $results['disk_usage'])
                ],
                'load1' => [
                    'label' => '系统负载',
                    'value' => $results['load1'],
                    'unit' => '',
                    'formatted_value' => isset($results['load1']) ? $results['load1'] : '未知',
                    'status' => $this->_getMetricStatus('load1', $results['load1'])
                ],
                'network_in' => [
                    'label' => '网络入流量',
                    'value' => $results['network_in'],
                    'unit' => 'B/s',
                    'formatted_value' => isset($results['network_in']) ? $this->_formatBytes($results['network_in']) . '/s' : '未知',
                    'status' => 'normal'
                ],
                'network_out' => [
                    'label' => '网络出流量',
                    'value' => $results['network_out'],
                    'unit' => 'B/s',
                    'formatted_value' => isset($results['network_out']) ? $this->_formatBytes($results['network_out']) . '/s' : '未知',
                    'status' => 'normal'
                ],
                'uptime' => [
                    'label' => '运行时间',
                    'value' => $results['uptime'],
                    'unit' => '秒',
                    'formatted_value' => $this->_formatUptime($results['uptime']),
                    'status' => 'normal'
                ]
            ];
            
            // 添加该节点的数据
            $nodes_data[$ip] = [
                'ip' => $ip,
                'instance' => $instance_pattern,
                'online' => !empty($results['uptime']),
                'metrics' => $formatted_results,
                'timestamp' => time()
            ];
            
            log_message('debug', '节点 ' . $ip . ' 的指标数据处理完成');
        }
        
        $response_data = [
            'code' => 200,
            'message' => '获取节点指标数据成功',
            'data' => $nodes_data,
            'node_count' => count($ip_list),
            'timestamp' => time()
        ];
        
        log_message('debug', 'getNodeMetrics 返回数据节点数：' . count($nodes_data) . '，总数据大小：' . strlen(json_encode($response_data)));
        echo json_encode($response_data);
    }
    
    /**
     * 查询节点时间序列指标数据
     * 获取一段时间内的指标变化趋势
     * 支持多个 IP 地址，以逗号分隔
     */
    public function getMetricRange()
    {
        // 获取参数
        $ip_input = $this->input->get('ip');
        $metric = $this->input->get('metric') ?: 'cpu_usage';
        $start = $this->input->get('start') ?: (time() - 3600); // 默认1小时前
        $end = $this->input->get('end') ?: time(); // 默认当前时间
        $step = $this->input->get('step') ?: '60'; // 默认60秒
        
        // 记录请求信息
        log_message('debug', 'getMetricRange 方法被调用，请求参数：ip=' . $ip_input . '，metric=' . $metric . 
                   '，start=' . $start . '，end=' . $end . '，step=' . $step . '，IP：' . $this->input->ip_address());
        log_message('debug', 'Prometheus 服务器地址：' . $this->prometheus_url . '，节点导出器端口：' . $this->node_exporter_port);
        
        if (empty($ip_input)) {
            echo json_encode([
                'code' => 400,
                'message' => '参数错误，请提供节点IP',
                'data' => null
            ]);
            return;
        }
        
        // 解析多个 IP，以逗号分隔
        $ip_list = array_map('trim', explode(',', $ip_input));
        
        // 设置 Prometheus 服务器地址
        $prometheus_url = $this->prometheus_url;
        
        // 存储所有节点的数据
        $all_data = [];
        $nodes_info = [];
        
        // 遍历每个 IP 获取数据
        foreach ($ip_list as $ip) {
            // 根据实际 Prometheus 实例格式修改查询表达式
            $instance_pattern = $ip . ':' . $this->node_exporter_port;
            $nodes_info[] = $instance_pattern;
            
            // 定义 node.json 中的指标对应的 PromQL 表达式
            $expressions = [
                // 与 getPrometheusMetrics 相同的表达式集合
                'cpu_usage' => '100 - (avg(irate(node_cpu_seconds_total{instance="' . $instance_pattern . '",mode="idle"}[5m])) * 100)',
                'memory_usage' => '(1 - (node_memory_MemAvailable_bytes{instance="' . $instance_pattern . '"} / node_memory_MemTotal_bytes{instance="' . $instance_pattern . '"})) * 100',
                'disk_usage' => '1-(node_filesystem_free_bytes{instance="' . $instance_pattern . '",fstype=~"ext4|xfs"} / node_filesystem_size_bytes{instance="' . $instance_pattern . '",fstype=~"ext4|xfs"})',
                'load1' => 'node_load1{instance="' . $instance_pattern . '"}',
                'network_in' => 'sum(irate(node_network_receive_bytes_total{instance="' . $instance_pattern . '",device!~"tap.*|veth.*|br.*|docker.*|virbr*|lo*"}[5m]))',
                'network_out' => 'sum(irate(node_network_transmit_bytes_total{instance="' . $instance_pattern . '",device!~"tap.*|veth.*|br.*|docker.*|virbr*|lo*"}[5m]))'
            ];
            
            if (!isset($expressions[$metric])) {
                echo json_encode([
                    'code' => 400,
                    'message' => '不支持的指标：' . $metric,
                    'data' => null
                ]);
                return;
            }
            
            // 构建查询参数
            $query_params = [
                'query' => $expressions[$metric],
                'start' => $start,
                'end' => $end,
                'step' => $step
            ];
            
            // 构建 URL
            $url = $prometheus_url . '/api/v1/query_range?' . http_build_query($query_params);
            
            // 发送请求
            $headers = [
                'Accept: application/json'
            ];
            $response = $this->_curlRequest($url, 'GET', [], $headers);
            $result = json_decode($response, true);
            
            // 处理返回结果
            $node_data = [];
            if (!empty($result) && isset($result['status']) && $result['status'] === 'success' && !empty($result['data']['result'])) {
                foreach ($result['data']['result'] as $series) {
                    $series_data = [
                        'node_ip' => $ip,
                        'instance' => $instance_pattern,
                        'metric' => isset($series['metric']) ? $series['metric'] : [],
                        'values' => []
                    ];
                    
                    if (isset($series['values']) && is_array($series['values'])) {
                        foreach ($series['values'] as $point) {
                            if (is_array($point) && count($point) > 1) {
                                $timestamp = $point[0];
                                $value = $point[1];
                                
                                // 如果是数字，尝试转换为浮点数
                                if (is_numeric($value)) {
                                    $value = (float)$value;
                                    // 保留两位小数
                                    $value = round($value, 2);
                                }
                                
                                $series_data['values'][] = [
                                    'timestamp' => $timestamp,
                                    'value' => $value,
                                    'formatted_value' => $this->_formatMetricValue($metric, $value)
                                ];
                            }
                        }
                    }
                    
                    $node_data[] = $series_data;
                }
            }
            
            // 如果没有数据，也添加一个节点条目
            if (empty($node_data)) {
                $node_data[] = [
                    'node_ip' => $ip,
                    'instance' => $instance_pattern,
                    'metric' => [],
                    'values' => [],
                    'error' => '节点未返回数据'
                ];
            }
            
            // 添加到所有数据中
            $all_data = array_merge($all_data, $node_data);
        }
        
        // 格式化指标标签
        $metric_label = $this->_getMetricLabel($metric);
        
        echo json_encode([
            'code' => 200,
            'message' => '获取指标时间序列数据成功',
            'metric_name' => $metric,
            'metric_label' => $metric_label,
            'query_template' => $expressions[$metric],
            'start_time' => $start,
            'end_time' => $end,
            'step' => $step,
            'data' => $all_data,
            'nodes' => $nodes_info,
            'node_count' => count($ip_list)
        ]);
    }
    
    /**
     * 获取节点的额外指标数据
     * 获取额外的监控指标，如TCP连接数、数据库状态等
     * 支持多个 IP 地址，以逗号分隔
     */
    public function getAdditionalMetrics()
    {
        // 获取参数
        $ip_input = $this->input->get('ip');
        
        // 记录请求信息
        log_message('debug', 'getAdditionalMetrics 方法被调用，请求参数：ip=' . $ip_input . '，IP：' . $this->input->ip_address());
        log_message('debug', 'Prometheus 服务器地址：' . $this->prometheus_url . '，节点导出器端口：' . $this->node_exporter_port);
        
        if (empty($ip_input)) {
            log_message('error', 'getAdditionalMetrics 缺少参数：ip');
            echo json_encode([
                'code' => 400,
                'message' => '参数错误，请提供节点IP',
                'data' => null
            ]);
            return;
        }
        
        // 解析多个 IP，以逗号分隔
        $ip_list = array_map('trim', explode(',', $ip_input));
        log_message('debug', 'getAdditionalMetrics 解析的IP列表：' . json_encode($ip_list));
        
        // 设置 Prometheus 服务器地址
        $prometheus_url = $this->prometheus_url;
        log_message('debug', 'Prometheus 服务器地址：' . $prometheus_url);
        
        // 存储所有节点的数据
        $nodes_data = [];
        
        // 遍历每个 IP 获取数据
        foreach ($ip_list as $ip) {
            // 根据实际 Prometheus 实例格式修改查询表达式
            $instance_pattern = $ip . ':' . $this->node_exporter_port;
            log_message('debug', '处理节点：' . $ip . '，实例模式：' . $instance_pattern);
            
            // 定义要获取的额外指标
            $additional_metrics = [
                'tcp_connections',
                'tcp_timewaits',
                'sockets_used',
                'load5',
                'load15',
                'file_descriptors',
                'context_switches'
            ];
            
            $results = [];
            
            // 定义指标对应的 PromQL 表达式
            $expressions = [
                'tcp_connections' => 'node_netstat_Tcp_CurrEstab{instance="' . $instance_pattern . '"}',
                'tcp_timewaits' => 'node_sockstat_TCP_tw{instance="' . $instance_pattern . '"}',
                'sockets_used' => 'node_sockstat_sockets_used{instance="' . $instance_pattern . '"}',
                'load5' => 'node_load5{instance="' . $instance_pattern . '"}',
                'load15' => 'node_load15{instance="' . $instance_pattern . '"}',
                'file_descriptors' => 'node_filefd_allocated{instance="' . $instance_pattern . '"}',
                'context_switches' => 'irate(node_context_switches_total{instance="' . $instance_pattern . '"}[5m])'
            ];
            
            // 发送请求获取每个指标
            foreach ($additional_metrics as $metric) {
                if (!isset($expressions[$metric])) {
                    log_message('warning', '未知的额外指标：' . $metric);
                    continue;
                }
                
                $query_params = ['query' => $expressions[$metric]];
                $url = $prometheus_url . '/api/v1/query?' . http_build_query($query_params);
                
                log_message('debug', '请求 Prometheus 额外指标：' . $metric . '，URL：' . $url);
                
                $headers = [
                    'Accept: application/json'
                ];
                $response = $this->_curlRequest($url, 'GET', [], $headers);
                log_message('debug', 'Prometheus 额外指标响应数据长度：' . strlen($response));
                
                $result = json_decode($response, true);
                
                if (json_last_error() !== JSON_ERROR_NONE) {
                    log_message('error', '额外指标JSON解析错误：' . json_last_error_msg() . '，响应数据：' . substr($response, 0, 500));
                }
                
                $value = null;
                if (!empty($result) && isset($result['status']) && $result['status'] === 'success' && !empty($result['data']['result'])) {
                    if (isset($result['data']['result'][0]['value']) && is_array($result['data']['result'][0]['value']) && count($result['data']['result'][0]['value']) > 1) {
                        $value = $result['data']['result'][0]['value'][1];
                        // 如果是数字，尝试转换为浮点数
                        if (is_numeric($value)) {
                            $value = (float)$value;
                            // 保留两位小数
                            $value = round($value, 2);
                        }
                        log_message('debug', '额外指标 ' . $metric . ' 的值：' . $value);
                    } else {
                        log_message('debug', '额外指标 ' . $metric . ' 无有效值，result结构：' . json_encode($result['data']['result']));
                    }
                } else {
                    log_message('warning', '额外指标 ' . $metric . ' 请求失败或无数据，状态：' . ($result['status'] ?? 'unknown'));
                }
                
                $results[$metric] = $value;
            }
            
            // 尝试获取数据库相关指标（如果安装了数据库导出器）
            $db_metrics = [
                'db_connections' => 'count(mysql_global_status_threads_connected{instance=~"' . $ip . ':9104"})',
                'db_queries' => 'irate(mysql_global_status_questions{instance=~"' . $ip . ':9104"}[5m])',
                'db_slow_queries' => 'mysql_global_status_slow_queries{instance=~"' . $ip . ':9104"}'
            ];
            
            foreach ($db_metrics as $metric => $expr) {
                $query_params = ['query' => $expr];
                $url = $prometheus_url . '/api/v1/query?' . http_build_query($query_params);
                
                log_message('debug', '请求数据库指标：' . $metric . '，URL：' . $url);
                
                $headers = [
                    'Accept: application/json'
                ];
                $response = $this->_curlRequest($url, 'GET', [], $headers);
                
                $result = json_decode($response, true);
                
                $value = null;
                if (!empty($result) && isset($result['status']) && $result['status'] === 'success' && !empty($result['data']['result'])) {
                    if (isset($result['data']['result'][0]['value']) && is_array($result['data']['result'][0]['value']) && count($result['data']['result'][0]['value']) > 1) {
                        $value = $result['data']['result'][0]['value'][1];
                        if (is_numeric($value)) {
                            $value = (float)$value;
                            $value = round($value, 2);
                        }
                        log_message('debug', '数据库指标 ' . $metric . ' 的值：' . $value);
                    }
                }
                
                // 只添加成功获取的数据库指标
                if ($value !== null) {
                    $results[$metric] = $value;
                }
            }
            
            // 格式化结果
            $formatted_metrics = [];
            
            // TCP 连接指标
            if (isset($results['tcp_connections'])) {
                $formatted_metrics['tcp_connections'] = [
                    'label' => 'TCP连接数',
                    'value' => $results['tcp_connections'],
                    'unit' => '',
                    'formatted_value' => isset($results['tcp_connections']) ? $results['tcp_connections'] : '未知',
                    'status' => 'normal'
                ];
            }
            
            if (isset($results['tcp_timewaits'])) {
                $formatted_metrics['tcp_timewaits'] = [
                    'label' => 'TCP TIME_WAIT数',
                    'value' => $results['tcp_timewaits'],
                    'unit' => '',
                    'formatted_value' => isset($results['tcp_timewaits']) ? $results['tcp_timewaits'] : '未知',
                    'status' => 'normal'
                ];
            }
            
            if (isset($results['sockets_used'])) {
                $formatted_metrics['sockets_used'] = [
                    'label' => '已用套接字数',
                    'value' => $results['sockets_used'],
                    'unit' => '',
                    'formatted_value' => isset($results['sockets_used']) ? $results['sockets_used'] : '未知',
                    'status' => 'normal'
                ];
            }
            
            // 系统负载指标
            if (isset($results['load5'])) {
                $formatted_metrics['load5'] = [
                    'label' => '5分钟负载',
                    'value' => $results['load5'],
                    'unit' => '',
                    'formatted_value' => isset($results['load5']) ? $results['load5'] : '未知',
                    'status' => $this->_getMetricStatus('load5', $results['load5'])
                ];
            }
            
            if (isset($results['load15'])) {
                $formatted_metrics['load15'] = [
                    'label' => '15分钟负载',
                    'value' => $results['load15'],
                    'unit' => '',
                    'formatted_value' => isset($results['load15']) ? $results['load15'] : '未知',
                    'status' => $this->_getMetricStatus('load15', $results['load15'])
                ];
            }
            
            if (isset($results['file_descriptors'])) {
                $formatted_metrics['file_descriptors'] = [
                    'label' => '文件描述符数',
                    'value' => $results['file_descriptors'],
                    'unit' => '',
                    'formatted_value' => isset($results['file_descriptors']) ? $results['file_descriptors'] : '未知',
                    'status' => 'normal'
                ];
            }
            
            if (isset($results['context_switches'])) {
                $formatted_metrics['context_switches'] = [
                    'label' => '上下文切换/秒',
                    'value' => $results['context_switches'],
                    'unit' => '/s',
                    'formatted_value' => isset($results['context_switches']) ? $results['context_switches'] . '/s' : '未知',
                    'status' => 'normal'
                ];
            }
            
            // 数据库指标
            if (isset($results['db_connections'])) {
                $formatted_metrics['db_connections'] = [
                    'label' => '数据库连接数',
                    'value' => $results['db_connections'],
                    'unit' => '',
                    'formatted_value' => isset($results['db_connections']) ? $results['db_connections'] : '未知',
                    'status' => 'normal'
                ];
            }
            
            if (isset($results['db_queries'])) {
                $formatted_metrics['db_queries'] = [
                    'label' => '数据库查询/秒',
                    'value' => $results['db_queries'],
                    'unit' => '/s',
                    'formatted_value' => isset($results['db_queries']) ? $results['db_queries'] . '/s' : '未知',
                    'status' => 'normal'
                ];
            }
            
            if (isset($results['db_slow_queries'])) {
                $formatted_metrics['db_slow_queries'] = [
                    'label' => '慢查询数',
                    'value' => $results['db_slow_queries'],
                    'unit' => '',
                    'formatted_value' => isset($results['db_slow_queries']) ? $results['db_slow_queries'] : '未知',
                    'status' => 'normal'
                ];
            }
            
            // 添加该节点的数据
            $nodes_data[$ip] = [
                'ip' => $ip,
                'instance' => $instance_pattern,
                'metrics' => $formatted_metrics,
                'timestamp' => time()
            ];
            
            log_message('debug', '节点 ' . $ip . ' 的额外指标数据处理完成');
        }
        
        $response_data = [
            'code' => 200,
            'message' => '获取节点额外指标数据成功',
            'data' => $nodes_data,
            'node_count' => count($ip_list),
            'timestamp' => time()
        ];
        
        log_message('debug', 'getAdditionalMetrics 返回数据节点数：' . count($nodes_data) . '，总数据大小：' . strlen(json_encode($response_data)));
        echo json_encode($response_data);
    }

    /**
     * 判断指标状态
     * 根据阈值判断指标状态
     */
    private function _getMetricStatus($metric, $value)
    {
        if ($value === null) {
            return 'unknown';
        }
        
        // 定义不同指标的告警阈值
        $thresholds = [
            'cpu_usage' => [
                'warning' => 80,
                'critical' => 90
            ],
            'memory_usage' => [
                'warning' => 80,
                'critical' => 90
            ],
            'disk_usage' => [
                'warning' => 80,
                'critical' => 90
            ],
            'load1' => [
                'warning' => 5,
                'critical' => 10
            ]
        ];
        
        if (!isset($thresholds[$metric])) {
            return 'normal';
        }
        
        if ($value >= $thresholds[$metric]['critical']) {
            return 'critical';
        } elseif ($value >= $thresholds[$metric]['warning']) {
            return 'warning';
        } else {
            return 'normal';
        }
    }

    /**
     * 同步系统监控告警
     * 从Prometheus获取指标数据，检查是否超过阈值，生成告警
     */
    public function syncSystemMonitorAlerts()
    {
        log_message('debug', 'syncSystemMonitorAlerts 方法被调用，IP：' . $this->input->ip_address());
        
        // 获取系统监控告警配置
        try {
            $categories = $this->AlarmCategory_model->getCategories();
            if (!is_array($categories)) {
                log_message('error', 'syncSystemMonitorAlerts: getCategories 返回了非数组结果');
                echo json_encode([
                    'code' => 500,
                    'message' => 'AlarmCategory_model->getCategories 返回非数组结果',
                    'data' => null
                ]);
                return;
            }
            
            $systemCategory = null;
            
            foreach ($categories as $category) {
                if ($category['id'] === 'system_monitor') {
                    $systemCategory = $category;
                    break;
                }
            }
            
            if (!$systemCategory) {
                echo json_encode([
                    'code' => 404,
                    'message' => '系统监控告警分类不存在',
                    'data' => null
                ]);
                return;
            }
            
            // 获取告警类型详情
            $alarmTypes = $this->AlarmCategory_model->getAlarmTypeDetails();
            if (!is_array($alarmTypes)) {
                log_message('error', 'syncSystemMonitorAlerts: getAlarmTypeDetails 返回了非数组结果');
                echo json_encode([
                    'code' => 500,
                    'message' => 'AlarmCategory_model->getAlarmTypeDetails 返回非数组结果',
                    'data' => null
                ]);
                return;
            }
        } catch (Exception $e) {
            log_message('error', 'syncSystemMonitorAlerts: 获取告警配置异常 - ' . $e->getMessage());
            echo json_encode([
                'code' => 500,
                'message' => '获取告警配置异常: ' . $e->getMessage(),
                'data' => null
            ]);
            return;
        }
        
        // 获取所有节点
        $nodes_sql = "SELECT hostaddr, prometheus_port FROM cluster_mgr_nodes ";
        $query_result = $this->db->query($nodes_sql);
        
        // 检查查询是否成功
        if ($query_result === FALSE) {
            log_message('error', 'syncSystemMonitorAlerts: 数据库查询失败 - ' . $this->db->error()['message']);
            echo json_encode([
                'code' => 500,
                'message' => '数据库查询错误: ' . $this->db->error()['message'],
                'data' => null
            ]);
            return;
        }
        
        $nodes = $query_result->result_array();
        
        if (empty($nodes)) {
            echo json_encode([
                'code' => 404,
                'message' => '未找到活跃节点',
                'data' => null
            ]);
            return;
        }
        
        // 用于记录已检测的告警
        $detected_alerts = [];
        
        // 遍历每个节点
        foreach ($nodes as $node) {
            $ip = $node['hostaddr'];
            $instance_pattern = $ip . ':' . $this->node_exporter_port;
            
            // 检查节点是否在线
            $up_query = 'up{instance="' . $instance_pattern . '"}';
            $up_response = $this->queryPrometheus($up_query);
            
            if (empty($up_response) || !isset($up_response['data']['result']) || empty($up_response['data']['result'])) {
                // 节点离线，生成machine_offline告警
                $this->createNodeOfflineAlert($ip, $alarmTypes);
                continue;
            }
            
            // 节点在线，检查系统指标
            $this->checkNodeSystemMetrics($ip, $instance_pattern, $systemCategory, $alarmTypes);
        }
        
        // 处理集群级别的告警
        $this->checkClusterMetrics($systemCategory, $alarmTypes);
        
        echo json_encode([
            'code' => 200,
            'message' => '系统监控告警同步成功',
            'data' => [
                'detected_alerts' => $detected_alerts
            ]
        ]);
    }
    
    /**
     * 创建节点离线告警
     */
    private function createNodeOfflineAlert($ip, $alarmTypes)
    {
        // 检查参数
        if (empty($ip) || !is_array($alarmTypes)) {
            log_message('error', 'createNodeOfflineAlert: 无效的参数');
            return false;
        }
        
        $alarm_type = 'machine_offline';
        
        if (!isset($alarmTypes[$alarm_type])) {
            log_message('error', 'createNodeOfflineAlert: 未定义的告警类型 - ' . $alarm_type);
            return false;
        }
        
        $alarm_data = [
            'node_ip' => $ip,
            'error_message' => '节点无法访问',
            'error_time' => date('Y-m-d H:i:s')
        ];
        
        $msg_data = json_encode($alarm_data);
        
        try {
            // 检查是否已存在未处理的告警
            $select_sql = "SELECT job_info FROM cluster_alarm_info WHERE alarm_type=? AND job_info LIKE ? AND status='unhandled'";
            
            // 使用预处理语句
            if (!isset($this->Cluster_model)) {
                log_message('error', 'createNodeOfflineAlert: Cluster_model 未加载');
                return false;
            }
            
            $res_select = $this->Cluster_model->getList($select_sql, [$alarm_type, '%' . $ip . '%']);
            
            if ($res_select === FALSE) {
                log_message('error', 'createNodeOfflineAlert: 查询告警记录失败');
                return false;
            }
            
            if (empty($res_select)) {
                // 记录告警信息
                $insert_sql = "INSERT INTO cluster_alarm_info(alarm_type, alarm_level, job_info, occur_timestamp) VALUES (?, ?, ?, NOW())";
                $result = $this->Cluster_model->updateList($insert_sql, [
                    $alarm_type, 
                    $alarmTypes[$alarm_type]['level'], 
                    $msg_data
                ]);
                
                if ($result === FALSE) {
                    log_message('error', 'createNodeOfflineAlert: 插入告警记录失败');
                    return false;
                }
                
                // 记录告警日志
                log_message('warning', '节点离线告警: IP=' . $ip);
                return true;
            }
        } catch (Exception $e) {
            log_message('error', 'createNodeOfflineAlert 异常: ' . $e->getMessage());
            return false;
        }
        
        return false;
    }
    
    /**
     * 检查节点系统指标
     */
    private function checkNodeSystemMetrics($ip, $instance_pattern, $systemCategory, $alarmTypes)
    {
        // 检查CPU使用率
        if (in_array('cluster_cpu_usage', $systemCategory['types'])) {
            $this->checkCpuUsage($ip, $instance_pattern, $alarmTypes);
        }
        
        // 检查内存使用率
        if (in_array('cluster_memory_usage', $systemCategory['types'])) {
            $this->checkMemoryUsage($ip, $instance_pattern, $alarmTypes);
        }
        
        // 检查存储空间使用率
        if (in_array('cluster_storage_usage', $systemCategory['types'])) {
            $this->checkStorageUsage($ip, $instance_pattern, $alarmTypes);
        }
        
        // 检查IO性能
        if (in_array('cluster_io_performance', $systemCategory['types'])) {
            $this->checkIOPerformance($ip, $instance_pattern, $alarmTypes);
        }
        
        // 检查系统负载
        if (in_array('cluster_load_average', $systemCategory['types'])) {
            $this->checkLoadAverage($ip, $instance_pattern, $alarmTypes);
        }
    }
    
    /**
     * 检查CPU使用率
     */
    private function checkCpuUsage($ip, $instance_pattern, $alarmTypes)
    {
        $alarm_type = 'cluster_cpu_usage';
        $threshold = 80; // 默认阈值80%
        
        // CPU使用率查询
        $query = '100 - (avg(irate(node_cpu_seconds_total{instance="' . $instance_pattern . '",mode="idle"}[5m])) * 100)';
        $response = $this->queryPrometheus($query);
        
        if (!empty($response) && isset($response['data']['result']) && !empty($response['data']['result'])) {
            $result = $response['data']['result'][0];
            $value = floatval($result['value'][1]);
            
            if ($value > $threshold) {
                // CPU使用率超过阈值，生成告警
                $alarm_data = [
                    'node_ip' => $ip,
                    'metric' => 'CPU使用率',
                    'value' => round($value, 2) . '%',
                    'threshold' => $threshold . '%',
                    'timestamp' => date('Y-m-d H:i:s')
                ];
                
                $this->createMetricAlert($alarm_type, $alarmTypes[$alarm_type]['level'], $alarm_data);
            }
        }
    }
    
    /**
     * 检查内存使用率
     */
    private function checkMemoryUsage($ip, $instance_pattern, $alarmTypes)
    {
        $alarm_type = 'cluster_memory_usage';
        $threshold = 85; // 默认阈值85%
        
        // 内存使用率查询
        $query = '100 * (1 - ((node_memory_MemFree_bytes{instance="' . $instance_pattern . '"} + node_memory_Cached_bytes{instance="' . $instance_pattern . '"} + node_memory_Buffers_bytes{instance="' . $instance_pattern . '"}) / node_memory_MemTotal_bytes{instance="' . $instance_pattern . '"}))';
        $response = $this->queryPrometheus($query);
        
        if (!empty($response) && isset($response['data']['result']) && !empty($response['data']['result'])) {
            $result = $response['data']['result'][0];
            $value = floatval($result['value'][1]);
            
            if ($value > $threshold) {
                // 内存使用率超过阈值，生成告警
                $alarm_data = [
                    'node_ip' => $ip,
                    'metric' => '内存使用率',
                    'value' => round($value, 2) . '%',
                    'threshold' => $threshold . '%',
                    'timestamp' => date('Y-m-d H:i:s')
                ];
                
                $this->createMetricAlert($alarm_type, $alarmTypes[$alarm_type]['level'], $alarm_data);
            }
        }
    }
    
    /**
     * 检查存储空间使用率
     */
    private function checkStorageUsage($ip, $instance_pattern, $alarmTypes)
    {
        $alarm_type = 'cluster_storage_usage';
        $threshold = 85; // 默认阈值85%
        
        // 存储空间使用率查询 - 检查所有挂载点
        $query = '100 * (1 - node_filesystem_avail_bytes{instance="' . $instance_pattern . '",fstype!="rootfs"} / node_filesystem_size_bytes{instance="' . $instance_pattern . '",fstype!="rootfs"})';
        $response = $this->queryPrometheus($query);
        
        if (!empty($response) && isset($response['data']['result']) && !empty($response['data']['result'])) {
            foreach ($response['data']['result'] as $result) {
                $value = floatval($result['value'][1]);
                $mountpoint = $result['metric']['mountpoint'] ?? 'unknown';
                
                if ($value > $threshold) {
                    // 存储空间使用率超过阈值，生成告警
                    $alarm_data = [
                        'node_ip' => $ip,
                        'metric' => '存储空间使用率',
                        'mountpoint' => $mountpoint,
                        'value' => round($value, 2) . '%',
                        'threshold' => $threshold . '%',
                        'timestamp' => date('Y-m-d H:i:s')
                    ];
                    
                    $this->createMetricAlert($alarm_type, $alarmTypes[$alarm_type]['level'], $alarm_data);
                }
            }
        }
    }
    
    /**
     * 检查IO性能
     */
    private function checkIOPerformance($ip, $instance_pattern, $alarmTypes)
    {
        $alarm_type = 'cluster_io_performance';
        $latency_threshold = 100; // 默认阈值100ms
        
        // IO等待时间查询
        $query = 'irate(node_disk_io_time_seconds_total{instance="' . $instance_pattern . '"}[5m]) * 1000'; // 转换为毫秒
        $response = $this->queryPrometheus($query);
        
        if (!empty($response) && isset($response['data']['result']) && !empty($response['data']['result'])) {
            foreach ($response['data']['result'] as $result) {
                $value = floatval($result['value'][1]);
                $device = $result['metric']['device'] ?? 'unknown';
                
                if ($value > $latency_threshold) {
                    // IO等待时间超过阈值，生成告警
                    $alarm_data = [
                        'node_ip' => $ip,
                        'metric' => 'IO等待时间',
                        'device' => $device,
                        'value' => round($value, 2) . 'ms',
                        'threshold' => $latency_threshold . 'ms',
                        'timestamp' => date('Y-m-d H:i:s')
                    ];
                    
                    $this->createMetricAlert($alarm_type, $alarmTypes[$alarm_type]['level'], $alarm_data);
                }
            }
        }
    }
    
    /**
     * 检查系统负载
     */
    private function checkLoadAverage($ip, $instance_pattern, $alarmTypes)
    {
        $alarm_type = 'cluster_load_average';
        
        // 获取CPU核心数
        $cpu_count_query = 'count(node_cpu_seconds_total{instance="' . $instance_pattern . '", mode="system"})';
        $cpu_count_response = $this->queryPrometheus($cpu_count_query);
        
        if (!empty($cpu_count_response) && isset($cpu_count_response['data']['result']) && !empty($cpu_count_response['data']['result'])) {
            $cpu_count = intval($cpu_count_response['data']['result'][0]['value'][1]);
            $threshold = $cpu_count * 2; // 阈值为CPU核心数的2倍
            
            // 系统负载查询
            $query = 'node_load1{instance="' . $instance_pattern . '"}';
            $response = $this->queryPrometheus($query);
            
            if (!empty($response) && isset($response['data']['result']) && !empty($response['data']['result'])) {
                $value = floatval($response['data']['result'][0]['value'][1]);
                
                if ($value > $threshold) {
                    // 系统负载超过阈值，生成告警
                    $alarm_data = [
                        'node_ip' => $ip,
                        'metric' => '系统负载',
                        'cpu_count' => $cpu_count,
                        'value' => round($value, 2),
                        'threshold' => $threshold,
                        'timestamp' => date('Y-m-d H:i:s')
                    ];
                    
                    $this->createMetricAlert($alarm_type, $alarmTypes[$alarm_type]['level'], $alarm_data);
                }
            }
        }
    }
    
    /**
     * 检查集群级别指标
     */
    private function checkClusterMetrics($systemCategory, $alarmTypes)
    {
        // 在这里可以添加集群级别的指标检查
        // 例如：检查计算节点和存储节点状态
        if (in_array('comp_node_exception', $systemCategory['types'])) {
            $this->checkCompNodeStatus($alarmTypes);
        }
        
        if (in_array('storage_node_exception', $systemCategory['types'])) {
            $this->checkStorageNodeStatus($alarmTypes);
        }
    }
    
    /**
     * 检查计算节点状态
     */
    private function checkCompNodeStatus($alarmTypes)
    {
        $alarm_type = 'comp_node_exception';
        
        // 从数据库查询计算节点状态
        $sql = "SELECT hostaddr, port, status FROM comp_nodes WHERE status = 'inactive'";
        $query_result = $this->db->query($sql);
        
        // 检查查询是否成功
        if ($query_result === FALSE) {
            log_message('error', 'checkCompNodeStatus: 数据库查询失败 - ' . $this->db->error()['message']);
            return;
        }
        
        $inactive_nodes = $query_result->result_array();
        
        if (!empty($inactive_nodes)) {
            foreach ($inactive_nodes as $node) {
                $alarm_data = [
                    'node_ip' => $node['hostaddr'],
                    'port' => $node['port'],
                    'status' => $node['status'],
                    'error_message' => '计算节点异常',
                    'timestamp' => date('Y-m-d H:i:s')
                ];
                
                $this->createMetricAlert($alarm_type, $alarmTypes[$alarm_type]['level'], $alarm_data);
            }
        }
    }
    
    /**
     * 检查存储节点状态
     */
    private function checkStorageNodeStatus($alarmTypes)
    {
        $alarm_type = 'storage_node_exception';
        
        // 从数据库查询存储节点状态
        $sql = "SELECT hostaddr, port, status FROM shard_nodes WHERE status = 'inactive'";
        $query_result = $this->db->query($sql);
        
        // 检查查询是否成功
        if ($query_result === FALSE) {
            log_message('error', 'checkStorageNodeStatus: 数据库查询失败 - ' . $this->db->error()['message']);
            return;
        }
        
        $inactive_nodes = $query_result->result_array();
        
        if (!empty($inactive_nodes)) {
            foreach ($inactive_nodes as $node) {
                $alarm_data = [
                    'node_ip' => $node['hostaddr'],
                    'port' => $node['port'],
                    'status' => $node['status'],
                    'error_message' => '存储节点异常',
                    'timestamp' => date('Y-m-d H:i:s')
                ];
                
                $this->createMetricAlert($alarm_type, $alarmTypes[$alarm_type]['level'], $alarm_data);
            }
        }
    }
    
    /**
     * 创建指标告警
     */
    private function createMetricAlert($alarm_type, $alarm_level, $alarm_data)
    {
        if (empty($alarm_type) || empty($alarm_level) || empty($alarm_data) || !is_array($alarm_data)) {
            log_message('error', 'createMetricAlert: 参数无效');
            return false;
        }
        
        try {
            $msg_data = json_encode($alarm_data);
            
            if ($msg_data === false) {
                log_message('error', 'createMetricAlert: 告警数据JSON编码失败: ' . json_last_error_msg());
                return false;
            }
            
            // 检查是否已存在未处理的告警
            $select_sql = "SELECT job_info FROM cluster_alarm_info WHERE alarm_type=? AND job_info LIKE ? AND status='unhandled'";
            
            if (!isset($this->Cluster_model)) {
                log_message('error', 'createMetricAlert: Cluster_model 未加载');
                return false;
            }
            
            $res_select = $this->Cluster_model->getList($select_sql, [$alarm_type, '%' . $alarm_data['node_ip'] . '%']);
            
            if ($res_select === FALSE) {
                log_message('error', 'createMetricAlert: 查询告警记录失败');
                return false;
            }
            
            if (empty($res_select)) {
                // 记录告警信息
                $insert_sql = "INSERT INTO cluster_alarm_info(alarm_type, alarm_level, job_info, occur_timestamp) VALUES (?, ?, ?, NOW())";
                $result = $this->Cluster_model->updateList($insert_sql, [
                    $alarm_type, 
                    $alarm_level, 
                    $msg_data
                ]);
                
                if ($result === FALSE) {
                    log_message('error', 'createMetricAlert: 插入告警记录失败');
                    return false;
                }
                
                // 记录告警日志
                log_message('warning', $alarm_type . ' 告警: IP=' . $alarm_data['node_ip'] . ', 值=' . ($alarm_data['value'] ?? 'unknown'));
                return true;
            }
            
            return false;
        } catch (Exception $e) {
            log_message('error', 'createMetricAlert 异常: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * 查询Prometheus API
     */
    private function queryPrometheus($query)
    {
        if (empty($query)) {
            log_message('error', 'queryPrometheus: 查询语句为空');
            return null;
        }
        
        $url = $this->prometheus_url . '/api/v1/query';
        $url .= '?query=' . urlencode($query);
        
        log_message('debug', 'Prometheus查询: ' . $url);
        
        try {
            $response = $this->_curlRequest($url);
            
            if (empty($response)) {
                log_message('error', 'queryPrometheus: Prometheus响应为空');
                return null;
            }
            
            $result = json_decode($response, true);
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                log_message('error', 'Prometheus响应JSON解析错误: ' . json_last_error_msg() . ', 响应内容: ' . substr($response, 0, 500));
                return null;
            }
            
            if (!isset($result['status']) || $result['status'] !== 'success') {
                $error_type = isset($result['errorType']) ? $result['errorType'] : 'unknown';
                $error_msg = isset($result['error']) ? $result['error'] : 'Unknown error';
                log_message('error', 'Prometheus查询失败: ' . $error_type . ' - ' . $error_msg);
                return null;
            }
            
            return $result;
        } catch (Exception $e) {
            log_message('error', 'queryPrometheus异常: ' . $e->getMessage());
            return null;
        }
    }
} 