<?php

use GuzzleHttp\Psr7\Uri;

defined('BASEPATH') or exit('No direct script access allowed');

class AlarmService {
    protected $CI;

    public function __construct() {
        $this->CI =& get_instance();
        $this->CI->load->model('Cluster_model');
        $this->CI->load->model('Db_model');
        $this->CI->load->library('AliyunSms');
    }

    /**
     * 测试代理连接
     */
    public function testProxyConnection($serverAddress, $templateId = '') {
        if (empty($serverAddress)) {
            throw new Exception('服务器地址不能为空');
        }

        $ch = curl_init();
        $postData = [
            'task_id' => trim($templateId),
            'title' => 'Test Connection Title', // 测试连接标题
            'text' => 'This is a test message for proxy connection.' // 测试连接内容
        ];
        $jsonData = json_encode($postData);

        $options = [
            CURLOPT_URL => $serverAddress,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_CUSTOMREQUEST => 'POST',
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json',
                'Content-Length: ' . strlen($jsonData)
            ],
            CURLOPT_POSTFIELDS => $jsonData,
            CURLOPT_TIMEOUT => 50,
            CURLOPT_CONNECTTIMEOUT => 3,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false
        ];

        curl_setopt_array($ch, $options);
        $result = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);

        if ($httpCode >= 200 && $httpCode < 300) {
            return ['success' => true, 'message' => '代理服务器连接成功'];
        } else {
            throw new Exception('代理服务器连接失败: ' . ($error ?: "HTTP状态码 {$httpCode}"));
        }
    }

    /**
     * 发送代理通知
     * @param string $serverAddress 服务器地址
     * @param string $title 消息标题
     * @param string $text 消息内容
     * @param string $templateId 模板ID
     * @return bool 发送是否成功
     */
    public function sendProxyNotification($serverAddress, $title, $text, $templateId = '') {
        log_message('debug', '开始发送代理通知');
        log_message('debug', '服务器地址: ' . $serverAddress);
        log_message('debug', '告警标题: ' . $title);
        log_message('debug', '模板ID: ' . ($templateId ?: '未指定'));

        try {
            log_message('debug', '初始化cURL请求');
            $ch = curl_init();
            $postData = [
                'task_id' => trim($templateId),
                'title' => $title,
                'text' => $text
            ];
            log_message('debug', '请求数据: ' . json_encode($postData));
            
            curl_setopt_array($ch, [
                CURLOPT_URL => $serverAddress,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_CUSTOMREQUEST => 'POST',
                CURLOPT_HTTPHEADER => [
                    'Content-Type: application/json',
                ],
                CURLOPT_POSTFIELDS => json_encode($postData),
                CURLOPT_FOLLOWLOCATION => true,
                CURLOPT_TIMEOUT => 10
            ]);
            
            log_message('debug', '执行cURL请求');
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $error = curl_error($ch);
            
            if (!empty($error)) {
                log_message('error', 'cURL错误: ' . $error);
            }
            
            log_message('debug', '收到HTTP状态码: ' . $httpCode);
            log_message('debug', '响应内容: ' . $response);
            
            curl_close($ch);

            if ($httpCode >= 200 && $httpCode < 300) {
                log_message('info', 'Proxy notification sent successfully: ' . $response);
                return true;
            } else {
                log_message('error', 'Proxy notification failed with HTTP code ' . $httpCode . ': ' . $response);
                return false;
            }
        } catch (Exception $e) {
            log_message('error', 'Proxy notification failed: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * 测试短信发送
     */
    public function testSmsAlarm($phone, $config = null) {
        if (empty($phone)) {
            throw new Exception('手机号不能为空');
        }

        if (!$config) {
            $config = $this->getMessageConfig('phone_message');
        }

        $result = $this->CI->aliyunsms->sendSms(
            $config['AccessKeyId'],
            $config['SecretKey'],
            $phone,
            $config['SigId'],
            $config['TemplateId'],
            ['content' => '这是一条测试告警短信']
        );

        if ($result['Code'] !== 'OK') {
            throw new Exception('发送失败：' . $result['Message']);
        }

        return ['success' => true, 'message' => '测试短信发送成功'];
    }

    /**
     * 发送短信通知
     */
    public function sendSmsNotification($config, $phoneNumber, $content) {
        try {
            $result = $this->CI->aliyunsms->sendSms(
                $config['AccessKeyId'],
                $config['SecretKey'],
                $phoneNumber,
                $config['SigId'],
                $config['TemplateId'],
                ['content' => $content]
            );

            return $result['Code'] === 'OK';
        } catch (Exception $e) {
            log_message('error', 'SMS notification failed: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * 获取邮件 HTML 模板
     */
    private function getEmailTemplate($title, $content, $buttonText = '', $buttonUrl = '') {
        return '
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
</head>
<body style="margin: 0; padding: 0; font-family: Arial, sans-serif; background-color: #f4f4f4;">
    <div style="max-width: 600px; margin: 0 auto; background-color: #ffffff; padding: 20px; border-radius: 10px; box-shadow: 0 2px 5px rgba(0,0,0,0.1); margin-top: 20px;">
        <div style="text-align: center; margin-bottom: 20px;">
            <img src="https://kunlunbase.com/static/img/logo.png" alt="Kunlun Monitor" style="max-width: 200px; height: auto;">
        </div>
        <div style="border-top: 2px solid #2d8cf0; margin: 20px 0;"></div>
        <h1 style="color: #2d8cf0; font-size: 24px; margin-bottom: 20px; text-align: center;">' . $title . '</h1>
        <div style="color: #333333; font-size: 16px; line-height: 1.6; margin-bottom: 20px; text-align: left;">
            ' . $content . '
        </div>
        ' . ($buttonText && $buttonUrl ? '
        <div style="text-align: center; margin-top: 30px;">
            <a href="' . $buttonUrl . '" style="display: inline-block; padding: 12px 24px; background-color: #2d8cf0; color: #ffffff; text-decoration: none; border-radius: 5px; font-weight: bold;">' . $buttonText . '</a>
        </div>
        ' : '') . '
        <div style="border-top: 1px solid #eeeeee; margin-top: 30px; padding-top: 20px; font-size: 12px; color: #999999; text-align: center;">
            <p>此邮件由 Kunlun Monitor 系统自动发送，请勿直接回复</p>
            <p>© ' . date('Y') . ' Kunlun Monitor. All rights reserved.</p>
        </div>
    </div>
</body>
</html>';
    }

    /**
     * 发送邮件的公共方法
     * @param array $config 邮件配置
     * @param string $to 收件人邮箱
     * @param string $subject 邮件主题
     * @param string $content 邮件内容
     * @param bool $isDebug 是否开启调试模式
     * @return array|bool 测试模式返回数组，普通模式返回布尔值
     * @throws Exception
     */
    private function sendEmail($config, $to, $subject, $content, $isDebug = false) {
        // 确保配置信息完整
        if (empty($config['stmp']) || empty($config['user_name']) || empty($config['password'])) {
            throw new Exception('邮件配置信息不完整');
        }

        // 检查文件是否存在
        $phpmailer_path = APPPATH . 'third_party/PHPMailer/class.phpmailer.php';
        $smtp_path = APPPATH . 'third_party/PHPMailer/class.smtp.php';

        if (!file_exists($phpmailer_path) || !file_exists($smtp_path)) {
            throw new Exception('PHPMailer 文件不存在');
        }

        try {
            // 加载 PHPMailer 类文件
            require_once $phpmailer_path;
            require_once $smtp_path;

            $mail = new PHPMailer();

            // 服务器配置
            $mail->SMTPDebug = $isDebug ? 2 : 0;
            $mail->Debugoutput = function($str, $level) {
                log_message('debug', "PHPMailer [$level]: $str");
            };
            
            $mail->isSMTP();
            $mail->Host = $config['stmp'];
            $mail->SMTPAuth = true;
            $mail->Username = $config['user_name'];
            $mail->Password = $config['password'];
            
            // 设置加密方式和端口
            if (isset($config['encryption']) && strtolower($config['encryption']) === 'tls') {
                $mail->SMTPSecure = 'tls';
                $mail->Port = 587;
            } else {
                $mail->SMTPSecure = 'ssl';
                $mail->Port = 465;
            }

            if ($isDebug) {
                log_message('debug', '邮件服务器配置: ' . json_encode([
                    'host' => $config['stmp'],
                    'username' => $config['user_name'],
                    'encryption' => $mail->SMTPSecure,
                    'port' => $mail->Port
                ]));
            }

            $mail->CharSet = 'UTF-8';

            // 收发件人
            $mail->setFrom($config['user_name'], 'Kunlun Monitor');
            $mail->addAddress($to);

            // 邮件内容
            $mail->isHTML(true);
            $mail->Subject = $subject;
            $mail->Body = $this->getEmailTemplate($subject, $content);
            $mail->AltBody = strip_tags($content);

            // 发送邮件
            if (!$mail->send()) {
                throw new Exception('邮件发送失败：' . $mail->ErrorInfo);
            }

            return $isDebug ? ['success' => true, 'message' => '邮件发送成功'] : true;
        } catch (Exception $e) {
            log_message('error', '邮件发送失败：' . $e->getMessage());
            if ($isDebug) {
                throw $e;
            }
            return false;
        }
    }

    /**
     * 测试邮件告警
     */
    public function testEmailAlarm($email, $config = null) {
        if (empty($email)) {
            throw new Exception('邮箱地址不能为空');
        }

        if (!$config) {
            $config = $this->getMessageConfig('email');
        }

        $subject = 'Kunlun Monitor 告警测试邮件';
        $content = '
            <p>您好！</p>
            <p>这是一封来自 Kunlun Monitor 系统的测试告警邮件，用于验证邮件推送配置是否正确。</p>
            <p>如果您收到这封邮件，说明邮件推送功能已配置成功！</p>
            <p>测试信息：</p>
            <ul>
                <li>发送时间：' . date('Y-m-d H:i:s') . '</li>
                <li>接收邮箱：' . $email . '</li>
                <li>发送服务器：' . $config['stmp'] . '</li>
            </ul>';

        return $this->sendEmail($config, $email, $subject, $content, true);
    }

    /**
     * 发送邮件通知
     */
    public function sendEmailNotification($config, $email, $content, $subject) {
        return $this->sendEmail($config, $email, $subject, $content, false);
    }

    /**
     * 测试企业微信发送
     */
    public function testWechatAlarm($config = null) {
        if (!$config) {
            $config = $this->getMessageConfig('wechat');
        }

        // 获取访问令牌
        $access_token_url = sprintf(
            'https://qyapi.weixin.qq.com/cgi-bin/gettoken?corpid=%s&corpsecret=%s',
            $config['corpid'],
            $config['secret']
        );

        $response = file_get_contents($access_token_url);
        $result = json_decode($response, true);

        if (!isset($result['access_token'])) {
            throw new Exception('获取访问令牌失败：' . ($result['errmsg'] ?? '未知错误'));
        }

        // 发送测试消息
        $send_url = 'https://qyapi.weixin.qq.com/cgi-bin/message/send?access_token=' . $result['access_token'];
        
        $message = [
            'touser' => '@all',
            'msgtype' => 'text',
            'agentid' => $config['agentid'],
            'text' => [
                'content' => '这是一条测试告警消息'
            ]
        ];

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $send_url);
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($message));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);

        $response = curl_exec($ch);
        $result = json_decode($response, true);
        curl_close($ch);

        if ($result['errcode'] !== 0) {
            throw new Exception('发送失败：' . $result['errmsg']);
        }

        return ['success' => true, 'message' => '测试消息发送成功'];
    }

    /**
     * 发送企业微信通知
     */
    public function sendWechatNotification($config, $userId, $content) {
        try {
            // 获取访问令牌
            $access_token_url = sprintf(
                'https://qyapi.weixin.qq.com/cgi-bin/gettoken?corpid=%s&corpsecret=%s',
                $config['corpid'],
                $config['secret']
            );

            $response = file_get_contents($access_token_url);
            $result = json_decode($response, true);

            if (!isset($result['access_token'])) {
                return false;
            }

            // 发送消息
            $send_url = 'https://qyapi.weixin.qq.com/cgi-bin/message/send?access_token=' . $result['access_token'];
            
            $message = [
                'touser' => $userId,
                'msgtype' => 'text',
                'agentid' => $config['agentid'],
                'text' => [
                    'content' => $content
                ]
            ];

            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $send_url);
            curl_setopt($ch, CURLOPT_POST, 1);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($message));
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);

            $response = curl_exec($ch);
            $result = json_decode($response, true);
            curl_close($ch);

            return $result['errcode'] === 0;
        } catch (Exception $e) {
            log_message('error', 'WeChat notification failed: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * 获取消息配置
     */
    private function getMessageConfig($type) {
        $sql = "SELECT message FROM cluster_alarm_message_config WHERE type = ? LIMIT 1";
        $result = $this->CI->db->query($sql, [$type]);
        $config = $result->result_array();
        
        if (empty($config)) {
            throw new Exception($type . '配置不存在');
        }

        return json_decode($config[0]['message'], true);
    }
} 
