<?php
defined('BASEPATH') or exit('No direct script access allowed');

class AlarmCategory_model extends CI_Model
{
    protected $db;

    public function __construct()
    {
        parent::__construct();
        $this->db = $this->load->database('default', true);
    }

    /**
     * 获取告警分类列表
     */
    public function getCategories()
    {
        // 定义告警分类 - 按照PRD文档定义
        $categories = [
            [
                'id' => 'system_monitor',
                'label' => '系统监控告警',
                'description' => '监控系统资源和状态的告警',
                'types' => [
                    'machine_offline', 
                    'comp_node_exception', 
                    'storage_node_exception', 
                    'cluster_cpu_usage', 
                    'cluster_memory_usage', 
                    'cluster_storage_usage', 
                    'cluster_io_performance', 
                    'cluster_load_average'
                ],
                'subCategories' => [
                    [
                        'id' => 'machine',
                        'label' => '机器监控',
                        'types' => ['machine_offline', 'comp_node_exception', 'storage_node_exception']
                    ],
                    [
                        'id' => 'resource',
                        'label' => '资源监控',
                        'types' => ['cluster_cpu_usage', 'cluster_memory_usage', 'cluster_storage_usage']
                    ],
                    [
                        'id' => 'performance',
                        'label' => '性能监控',
                        'types' => ['cluster_io_performance', 'cluster_load_average']
                    ]
                ]
            ],
            [
                'id' => 'app_monitor',
                'label' => '系统应用监控告警',
                'description' => '监控应用同步状态的告警',
                'types' => ['standby_delay', 'rcr_sync_abnormal', 'rcr_delay'],
                'subCategories' => [
                    [
                        'id' => 'sync',
                        'label' => '同步监控',
                        'types' => ['standby_delay', 'rcr_sync_abnormal', 'rcr_delay']
                    ]
                ]
            ],
            [
                'id' => 'cluster_operation',
                'label' => '集群操作告警',
                'description' => '集群相关操作任务的告警',
                'types' => [
                    'create_cluster', 
                    'delete_cluster', 
                    'add_shards', 
                    'delete_shard', 
                    'add_comps', 
                    'delete_comp', 
                    'add_nodes', 
                    'delete_node',
                    'manual_switch'
                ],
                'subCategories' => [
                    [
                        'id' => 'cluster',
                        'label' => '集群操作',
                        'types' => ['create_cluster', 'delete_cluster']
                    ],
                    [
                        'id' => 'shard',
                        'label' => '分片操作',
                        'types' => ['add_shards', 'delete_shard']
                    ],
                    [
                        'id' => 'node',
                        'label' => '节点操作',
                        'types' => ['add_comps', 'delete_comp', 'add_nodes', 'delete_node']
                    ],
                    [
                        'id' => 'switch',
                        'label' => '主备切换',
                        'types' => ['manual_switch']
                    ]
                ]
            ],
            [
                'id' => 'backup_task',
                'label' => '备份任务告警',
                'description' => '备份任务相关的告警',
                'types' => ['shard_coldbackup', 'manual_backup_cluster'],
                'subCategories' => [
                    [
                        'id' => 'auto',
                        'label' => '自动备份',
                        'types' => ['shard_coldbackup']
                    ],
                    [
                        'id' => 'manual',
                        'label' => '手动备份',
                        'types' => ['manual_backup_cluster']
                    ]
                ]
            ],
            [
                'id' => 'data_operation',
                'label' => '数据操作告警',
                'description' => '数据操作相关的告警',
                'types' => ['expand_cluster'],
                'subCategories' => [
                    [
                        'id' => 'table_move',
                        'label' => '搬表操作',
                        'types' => ['expand_cluster']
                    ]
                ]
            ],
            [
                'id' => 'cdc_monitor',
                'label' => 'CDC监控告警',
                'description' => 'CDC集群监控相关的告警',
                'types' => [
                    'cdc_leader_change',
                    'cdc_node_exception',
                    'cdc_sync_target_failed',
                    'cdc_parse_ddl_failed'
                ],
                'subCategories' => [
                    [
                        'id' => 'cdc_status',
                        'label' => 'CDC状态监控',
                        'types' => ['cdc_leader_change', 'cdc_node_exception']
                    ],
                    [
                        'id' => 'cdc_sync',
                        'label' => 'CDC同步监控',
                        'types' => ['cdc_sync_target_failed', 'cdc_parse_ddl_failed']
                    ]
                ]
            ]
        ];

        return $categories;
    }

    /**
     * 获取告警类型详细信息
     */
    public function getAlarmTypeDetails()
    {
        // 按照PRD文档定义告警类型的详细信息
        return [
            // 系统监控告警
            'machine_offline' => [
                'label' => '机器离线告警',
                'level' => 'WARNING',
                'trigger_condition' => "节点状态为'dead'",
                'monitor_metric' => 'server_nodes.node_stats',
                'default_push' => 'system'
            ],
            'comp_node_exception' => [
                'label' => '计算节点异常',
                'level' => 'WARNING',
                'trigger_condition' => "计算节点状态为'inactive'",
                'monitor_metric' => 'comp_nodes.status',
                'default_push' => 'system'
            ],
            'storage_node_exception' => [
                'label' => '存储节点异常',
                'level' => 'WARNING',
                'trigger_condition' => "存储节点状态为'inactive'",
                'monitor_metric' => 'shard_nodes.status',
                'default_push' => 'system'
            ],
            'cluster_cpu_usage' => [
                'label' => 'CPU使用率告警',
                'level' => 'WARNING',
                'trigger_condition' => 'CPU使用率 > 配置阈值（默认80%）',
                'monitor_metric' => 'cluster_nodes.cpu_usage',
                'default_push' => 'system'
            ],
            'cluster_memory_usage' => [
                'label' => '内存使用率告警',
                'level' => 'WARNING',
                'trigger_condition' => '内存使用率 > 配置阈值（默认85%）',
                'monitor_metric' => 'cluster_nodes.memory_usage',
                'default_push' => 'system'
            ],
            'cluster_storage_usage' => [
                'label' => '存储空间告警',
                'level' => 'WARNING',
                'trigger_condition' => '存储空间使用率 > 配置阈值（默认85%）',
                'monitor_metric' => 'cluster_nodes.storage_usage',
                'default_push' => 'system'
            ],
            'cluster_io_performance' => [
                'label' => 'IO性能告警',
                'level' => 'WARNING',
                'trigger_condition' => 'IO等待时间 > 配置阈值（默认100ms）或 IOPS > 配置阈值',
                'monitor_metric' => 'cluster_nodes.io_stats',
                'default_push' => 'system'
            ],
            'cluster_load_average' => [
                'label' => '系统负载告警',
                'level' => 'WARNING',
                'trigger_condition' => '系统负载 > 配置阈值（默认当前CPU核心数的2倍）',
                'monitor_metric' => 'cluster_nodes.load_average',
                'default_push' => 'system'
            ],

            // 系统应用监控告警
            'standby_delay' => [
                'label' => '主备延迟告警',
                'level' => 'WARNING',
                'trigger_condition' => 'replica_delay > 配置阈值',
                'monitor_metric' => 'shard_nodes.replica_delay',
                'default_push' => 'system'
            ],
            'rcr_sync_abnormal' => [
                'label' => 'RCR同步异常',
                'level' => 'WARNING',
                'trigger_condition' => "meta_sync_state = 'disconnect'",
                'monitor_metric' => 'cluster_rcr_meta_sync.meta_sync_state',
                'default_push' => 'system'
            ],
            'rcr_delay' => [
                'label' => 'RCR延迟过大',
                'level' => 'WARNING',
                'trigger_condition' => 'RCR复制延迟 > 配置阈值',
                'monitor_metric' => 'cluster_rcr_infos.replica_delay',
                'default_push' => 'system'
            ],

            // 集群操作告警
            'create_cluster' => [
                'label' => '集群创建告警',
                'level' => 'WARNING',
                'trigger_condition' => '创建集群任务失败',
                'monitor_metric' => 'cluster_general_job_log',
                'default_push' => 'system'
            ],
            'delete_cluster' => [
                'label' => '集群删除告警',
                'level' => 'WARNING',
                'trigger_condition' => '删除集群任务失败',
                'monitor_metric' => 'cluster_general_job_log',
                'default_push' => 'system'
            ],
            'add_shards' => [
                'label' => '新增分片告警',
                'level' => 'WARNING',
                'trigger_condition' => '新增分片任务失败',
                'monitor_metric' => 'cluster_general_job_log',
                'default_push' => 'system'
            ],
            'delete_shard' => [
                'label' => '删除分片告警',
                'level' => 'WARNING',
                'trigger_condition' => '删除分片任务失败',
                'monitor_metric' => 'cluster_general_job_log',
                'default_push' => 'system'
            ],
            'add_comps' => [
                'label' => '新增计算节点告警',
                'level' => 'WARNING',
                'trigger_condition' => '新增计算节点任务失败',
                'monitor_metric' => 'cluster_general_job_log',
                'default_push' => 'system'
            ],
            'delete_comp' => [
                'label' => '删除计算节点告警',
                'level' => 'WARNING',
                'trigger_condition' => '删除计算节点任务失败',
                'monitor_metric' => 'cluster_general_job_log',
                'default_push' => 'system'
            ],
            'add_nodes' => [
                'label' => '新增存储节点告警',
                'level' => 'WARNING',
                'trigger_condition' => '新增存储节点任务失败',
                'monitor_metric' => 'cluster_general_job_log',
                'default_push' => 'system'
            ],
            'delete_node' => [
                'label' => '删除存储节点告警',
                'level' => 'WARNING',
                'trigger_condition' => '删除存储节点任务失败',
                'monitor_metric' => 'cluster_general_job_log',
                'default_push' => 'system'
            ],
            'manual_switch' => [
                'label' => '主备切换失败',
                'level' => 'WARNING',
                'trigger_condition' => '主备切换操作失败',
                'monitor_metric' => 'rbr_consfailover',
                'default_push' => 'system'
            ],

            // 备份任务告警
            'shard_coldbackup' => [
                'label' => '自动备份告警',
                'level' => 'WARNING',
                'trigger_condition' => '备份任务失败',
                'monitor_metric' => 'backup_jobs',
                'default_push' => 'system'
            ],
            'manual_backup_cluster' => [
                'label' => '全量备份告警',
                'level' => 'WARNING',
                'trigger_condition' => '全量备份任务失败',
                'monitor_metric' => 'backup_jobs',
                'default_push' => 'system'
            ],
            
            // 数据操作告警
            'expand_cluster' => [
                'label' => '搬表失败',
                'level' => 'WARNING',
                'trigger_condition' => '搬表任务失败',
                'monitor_metric' => 'cluster_general_job_log',
                'default_push' => 'system'
            ],
            
            // CDC监控告警
            'cdc_leader_change' => [
                'label' => 'CDC主备切换告警',
                'level' => 'WARNING',
                'trigger_condition' => 'CDC集群主节点发生变化',
                'monitor_metric' => 'cdc_leader_info',
                'default_push' => 'system'
            ],
            'cdc_node_exception' => [
                'label' => 'CDC节点异常告警',
                'level' => 'WARNING',
                'trigger_condition' => 'CDC节点不响应get_leader API请求',
                'monitor_metric' => 'cdc_node_status',
                'default_push' => 'system'
            ],
            'cdc_sync_target_failed' => [
                'label' => 'CDC同步目标数据库失败',
                'level' => 'WARNING',
                'trigger_condition' => 'CDC event_sql插件写入目标数据库失败',
                'monitor_metric' => 'cdc_job_state',
                'default_push' => 'system'
            ],
            'cdc_parse_ddl_failed' => [
                'label' => 'CDC解析DDL失败',
                'level' => 'WARNING',
                'trigger_condition' => 'CDC解析binlog中DDL语句失败，dump_state为"stop by cdc inner error"',
                'monitor_metric' => 'cdc_job_state',
                'default_push' => 'system'
            ]
        ];
    }

    /**
     * 根据告警类型获取分类
     */
    public function getCategoryByType($alarmType)
    {
        $categories = $this->getCategories();
        
        foreach ($categories as $category) {
            if (in_array($alarmType, $category['types'])) {
                return $category;
            }
        }
        
        return [
            'id' => 'unknown',
            'label' => '未知分类'
        ];
    }

    /**
     * 根据告警类型获取详细信息
     */
    public function getAlarmTypeDetail($alarmType)
    {
        $details = $this->getAlarmTypeDetails();
        
        if (isset($details[$alarmType])) {
            return $details[$alarmType];
        }
        
        return [
            'label' => $alarmType,
            'level' => 'WARNING',
            'default_push' => 'system'
        ];
    }

    /**
     * 获取告警级别列表
     */
    public function getAlarmLevels()
    {
        return [
            ['id' => 'FATAL', 'label' => '致命错误', 'description' => '需立即处理的致命错误'],
            ['id' => 'ERROR', 'label' => '严重错误', 'description' => '需尽快处理的严重错误'],
            ['id' => 'WARNING', 'label' => '警告信息', 'description' => '需关注处理的警告信息']
        ];
    }

    /**
     * 添加告警类型到分类
     */
    public function addTypeToCategory($categoryId, $alarmType)
    {
        // 这里可以实现将新的告警类型添加到分类中的逻辑
        // 由于我们目前使用的是静态数组，这个方法暂时不实现具体功能
        // 如果将来使用数据库存储分类信息，可以在这里实现添加逻辑
        return true;
    }
} 