package logger

import (
	"os"
	"path/filepath"
	"sync"
	"time"

	"github.com/sirupsen/logrus"
	"gopkg.in/natefinch/lumberjack.v2"
)

type Logger struct {
	*logrus.Logger
}

var (
	instance *Logger
	once     sync.Once
)

// NewLogger 创建日志实例，支持日志轮转
func NewLogger(logDir string) (*Logger, error) {
	var err error
	once.Do(func() {
		if err = os.MkdirAll(logDir, os.ModePerm); err != nil {
			return
		}

		logFilePath := filepath.Join(logDir, "consistent_sql.log")

		// 使用lumberjack进行日志轮转
		rotateLogger := &lumberjack.Logger{
			Filename:   logFilePath,
			MaxSize:    100,  // 每个日志文件最大100MB
			MaxBackups: 10,   // 保留10个备份
			MaxAge:     30,   // 保留30天
			Compress:   true, // 压缩旧日志
		}

		logger := logrus.New()
		// 同时输出到文件和标准输出
		logger.SetOutput(rotateLogger)
		logger.SetFormatter(&logrus.JSONFormatter{
			TimestampFormat: time.RFC3339,
		})
		logger.SetLevel(logrus.InfoLevel)

		instance = &Logger{Logger: logger}
	})

	return instance, err
}

// GetLogger 获取日志实例
func GetLogger() *Logger {
	return instance
}

// 提供各种级别的日志函数

// Debug 输出调试级别日志
func Debug(format string, args ...interface{}) {
	if instance != nil {
		instance.Debugf(format, args...)
	}
}

// Info 输出信息级别日志
func Info(format string, args ...interface{}) {
	if instance != nil {
		instance.Infof(format, args...)
	}
}

// Warn 输出警告级别日志
func Warn(format string, args ...interface{}) {
	if instance != nil {
		instance.Warnf(format, args...)
	}
}

// Error 输出错误级别日志
func Error(format string, args ...interface{}) {
	if instance != nil {
		instance.Errorf(format, args...)
	}
}

// Fatal 输出致命错误日志并终止程序
func Fatal(format string, args ...interface{}) {
	if instance != nil {
		instance.Fatalf(format, args...)
	}
}

// WithField 添加单个字段到日志
func WithField(key string, value interface{}) *logrus.Entry {
	if instance != nil {
		return instance.WithField(key, value)
	}
	return nil
}

// WithFields 添加多个字段到日志
func WithFields(fields logrus.Fields) *logrus.Entry {
	if instance != nil {
		return instance.WithFields(fields)
	}
	return nil
}

// 保留旧接口以兼容现有代码
func InfoLog(message string) {
	if instance != nil {
		instance.Info(message)
	}
}

func Printf(format string, args ...interface{}) {
	if instance != nil {
		instance.Infof(format, args...)
	}
}
