package main

import (
	"consistent_sql/logger"
	"consistent_sql/model"
	"consistent_sql/protocol"
	"consistent_sql/task"
	"encoding/binary"
	"encoding/json"
	"flag"
	"fmt"
	"io"
	"log"
	"net"
	"os"
	"os/signal"
	"path/filepath"
	"runtime/debug"
	"strings"
	"sync"
	"syscall"
	"time"
)

var taskManager = task.NewMongoTaskManager()

var taskTdManager = task.NewTdTaskManager()

// 添加ES任务管理器
var taskESManager = task.NewESTaskManager()

// 添加流式同步任务管理器
var streamTaskManager *task.StreamTaskManager

// 添加缓冲区池
var bufferPool = sync.Pool{
	New: func() interface{} {
		return make([]byte, PerformanceConfig.SocketBufferSize)
	},
}

// 任务并发控制
var taskSemaphore = make(chan struct{}, PerformanceConfig.MaxConcurrentTasks)

// syncTask 是同步任务的示例函数

func handleSourceConnection(conn net.Conn) {
	defer func() {
		if r := recover(); r != nil {
			stackTrace := debug.Stack()
			logger.Error("连接处理中发生异常: %v\n堆栈跟踪: %s", r, stackTrace)
		}
		conn.Close()
	}()

	// 设置Socket参数以提高性能
	if tcpConn, ok := conn.(*net.TCPConn); ok {
		tcpConn.SetNoDelay(true)
		tcpConn.SetKeepAlive(true)
		tcpConn.SetKeepAlivePeriod(60 * time.Second)
	}

	logger.Info("从 %s 接收到连接", conn.RemoteAddr().String())

	for {
		var length uint32
		if err := binary.Read(conn, binary.LittleEndian, &length); err != nil {
			if err == io.EOF {
				logger.Info("客户端断开连接")
				return
			} else {
				logger.Error("读取消息长度失败: %s", err.Error())
			}
			continue
		}

		// 检查消息大小是否合理
		if length > uint32(PerformanceConfig.SocketBufferSize) {
			logger.Warn("消息大小超过缓冲区大小: %d > %d", length, PerformanceConfig.SocketBufferSize)
			protocol.SendError(conn, "1", "消息大小超过限制", "")
			continue
		}

		// 从池中获取缓冲区
		buffer := bufferPool.Get().([]byte)[:length]
		defer bufferPool.Put(buffer)

		// 读取消息内容
		_, err := io.ReadFull(conn, buffer)
		if err != nil {
			// 增强客户端中断日志记录
			if err == io.EOF || strings.Contains(err.Error(), "use of closed network connection") {
				logger.Info("读取消息内容时客户端断开连接: %s", conn.RemoteAddr().String())
				return
			} else if ne, ok := err.(net.Error); ok {
				if ne.Timeout() {
					logger.Warn("读取消息内容超时: %s, 客户端地址: %s", err.Error(), conn.RemoteAddr().String())
				} else if ne.Temporary() {
					logger.Warn("读取消息内容临时网络错误: %s, 客户端地址: %s", err.Error(), conn.RemoteAddr().String())
				} else {
					logger.Error("读取消息内容致命网络错误: %s, 客户端地址: %s", err.Error(), conn.RemoteAddr().String())
					return
				}
			} else {
				logger.Error("读取消息数据失败: %s, 客户端地址: %s", err.Error(), conn.RemoteAddr().String())
			}
			continue
		}

		// 记录收到的消息
		logger.Debug("接收到消息: %s", string(buffer))

		// 检查是否是确认消息(MRsprowmsg)
		// 检查第一个字节是否是MRsprowmsg类型
		if len(buffer) > 0 && buffer[0] == byte(protocol.MRsprowmsg) {
			// 这是一个确认消息，需要分发到相应的任务

			// 分发到所有活动的MongoDB任务
			for _, taskID := range taskManager.GetAllTaskIDs() {
				// 分发确认消息到相应的任务
				taskManager.HandleAckMessage(taskID, buffer[0], buffer)
			}

			// 分发到所有活动的ES任务
			for _, taskID := range taskESManager.GetAllTaskIDs() {
				// 分发确认消息到相应的ES任务
				taskESManager.HandleAckMessage(taskID, buffer[0], buffer)
			}
			continue
		}

		// 检查旧格式的确认消息
		rawMsg := string(buffer)
		if strings.Contains(rawMsg, "\"type\":4") || strings.Contains(rawMsg, "M_RspRowMsg") {
			// 这是一个旧格式的确认消息
			// 同样，分发到所有活动的任务
			for _, taskID := range taskManager.GetAllTaskIDs() {
				taskManager.HandleAckMessage(taskID, 0, buffer) // 类型0表示文本格式
			}

			// 同样分发到所有活动的ES任务
			for _, taskID := range taskESManager.GetAllTaskIDs() {
				taskESManager.HandleAckMessage(taskID, 0, buffer) // 类型0表示文本格式
			}
			continue
		}

		// 解析其他类型的消息
		var message protocol.Message
		err = json.Unmarshal(buffer, &message)
		if err != nil {
			logger.Error("解析消息失败: %s", err.Error())
			continue
		}

		logger.Info("源接收到: %s - %s", message.Type.String(), message.Data)

		// 处理消息
		switch message.Type {
		case protocol.MReqdumpjob:
			var recv model.RecvData
			if err := json.Unmarshal([]byte(message.Data), &recv); err != nil {
				logger.Error("解析消息数据失败: %s", err.Error())
				protocol.SendError(conn, "1", "RspDumpJob 参数错误", recv.JobId)
				continue
			}

			// 为非必填字段设置默认值
			// ES相关字段默认值
			if recv.ESScrollTime == "" {
				recv.ESScrollTime = PerformanceConfig.ESDefaultScrollTime
			}
			if recv.ESBatchSize <= 0 {
				recv.ESBatchSize = PerformanceConfig.ESDefaultBatchSize
			}

			// PB级数据同步相关字段默认值
			if recv.ChunkSize <= 0 {
				recv.ChunkSize = PerformanceConfig.ChunkSize
			}
			if recv.ParallelStreams <= 0 {
				recv.ParallelStreams = PerformanceConfig.ParallelStreams
			}
			// 默认不启用压缩
			if !recv.EnableCompression {
				recv.EnableCompression = PerformanceConfig.StreamCompression
			}
			if recv.CompressionLevel <= 0 {
				recv.CompressionLevel = PerformanceConfig.CompressionLevel
			}
			// 默认启用检查点
			if !recv.CheckpointEnabled {
				recv.CheckpointEnabled = true
			}
			if recv.CheckpointPath == "" {
				recv.CheckpointPath = filepath.Join(streamTaskManager.CheckpointDir, recv.JobId+".checkpoint")
			}

			// 验证GTID
			if recv.EventType == "create" && recv.Gtid == "" {
				logger.Warn("未提供GTID，无法创建同步任务")
				protocol.SendError(conn, "1", "必须提供GTID才能创建同步任务", recv.JobId)
				continue
			}

			// 只有创建事件需要获取信号量控制并发
			if recv.EventType == "create" {
				select {
				case taskSemaphore <- struct{}{}: // 获取信号量
					// 继续处理
				default:
					// 并发限制达到上限，拒绝请求
					logger.Warn("并发任务数已达上限，拒绝请求")
					protocol.SendError(conn, "1", "系统负载过高，请稍后再试", recv.JobId)
					continue
				}
			}

			// 检查是否是流式数据同步模式
			if recv.UseStreamMode {
				// 流模式处理逻辑
				if recv.EventType == "create" {
					logger.Info("接收到流式同步任务 ID: %s, 类型: %s, GTID: %s", recv.JobId, recv.DumpDbType, recv.Gtid)
					go func() {
						defer func() { <-taskSemaphore }() // 释放信号量
						streamTaskManager.StartStreamTask(recv, conn)
					}()
				} else if recv.EventType == "status" {
					logger.Info("查询流式任务状态: %s", recv.JobId)
					status, exists := streamTaskManager.GetTaskStatus(recv.JobId)
					if exists {
						// 发送状态响应
						statusResp := map[string]interface{}{
							"error_code":      "0",
							"error_msg":       "",
							"jobId":           recv.JobId,
							"status":          status.IsRunning,
							"progress":        status.Progress,
							"last_checkpoint": status.LastCheckpoint,
							"last_gtid":       status.Progress.LastGtid, // 添加返回最后处理的GTID
							"error":           "",
						}

						if status.Error != nil {
							statusResp["error"] = status.Error.Error()
						}

						statusJSON, _ := json.Marshal(statusResp)
						protocol.SendResponse(conn, protocol.MRspdumpjob, string(statusJSON))
					} else {
						logger.Info("流式任务 %s 不存在", recv.JobId)
						protocol.SendError(conn, "1", "任务不存在", recv.JobId)
					}
				} else if recv.EventType == "delete" {
					logger.Info("接收到流式同步任务删除请求 ID: %s", recv.JobId)
					if streamTaskManager.StopTask(recv.JobId) {
						// 任务成功停止，发送成功响应
						logger.Info("流式同步任务 %s 已成功停止", recv.JobId)
						protocol.SendError(conn, "0", fmt.Sprintf("任务 %s 已成功停止", recv.JobId), recv.JobId)
					} else {
						// 任务不存在或无法停止，发送失败响应
						logger.Warn("流式同步任务 %s 不存在或无法停止", recv.JobId)
						protocol.SendError(conn, "1", fmt.Sprintf("任务 %s 不存在或无法停止", recv.JobId), recv.JobId)
					}
				}
				continue
			}

			// 处理MongoDB任务
			if recv.DumpDbType == "mongodb" {
				if recv.EventType == "create" {
					logger.Info("接收到 MongoDB 创建任务 ID: %s, GTID: %s", recv.JobId, recv.Gtid)
					go func() {
						defer func() { <-taskSemaphore }() // 释放信号量
						taskManager.StartTaskMongo(recv, conn)
					}()
				}

				if recv.EventType == "status" {
					logger.Info("查询MongoDB任务状态: %s", recv.JobId)
					status, exists := taskManager.GetTaskStatus(recv.JobId)
					if exists {
						logger.Info("任务 %s 状态: 正在运行: %v, 最后同步的 ID: %s, 错误: %v",
							recv.JobId, status.IsRunning, status.LastID, status.Error)

						// 发送状态响应
						statusResp := map[string]interface{}{
							"error_code": "0",
							"error_msg":  "",
							"jobId":      recv.JobId,
							"status":     status.IsRunning,
							"last_id":    status.LastID,
							"gtid":       recv.Gtid, // 返回GTID信息
							"error":      "",
						}
						if status.Error != nil {
							statusResp["error"] = status.Error.Error()
						}

						statusJSON, _ := json.Marshal(statusResp)
						protocol.SendResponse(conn, protocol.MRspdumpjob, string(statusJSON))
					} else {
						logger.Info("任务 %s 不存在", recv.JobId)
						protocol.SendError(conn, "1", "任务不存在", recv.JobId)
					}
				}

				// 处理delete事件
				if recv.EventType == "delete" {
					logger.Info("接收到 MongoDB 删除任务请求 ID: %s", recv.JobId)
					if taskManager.StopTask(recv.JobId) {
						// 任务成功停止，发送成功响应
						logger.Info("MongoDB任务 %s 已成功停止", recv.JobId)
						protocol.SendError(conn, "0", fmt.Sprintf("任务 %s 已成功停止", recv.JobId), recv.JobId)
					} else {
						// 任务不存在或无法停止，发送失败响应
						logger.Warn("MongoDB任务 %s 不存在或无法停止", recv.JobId)
						protocol.SendError(conn, "1", fmt.Sprintf("任务 %s 不存在或无法停止", recv.JobId), recv.JobId)
					}
				}
			}

			// 处理TDengine任务
			if recv.DumpDbType == "tdengine" {
				if recv.EventType == "create" {
					logger.Info("接收到 TDengine 创建任务 ID: %s, GTID: %s", recv.JobId, recv.Gtid)
					go func() {
						defer func() { <-taskSemaphore }() // 释放信号量
						var taskTdManager = task.NewTdTaskManager()
						taskTdManager.StartTaskTD(recv, conn)
					}()
				}

				if recv.EventType == "status" {
					logger.Info("查询TDengine任务状态: %s", recv.JobId)
					for _, taskName := range []string{recv.JobId} {
						status, exists := taskManager.GetTaskStatus(taskName)
						if exists {
							logger.Info(fmt.Sprintf("任务 %s 状态: 正在运行: %v, 最后同步的 ID: %s, 错误: %v", taskName, status.IsRunning, status.LastID, status.Error), "")
						} else {
							logger.Info("任务 %s 不存在", taskName)
						}
					}
				}

				// 处理delete事件
				if recv.EventType == "delete" {
					logger.Info("接收到 TDengine 删除任务请求 ID: %s", recv.JobId)
					// 创建新的TDengine任务管理器并停止任务
					var tdManager = task.NewTdTaskManager()
					if tdManager.StopTask(recv.JobId) {
						// 任务成功停止，发送成功响应
						logger.Info("TDengine任务 %s 已成功停止", recv.JobId)
						protocol.SendError(conn, "0", fmt.Sprintf("任务 %s 已成功停止", recv.JobId), recv.JobId)
					} else {
						// 任务不存在或无法停止，发送失败响应
						logger.Warn("TDengine任务 %s 不存在或无法停止", recv.JobId)
						protocol.SendError(conn, "1", fmt.Sprintf("任务 %s 不存在或无法停止", recv.JobId), recv.JobId)
					}
				}
			}

			// 处理Elasticsearch任务
			if recv.DumpDbType == "es" {
				if recv.EventType == "create" {
					logger.Info("接收到 Elasticsearch 创建任务 ID: %s, GTID: %s", recv.JobId, recv.Gtid)

					// 确保ESIndex参数正确设置
					if recv.ESIndex == "" {
						// 尝试从db_database获取索引名称
						if recv.DbDatabase != "" {
							logger.Info("任务ID=%s: 未设置ESIndex，使用DbDatabase(%s)作为索引名称",
								recv.JobId, recv.DbDatabase)
							recv.ESIndex = recv.DbDatabase
						}
					}

					logger.Info("任务ID=%s: 将使用索引 %s 处理ES数据", recv.JobId, recv.ESIndex)

					go func() {
						defer func() { <-taskSemaphore }() // 释放信号量
						taskESManager.StartTaskES(recv, conn)
					}()
				}

				if recv.EventType == "status" {
					logger.Info("查询ES任务状态: %s", recv.JobId)
					status, exists := taskESManager.GetTaskStatus(recv.JobId)
					if exists {
						logger.Info("ES任务 %s 状态: 正在运行: %v, 最后Scroll ID: %s, 最后时间戳: %d, 错误: %v",
							recv.JobId, status.IsRunning, status.LastScrollID, status.LastTimestamp, status.Error)

						// 发送状态响应
						statusResp := map[string]interface{}{
							"error_code":     "0",
							"error_msg":      "",
							"jobId":          recv.JobId,
							"status":         status.IsRunning,
							"last_scroll_id": status.LastScrollID,
							"last_timestamp": status.LastTimestamp,
							"gtid":           recv.Gtid, // 返回GTID信息
							"error":          "",
						}
						if status.Error != nil {
							statusResp["error"] = status.Error.Error()
						}

						statusJSON, _ := json.Marshal(statusResp)
						protocol.SendResponse(conn, protocol.MRspdumpjob, string(statusJSON))
					} else {
						logger.Info("ES任务 %s 不存在", recv.JobId)
						protocol.SendError(conn, "1", "任务不存在", recv.JobId)
					}
				}

				// 处理delete事件
				if recv.EventType == "delete" {
					logger.Info("接收到 Elasticsearch 删除任务请求 ID: %s", recv.JobId)
					if taskESManager.StopTask(recv.JobId) {
						// 任务成功停止，发送成功响应
						logger.Info("ES任务 %s 已成功停止", recv.JobId)
						protocol.SendError(conn, "0", fmt.Sprintf("任务 %s 已成功停止", recv.JobId), recv.JobId)
					} else {
						// 任务不存在或无法停止，发送失败响应
						logger.Warn("ES任务 %s 不存在或无法停止", recv.JobId)
						protocol.SendError(conn, "1", fmt.Sprintf("任务 %s 不存在或无法停止", recv.JobId), recv.JobId)
					}
				}
			}
		}
	}
}

func main() {
	// 捕获全局panic
	defer func() {
		if r := recover(); r != nil {
			stackTrace := debug.Stack()
			logger.Error("服务发生致命错误: %v\n堆栈跟踪: %s", r, stackTrace)
			// 确保错误日志被刷新到磁盘
			time.Sleep(100 * time.Millisecond)
			os.Exit(1)
		}
	}()

	// 定义命令行参数
	workDir := flag.String("workDir", "/tmp", "工作目录")
	logDir := flag.String("logDir", "./", "logdir")
	serverUnixFile := flag.String("serverUnixFile", "/Users/helay/Documents/GitHub/Kunlun-XPanel/KunlunMonitor/go/consistent_sql/socket.sock", "Unix socket 文件路径")
	maxConcurrent := flag.Int("maxConcurrent", PerformanceConfig.MaxConcurrentTasks, "最大并发任务数")
	checkpointDir := flag.String("checkpointDir", "./checkpoints", "检查点目录")

	// PB级数据同步参数
	chunkSize := flag.Int64("chunkSize", PerformanceConfig.ChunkSize, "数据分块大小(MB)")
	maxMemoryUsage := flag.Int64("maxMemoryUsage", PerformanceConfig.MaxMemoryUsage, "最大内存使用量(MB)")
	parallelStreams := flag.Int("parallelStreams", PerformanceConfig.ParallelStreams, "并行数据流数量")
	enableCompression := flag.Bool("enableCompression", PerformanceConfig.StreamCompression, "是否启用流压缩")
	compressionLevel := flag.Int("compressionLevel", PerformanceConfig.CompressionLevel, "压缩级别(1-9)，1最快，9压缩率最高")

	// 解析命令行参数
	flag.Parse()

	// 更新配置
	PerformanceConfig.MaxConcurrentTasks = *maxConcurrent
	PerformanceConfig.ChunkSize = *chunkSize
	PerformanceConfig.MaxMemoryUsage = *maxMemoryUsage
	PerformanceConfig.ParallelStreams = *parallelStreams
	PerformanceConfig.StreamCompression = *enableCompression
	PerformanceConfig.CompressionLevel = *compressionLevel

	taskSemaphore = make(chan struct{}, PerformanceConfig.MaxConcurrentTasks)

	// 初始化流任务管理器
	streamTaskManager = task.NewStreamTaskManager(*checkpointDir)

	// 初始化Logger
	_, err := logger.NewLogger(*logDir)
	if err != nil {
		log.Fatal(err)
	}

	logger.Info("启动服务，最大并发任务数: %d, 流并行度: %d",
		PerformanceConfig.MaxConcurrentTasks,
		PerformanceConfig.ParallelStreams)

	startSource(workDir, serverUnixFile)
}

func startSource(workDir, serverUnixFile *string) {
	// 改用结构化日志
	logger.Info("启动服务，最大并发任务数: %d, 流并行度: %d",
		PerformanceConfig.MaxConcurrentTasks, PerformanceConfig.ParallelStreams)

	// 设置信号处理
	sigChan := make(chan os.Signal, 1)
	signal.Notify(sigChan, syscall.SIGINT, syscall.SIGTERM, syscall.SIGHUP)

	go func() {
		sig := <-sigChan
		logger.Info("接收到系统信号 %v，服务准备关闭", sig)
		// 在这里可以添加清理资源的代码
		os.Exit(0)
	}()

	// 切换到指定的工作目录
	err := os.Chdir(*workDir)
	if err != nil {
		logger.Error("无法切换到工作目录: %s", err.Error())
		return
	}

	// 删除旧的 socket 文件（如果存在）
	if err := os.Remove(*serverUnixFile); err != nil && !os.IsNotExist(err) {
		logger.Error("无法删除旧的 socket: %s", err.Error())
		return
	}

	// 创建 socket
	listener, err := net.Listen("unix", *serverUnixFile)
	if err != nil {
		logger.Error("无法监听 Unix socket: %s", err.Error())
		return
	}

	// 确保listener在程序退出时关闭
	defer func() {
		listener.Close()
		logger.Info("服务器已关闭 Unix socket: %s", *serverUnixFile)
		// 清理socket文件
		os.Remove(*serverUnixFile)
	}()

	logger.Info("服务器正在监听 Unix socket: %s", *serverUnixFile)

	for {
		conn, err := listener.Accept()
		if err != nil {
			logger.Error("接受连接失败: %s", err.Error())
			// 判断是否是致命错误导致的监听中断
			if netErr, ok := err.(net.Error); ok && !netErr.Temporary() {
				logger.Error("发生致命网络错误，服务监听中断: %s", err.Error())
				return
			}
			continue
		}

		go handleSourceConnection(conn)
	}
}
