package task

import (
	"bytes"
	"consistent_sql/logger"
	"consistent_sql/model"
	"consistent_sql/protocol"
	"context"
	"encoding/json"
	"fmt"
	"net"
	"net/http"
	"strconv"
	"strings"
	"sync"
	"time"
)

// ESTaskStatus ES任务状态
type ESTaskStatus struct {
	IsRunning     bool
	LastScrollID  string // scroll ID
	LastTimestamp int64  // 最后一条记录的时间戳
	Error         error
	FirstSend     bool // 标记是否是第一次发送
	WaitingAck    bool // 是否正在等待确认
}

// ESTaskManager ES任务管理器
type ESTaskManager struct {
	Tasks    map[string]context.CancelFunc
	Statuses map[string]*ESTaskStatus
	Mu       sync.Mutex

	// 连接池
	clients   map[string]*http.Client
	clientsMu sync.Mutex

	// 添加消息处理回调
	ackHandlers   map[string]func(byte, []byte)
	ackHandlersMu sync.RWMutex
}

// NewESTaskManager 创建新的ES任务管理器
func NewESTaskManager() *ESTaskManager {
	return &ESTaskManager{
		Tasks:       make(map[string]context.CancelFunc),
		Statuses:    make(map[string]*ESTaskStatus),
		clients:     make(map[string]*http.Client),
		ackHandlers: make(map[string]func(byte, []byte)),
	}
}

// getESClient 获取或创建ES HTTP客户端
func (tm *ESTaskManager) getESClient(url string) (*http.Client, error) {
	tm.clientsMu.Lock()
	defer tm.clientsMu.Unlock()

	// 检查连接池中是否已有客户端
	if client, exists := tm.clients[url]; exists {
		return client, nil
	}

	// 创建新的优化过的HTTP客户端
	client := &http.Client{
		Transport: &http.Transport{
			MaxIdleConnsPerHost:   100,
			MaxIdleConns:          100,
			IdleConnTimeout:       90 * time.Second,
			ResponseHeaderTimeout: 10 * time.Second,
			DisableCompression:    false,
		},
		Timeout: 30 * time.Second,
	}

	// 保存到缓存
	tm.clients[url] = client
	return client, nil
}

// parseESGtid 解析ES的GTID
func (tm *ESTaskManager) parseESGtid(gtidStr string) (*model.ESGTID, error) {
	// 如果为空，返回空值
	if gtidStr == "" {
		return &model.ESGTID{
			ID:        "",
			SeqNo:     0,
			ScrollID:  "",
			Timestamp: 0,
		}, nil
	}

	// 去掉可能包裹的单引号
	gtidStr = strings.Trim(gtidStr, "'")

	// 尝试解析为新格式JSON
	var esGtid model.ESGTID
	err := json.Unmarshal([]byte(gtidStr), &esGtid)
	if err == nil {
		return &esGtid, nil
	}

	// 如果解析失败，尝试将其作为旧格式处理（即单纯的文档ID）
	logger.Warn("无法解析GTID为新格式，将作为旧格式处理: %v", err)
	return &model.ESGTID{
		ID:        gtidStr,
		SeqNo:     0,
		ScrollID:  "",
		Timestamp: 0,
	}, nil
}

// StartTaskES 启动ES同步任务
func (tm *ESTaskManager) StartTaskES(es model.RecvData, conn net.Conn) {
	tm.Mu.Lock()

	// 检查任务是否已存在
	if _, exists := tm.Tasks[es.JobId]; exists {
		tm.Mu.Unlock()
		logger.Info("ES任务已存在: %s", es.JobId)
		protocol.SendError(conn, "1", "job id "+es.JobId+" exists", es.JobId)
		return
	}

	// 创建新的上下文和取消函数
	ctx, cancel := context.WithCancel(context.Background())
	tm.Tasks[es.JobId] = cancel
	tm.Statuses[es.JobId] = &ESTaskStatus{
		IsRunning:  true,
		FirstSend:  true,  // 初始设置为第一次发送
		WaitingAck: false, // 初始设置为未等待确认
	}
	logger.Info("任务ID=%s：初始化状态，FirstSend=true（首次发送不需等待确认），WaitingAck=false（第二次及后续发送会等待确认）", es.JobId)
	tm.Mu.Unlock()

	// 运行ES任务
	go tm.syncESTask(ctx, es, conn)
}

// syncESTask 执行ES同步任务
func (tm *ESTaskManager) syncESTask(ctx context.Context, es model.RecvData, conn net.Conn) {
	logger.Info("开始执行ES同步任务: %s", es.JobId)

	// 参数验证
	if es.URL == "" {
		err := fmt.Errorf("ES URL不能为空")
		tm.updateTaskStatus(es.JobId, "", 0, err)
		protocol.SendError(conn, "1", err.Error(), es.JobId)
		return
	}

	// 确保ESIndex参数正确设置
	if es.ESIndex == "" {
		// 尝试使用db_database作为索引名称
		if es.DbDatabase != "" {
			logger.Info("ESIndex未设置，使用DbDatabase值(%s)作为索引名称", es.DbDatabase)
			es.ESIndex = es.DbDatabase
		} else {
			err := fmt.Errorf("ES索引名称不能为空")
			tm.updateTaskStatus(es.JobId, "", 0, err)
			protocol.SendError(conn, "1", err.Error(), es.JobId)
			return
		}
	}

	logger.Info("任务ID=%s: 使用索引 %s", es.JobId, es.ESIndex)

	// 解析初始GTID
	initialGtid, err := tm.parseESGtid(es.Gtid)
	if err != nil {
		errorMsg := fmt.Sprintf("解析初始GTID失败: %v", err)
		logger.Error(errorMsg)
		tm.updateTaskStatus(es.JobId, "", 0, fmt.Errorf(errorMsg))
		protocol.SendError(conn, "1", errorMsg, es.JobId)
		return
	}

	// 初始化变量
	var scrollID string = ""
	var lastTimestamp int64 = 0

	// 从初始GTID中提取scroll ID和时间戳
	if initialGtid.ID != "" {
		logger.Info("任务ID=%s: 使用初始GTID: ID=%s, SeqNo=%d, ScrollID=%s, Timestamp=%d",
			es.JobId, initialGtid.ID, initialGtid.SeqNo, initialGtid.ScrollID, initialGtid.Timestamp)

		// 如果GTID中有scroll ID，使用它来恢复查询
		if initialGtid.ScrollID != "" && initialGtid.ScrollID != "null" {
			scrollID = initialGtid.ScrollID
			logger.Info("任务ID=%s: 从GTID中提取到scroll ID: %s，将使用它恢复查询", es.JobId, scrollID)
		}

		// 设置上次时间戳
		if initialGtid.Timestamp > 0 {
			lastTimestamp = initialGtid.Timestamp
			logger.Info("任务ID=%s: 从GTID中提取到时间戳: %d", es.JobId, lastTimestamp)
		}
	} else {
		logger.Info("任务ID=%s: 未指定初始GTID", es.JobId)
	}

	// 获取ES客户端
	client, err := tm.getESClient(es.URL)
	if err != nil {
		tm.updateTaskStatus(es.JobId, "", 0, err)
		protocol.SendError(conn, "1", fmt.Sprintf("创建ES客户端失败: %v", err), es.JobId)
		return
	}

	// 验证ES连接
	pingURL := fmt.Sprintf("%s/_cluster/health", es.URL)
	pingReq, err := http.NewRequestWithContext(ctx, "GET", pingURL, nil)
	if err != nil {
		errorMsg := fmt.Sprintf("创建ES连接测试请求失败: %v", err)
		logger.Error(errorMsg)
		tm.updateTaskStatus(es.JobId, "", 0, fmt.Errorf(errorMsg))
		protocol.SendError(conn, "1", errorMsg, es.JobId)
		return
	}

	// 设置认证
	if es.DbUser != "" && es.DbPasswd != "" {
		pingReq.SetBasicAuth(es.DbUser, es.DbPasswd)
	}

	// 执行连接测试
	pingResp, err := client.Do(pingReq)
	if err != nil {
		errorMsg := fmt.Sprintf("ES连接测试失败: %v", err)
		logger.Error(errorMsg)
		tm.updateTaskStatus(es.JobId, "", 0, fmt.Errorf(errorMsg))
		protocol.SendError(conn, "1", errorMsg, es.JobId)
		return
	}
	defer pingResp.Body.Close()

	if pingResp.StatusCode >= 400 {
		errorMsg := fmt.Sprintf("ES服务器连接失败，状态码: %d", pingResp.StatusCode)
		logger.Error(errorMsg)
		tm.updateTaskStatus(es.JobId, "", 0, fmt.Errorf(errorMsg))
		protocol.SendError(conn, "1", errorMsg, es.JobId)
		return
	}

	logger.Info("成功连接到ES服务器: %s", es.URL)

	// 验证索引是否存在
	indexURL := fmt.Sprintf("%s/%s", es.URL, es.ESIndex)
	indexReq, err := http.NewRequestWithContext(ctx, "HEAD", indexURL, nil)
	if err != nil {
		errorMsg := fmt.Sprintf("创建索引验证请求失败: %v", err)
		logger.Error(errorMsg)
		tm.updateTaskStatus(es.JobId, "", 0, fmt.Errorf(errorMsg))
		protocol.SendError(conn, "1", errorMsg, es.JobId)
		return
	}

	// 设置认证
	if es.DbUser != "" && es.DbPasswd != "" {
		indexReq.SetBasicAuth(es.DbUser, es.DbPasswd)
	}

	// 执行索引验证
	indexResp, err := client.Do(indexReq)
	if err != nil {
		errorMsg := fmt.Sprintf("验证索引 %s 失败: %v", es.ESIndex, err)
		logger.Error(errorMsg)
		tm.updateTaskStatus(es.JobId, "", 0, fmt.Errorf(errorMsg))
		protocol.SendError(conn, "1", errorMsg, es.JobId)
		return
	}
	indexResp.Body.Close()

	if indexResp.StatusCode >= 400 {
		errorMsg := fmt.Sprintf("索引 %s 不存在，状态码: %d", es.ESIndex, indexResp.StatusCode)
		logger.Error(errorMsg)
		tm.updateTaskStatus(es.JobId, "", 0, fmt.Errorf(errorMsg))
		protocol.SendError(conn, "1", errorMsg, es.JobId)
		return
	}

	logger.Info("成功验证索引 %s 存在", es.ESIndex)

	// 设置TCP保活选项
	if tcpConn, ok := conn.(*net.TCPConn); ok {
		tcpConn.SetKeepAlive(true)
		tcpConn.SetKeepAlivePeriod(30 * time.Second)
	}

	// 准备批处理参数
	batchSize := 5 // 默认值
	if es.ESBatchSize > 0 {
		batchSize = es.ESBatchSize
	}

	scrollTime := "1m" // 默认值
	if es.ESScrollTime != "" {
		scrollTime = es.ESScrollTime
	}

	// 发送开始消息
	protocol.SendError(conn, "0", "job "+es.JobId+" start", es.JobId)

	// 创建接收确认消息的通道
	ackChan := make(chan bool, 1)

	// 注册确认消息处理函数
	tm.registerAckHandler(es.JobId, func(msgType byte, msgData []byte) {
		if msgType == byte(protocol.MRsprowmsg) {
			logger.Info("任务ID=%s：收到 MRsprowmsg 确认消息", es.JobId)
			// 更新任务状态
			tm.Mu.Lock()
			if status, exists := tm.Statuses[es.JobId]; exists {
				status.WaitingAck = false
			}
			tm.Mu.Unlock()

			// 通知主循环
			select {
			case ackChan <- true:
			default:
				// 如果通道已满，先清空，再发送
				select {
				case <-ackChan:
				default:
				}
				ackChan <- true
			}
		} else if len(msgData) > 0 {
			// 兼容旧消息格式（文本格式）
			rawMsg := string(msgData)
			if strings.Contains(rawMsg, "\"type\":4") || strings.Contains(rawMsg, "M_RspRowMsg") {
				logger.Info("任务ID=%s：收到旧格式 MRsprowmsg 确认消息", es.JobId)
				// 更新任务状态
				tm.Mu.Lock()
				if status, exists := tm.Statuses[es.JobId]; exists {
					status.WaitingAck = false
				}
				tm.Mu.Unlock()

				// 通知主循环
				select {
				case ackChan <- true:
				default:
					// 如果通道已满，先清空，再发送
					select {
					case <-ackChan:
					default:
					}
					ackChan <- true
				}
			}
		}
	})

	// 清理函数：在任务结束时取消注册
	defer tm.unregisterAckHandler(es.JobId)

	// 设置增量同步的时间间隔
	incremental_sync_interval := 5 * time.Second
	var lastSyncTime time.Time = time.Now()
	var noDataCount int = 0
	var useScrolling bool = scrollID != "" // 如果有初始scroll ID，设置为使用滚动模式

	// 更新任务状态
	tm.updateTaskStatusFull(es.JobId, scrollID, lastTimestamp, nil, false, false)

	// 主同步循环
	for {
		select {
		case <-ctx.Done():
			logger.Info("任务已取消: %s", es.JobId)
			tm.updateTaskStatusFull(es.JobId, scrollID, lastTimestamp, nil, false, false)
			return
		default:
			// 检查是否在等待确认
			tm.Mu.Lock()
			status, exists := tm.Statuses[es.JobId]
			isWaiting := exists && status.WaitingAck
			isFirstSend := exists && status.FirstSend
			tm.Mu.Unlock()

			if isWaiting {
				// 等待确认
				select {
				case <-ackChan:
					// 收到确认，继续处理
					logger.Info("任务ID=%s：收到MRsprowmsg确认，继续处理下一批数据", es.JobId)
				}
				// 移除超时处理，如果没收到确认就一直等待，直到客户端关闭
			}

			// 首次运行或者GTID为空时，尝试获取第一条记录ID作为起始GTID
			if es.Gtid == "" && isFirstSend && scrollID == "" {
				firstDocID, seqNo, err := tm.getFirstESDocument(es.URL, es.ESIndex, es.DbUser, es.DbPasswd)
				if err == nil && firstDocID != "" {
					logger.Info("任务ID=%s：自动获取第一条ES记录ID作为GTID: %s, seq_no: %d", es.JobId, firstDocID, seqNo)

					// 创建新格式的GTID
					newGtid := model.ESGTID{
						ID:        firstDocID,
						SeqNo:     seqNo,
						ScrollID:  "",
						Timestamp: time.Now().Unix(),
					}

					// 序列化为JSON字符串
					gtidBytes, _ := json.Marshal(newGtid)
					es.Gtid = string(gtidBytes)
					logger.Info("任务ID=%s：创建新格式GTID: %s", es.JobId, es.Gtid)
				} else {
					logger.Warn("任务ID=%s：无法自动获取第一条记录ID: %v", es.JobId, err)
				}
			}

			// 获取任务状态以检查上次同步位置
			tm.Mu.Lock()
			taskStatus, exists := tm.Statuses[es.JobId]
			if exists {
				scrollID = taskStatus.LastScrollID
				lastTimestamp = taskStatus.LastTimestamp
			}
			tm.Mu.Unlock()

			// 构建查询
			var searchBody map[string]interface{}
			if es.ESQuery != "" {
				err := json.Unmarshal([]byte(es.ESQuery), &searchBody)
				if err != nil {
					errorMsg := fmt.Sprintf("无法解析查询: %v", err)
					tm.updateTaskStatusFull(es.JobId, "", 0, fmt.Errorf(errorMsg), false, false)
					protocol.SendError(conn, "1", errorMsg, es.JobId)
					return
				}
				logger.Info("任务ID=%s: 使用自定义查询: %s", es.JobId, es.ESQuery)
			} else {
				// 默认查询所有文档
				searchBody = map[string]interface{}{
					"query": map[string]interface{}{
						"match_all": map[string]interface{}{},
					},
				}
				logger.Info("任务ID=%s: 使用默认查询(match_all)", es.JobId)
			}

			// 如果有时间字段和上次同步位置，添加时间过滤
			if es.ESTimeField != "" && lastTimestamp > 0 {
				// 向查询添加时间范围条件，使用大于而不是大于等于，确保边界记录不会被重复处理
				rangeQuery := map[string]interface{}{
					"range": map[string]interface{}{
						es.ESTimeField: map[string]interface{}{
							"gt": lastTimestamp,
						},
					},
				}

				// 检查查询结构
				if query, ok := searchBody["query"]; ok {
					// 已有查询，需要组合
					searchBody["query"] = map[string]interface{}{
						"bool": map[string]interface{}{
							"must": []interface{}{
								query,
								rangeQuery,
							},
						},
					}
				} else {
					// 直接使用范围查询
					searchBody["query"] = rangeQuery
				}
				logger.Info("任务ID=%s: 添加时间范围过滤，字段=%s, 起始时间戳=%d, 使用gt(大于)条件", es.JobId, es.ESTimeField, lastTimestamp)
			}

			// 设置排序，如果有时间字段
			if es.ESTimeField != "" {
				searchBody["sort"] = []interface{}{
					map[string]interface{}{
						es.ESTimeField: map[string]interface{}{
							"order": "asc",
						},
					},
				}
				logger.Info("任务ID=%s: 按时间字段排序 %s(asc)", es.JobId, es.ESTimeField)
			}

			// 设置大小
			searchBody["size"] = batchSize
			logger.Info("任务ID=%s: 设置批次大小=%d", es.JobId, batchSize)

			// 打印完整查询语句
			queryJson, _ := json.Marshal(searchBody)
			logger.Info("任务ID=%s: 最终查询语句: %s", es.JobId, string(queryJson))

			var result model.SyncESResult

			// 初始搜索或使用现有scroll
			var requestBody []byte
			var requestURL string
			var req *http.Request

			// 如果连续多次无数据，清除scroll重新开始
			if noDataCount > 5 {
				useScrolling = false
				scrollID = ""
				noDataCount = 0
				logger.Info("任务ID=%s：连续多次无数据，清除scroll并重新开始查询", es.JobId)
			}

			if useScrolling && scrollID != "" {
				// 使用现有scroll
				scrollReq := map[string]interface{}{
					"scroll":    scrollTime,
					"scroll_id": scrollID,
				}
				requestBody, _ = json.Marshal(scrollReq)
				requestURL = fmt.Sprintf("%s/_search/scroll", es.URL)
				logger.Info("任务ID=%s: 使用现有scroll ID: %s 继续查询", es.JobId, scrollID)

				req, err = http.NewRequestWithContext(ctx, "POST", requestURL, bytes.NewReader(requestBody))
			} else {
				// 开始新的scroll搜索
				requestBody, _ = json.Marshal(searchBody)
				requestURL = fmt.Sprintf("%s/%s/_search?scroll=%s", es.URL, es.ESIndex, scrollTime)
				logger.Info("任务ID=%s: 开始新的scroll搜索", es.JobId)

				req, err = http.NewRequestWithContext(ctx, "POST", requestURL, bytes.NewReader(requestBody))
				useScrolling = true
			}

			if err != nil {
				errorMsg := fmt.Sprintf("创建请求失败: %v", err)
				tm.updateTaskStatusFull(es.JobId, "", lastTimestamp, fmt.Errorf(errorMsg), false, false)
				protocol.SendError(conn, "1", errorMsg, es.JobId)
				return
			}

			// 设置请求头
			req.Header.Set("Content-Type", "application/json")
			if es.DbUser != "" && es.DbPasswd != "" {
				req.SetBasicAuth(es.DbUser, es.DbPasswd)
			}

			// 执行请求
			logger.Info("任务ID=%s: 发送ES查询请求到 %s", es.JobId, requestURL)
			resp, err := client.Do(req)
			if err != nil {
				errorMsg := fmt.Sprintf("ES请求失败: %v", err)
				logger.Error("任务ID=%s: %s", es.JobId, errorMsg)
				tm.updateTaskStatusFull(es.JobId, "", lastTimestamp, fmt.Errorf(errorMsg), false, false)
				protocol.SendError(conn, "1", errorMsg, es.JobId)
				return
			}

			// 检查响应状态
			if resp.StatusCode >= 400 {
				var errorResp map[string]interface{}
				json.NewDecoder(resp.Body).Decode(&errorResp)
				resp.Body.Close()
				errorMsg := fmt.Sprintf("ES响应错误: 状态码 %d, 内容: %v", resp.StatusCode, errorResp)
				logger.Error("任务ID=%s: %s", es.JobId, errorMsg)

				// 如果是scroll_id过期等问题，重置scroll状态
				if resp.StatusCode == 404 {
					logger.Warn("任务ID=%s：Scroll ID可能已过期，将重置scroll状态", es.JobId)
					scrollID = ""
					useScrolling = false
					noDataCount = 0
					time.Sleep(1 * time.Second)
					continue
				}

				tm.updateTaskStatusFull(es.JobId, "", lastTimestamp, fmt.Errorf(errorMsg), false, false)
				protocol.SendError(conn, "1", errorMsg, es.JobId)
				return
			}

			// 解析响应
			var searchResponse map[string]interface{}
			if err := json.NewDecoder(resp.Body).Decode(&searchResponse); err != nil {
				resp.Body.Close()
				errorMsg := fmt.Sprintf("解析ES响应失败: %v", err)
				logger.Error("任务ID=%s: %s", es.JobId, errorMsg)
				tm.updateTaskStatusFull(es.JobId, "", lastTimestamp, fmt.Errorf(errorMsg), false, false)
				protocol.SendError(conn, "1", errorMsg, es.JobId)
				return
			}
			resp.Body.Close()

			logger.Info("任务ID=%s: 成功收到ES响应", es.JobId)

			// 提取scroll ID
			if sid, ok := searchResponse["_scroll_id"].(string); ok && sid != "" {
				result.LastScrollID = sid
				scrollID = sid
				logger.Info("任务ID=%s: 获取到新的scroll ID: %s", es.JobId, sid)
			} else {
				logger.Warn("任务ID=%s: 响应中没有找到有效的scroll ID", es.JobId)
			}

			// 处理hits
			var processedCount int = 0
			if hits, ok := searchResponse["hits"].(map[string]interface{}); ok {
				totalHits := "未知"
				if total, ok := hits["total"].(map[string]interface{}); ok {
					if value, ok := total["value"].(float64); ok {
						totalHits = fmt.Sprintf("%d", int(value))
					}
				} else if total, ok := hits["total"].(float64); ok {
					totalHits = fmt.Sprintf("%d", int(total))
				}
				logger.Info("任务ID=%s: 查询匹配总数: %s", es.JobId, totalHits)

				if hitsArray, ok := hits["hits"].([]interface{}); ok {
					logger.Info("任务ID=%s: 本次返回记录数: %d", es.JobId, len(hitsArray))
					for _, hit := range hitsArray {
						hitObj, _ := hit.(map[string]interface{})
						id, _ := hitObj["_id"].(string)
						index, _ := hitObj["_index"].(string)
						source, _ := hitObj["_source"].(map[string]interface{})

						// 获取seq_no
						var seqNo int64 = 0
						if seqNoValue, ok := hitObj["_seq_no"].(float64); ok {
							seqNo = int64(seqNoValue)
						}

						// 提取时间戳
						var timestamp int64
						if es.ESTimeField != "" {
							if tsValue, ok := source[es.ESTimeField]; ok {
								switch v := tsValue.(type) {
								case float64:
									timestamp = int64(v)
								case string:
									// 尝试解析数字字符串
									if ts, err := strconv.ParseInt(v, 10, 64); err == nil {
										timestamp = ts
									} else {
										// 尝试解析时间字符串
										if parsedTime, err := time.Parse(time.RFC3339, v); err == nil {
											timestamp = parsedTime.UnixNano() / int64(time.Millisecond)
										}
									}
								}
							}
						}

						if timestamp > result.LastTimestamp {
							result.LastTimestamp = timestamp
							lastTimestamp = timestamp
						}

						// 添加到结果集
						result.Data = append(result.Data, model.ESSourceDBData{
							ID:        id,
							Index:     index,
							Data:      source,
							Timestamp: timestamp,
							Optype:    "insert", // 修改为"insert"，与MongoDB保持一致
							Gtid: fmt.Sprintf(`{"id":"%s","seq_no":%d,"scroll_id":%s,"timestamp":%d}`,
								id,
								seqNo,
								getJsonNullOrString(result.LastScrollID),
								timestamp),
						})

						processedCount++
					}
				}
			}

			// 发送数据到客户端
			if len(result.Data) > 0 {
				// 判断是否是首次发送
				tm.Mu.Lock()
				status, exists := tm.Statuses[es.JobId]
				isFirstSendFlag := exists && status.FirstSend
				tm.Mu.Unlock()

				if isFirstSendFlag {
					logger.Info("任务ID=%s：处理首包数据，首包数据不进行去重处理，当前GTID=%s", es.JobId, es.Gtid)
				} else {
					logger.Info("任务ID=%s：处理后续包数据，将过滤掉与GTID=%s相同的记录", es.JobId, es.Gtid)
				}

				// 如果不是首次发送，才进行过滤去重
				filteredData := result.Data
				if !isFirstSendFlag {
					// 解析当前GTID
					esGtid, err := tm.parseESGtid(es.Gtid)
					if err != nil {
						logger.Warn("任务ID=%s：解析GTID失败: %v, 将不进行过滤", es.JobId, err)
					} else {
						logger.Info("任务ID=%s：准备过滤数据，基于GTID: ID=%s", es.JobId, esGtid.ID)

						filteredData = make([]model.ESSourceDBData, 0, len(result.Data))
						for _, doc := range result.Data {
							// 数据ID与起始GTID的ID字段不同，则保留
							if doc.ID != esGtid.ID {
								filteredData = append(filteredData, doc)
							} else {
								logger.Info("任务ID=%s：过滤掉重复记录，ID=%s", es.JobId, doc.ID)
							}
						}
						logger.Info("任务ID=%s：过滤结果 - 过滤前%d条记录，过滤后%d条记录，过滤掉%d条重复记录",
							es.JobId, len(result.Data), len(filteredData), len(result.Data)-len(filteredData))
					}
				} else {
					logger.Info("任务ID=%s：首包数据不进行去重，保留所有%d条记录", es.JobId, len(result.Data))
				}

				// 更新处理后的数据
				result.Data = filteredData

				// 如果过滤后没有数据了，则跳过此次发送
				if len(result.Data) == 0 {
					logger.Info("任务ID=%s：过滤重复记录后无新数据，继续下一轮查询", es.JobId)
					time.Sleep(100 * time.Millisecond)
					continue
				}

				noDataCount = 0
				lastSyncTime = time.Now()

				// 发送数据
				if err := tm.sendSyncResult(conn, es.JobId, es.DbDatabase, &result); err != nil {
					logger.Error("发送数据失败: %v", err)
					time.Sleep(100 * time.Millisecond)
					continue
				}

				// 更新等待状态
				if !isFirstSendFlag {
					tm.Mu.Lock()
					if status, exists := tm.Statuses[es.JobId]; exists {
						status.WaitingAck = true
					}
					tm.Mu.Unlock()
					logger.Info("任务ID=%s：非首次批量数据发送完成，本批次共发送%d条记录，进入等待确认状态，收到MRsprowmsg后才能继续",
						es.JobId, len(result.Data))
				} else {
					tm.Mu.Lock()
					if status, exists := tm.Statuses[es.JobId]; exists {
						status.FirstSend = false
					}
					tm.Mu.Unlock()
					logger.Info("任务ID=%s：首次批量数据发送完成，本批次共发送%d条记录，将继续处理下一批数据，无需等待确认",
						es.JobId, len(result.Data))
				}

				// 更新任务状态
				var waitingFlag bool = false
				if !isFirstSendFlag {
					waitingFlag = true
				}
				tm.updateTaskStatusFull(es.JobId, scrollID, lastTimestamp, nil, false, waitingFlag)
			} else {
				// 没有新数据，增加计数器
				noDataCount++

				// 如果没有搜索到新数据，并且距离上次同步已经超过间隔时间，清理当前scroll并等待
				if time.Since(lastSyncTime) > incremental_sync_interval {
					// 如果无数据但有scroll ID，清理scroll
					if scrollID != "" && noDataCount > 3 {
						tm.cleanupScroll(es.URL, scrollID, es.DbUser, es.DbPasswd)
						scrollID = ""
						useScrolling = false
						logger.Info("任务ID=%s：没有新数据，已清理scroll，下次将重新搜索", es.JobId)
					}

					logger.Info("任务ID=%s：没有新数据，等待增量同步间隔(%v)后继续", es.JobId, incremental_sync_interval)
					time.Sleep(incremental_sync_interval)
					lastSyncTime = time.Now()
				} else {
					// 短暂等待后继续
					time.Sleep(100 * time.Millisecond)
				}
			}
		}
	}
}

// getFirstESDocument 获取ES索引中的第一条文档ID及序列号
func (tm *ESTaskManager) getFirstESDocument(url, index, username, password string) (string, int64, error) {
	client, err := tm.getESClient(url)
	if err != nil {
		return "", 0, fmt.Errorf("获取ES客户端失败: %v", err)
	}

	// 构建查询请求，获取第一条记录
	searchURL := fmt.Sprintf("%s/%s/_search?size=1&sort=_id:asc&seq_no_primary_term=true", url, index)
	req, err := http.NewRequest("GET", searchURL, nil)
	if err != nil {
		return "", 0, fmt.Errorf("创建请求失败: %v", err)
	}

	req.Header.Set("Content-Type", "application/json")
	if username != "" && password != "" {
		req.SetBasicAuth(username, password)
	}

	// 执行请求
	resp, err := client.Do(req)
	if err != nil {
		return "", 0, fmt.Errorf("执行请求失败: %v", err)
	}
	defer resp.Body.Close()

	// 检查响应状态
	if resp.StatusCode >= 400 {
		var errorResp map[string]interface{}
		json.NewDecoder(resp.Body).Decode(&errorResp)
		return "", 0, fmt.Errorf("查询失败，状态码: %d, 错误: %v", resp.StatusCode, errorResp)
	}

	// 解析响应
	var result map[string]interface{}
	if err := json.NewDecoder(resp.Body).Decode(&result); err != nil {
		return "", 0, fmt.Errorf("解析响应失败: %v", err)
	}

	// 提取第一条记录的ID和seq_no
	if hits, ok := result["hits"].(map[string]interface{}); ok {
		if hitsArray, ok := hits["hits"].([]interface{}); ok && len(hitsArray) > 0 {
			hit := hitsArray[0].(map[string]interface{})
			id, _ := hit["_id"].(string)

			// 获取seq_no
			var seqNo int64 = 0
			if seqNoValue, ok := hit["_seq_no"].(float64); ok {
				seqNo = int64(seqNoValue)
			}

			return id, seqNo, nil
		}
	}

	return "", 0, fmt.Errorf("未找到任何记录")
}

// 更新任务状态
func (tm *ESTaskManager) updateTaskStatus(jobId, scrollID string, timestamp int64, err error) {
	tm.Mu.Lock()
	defer tm.Mu.Unlock()

	if status, exists := tm.Statuses[jobId]; exists {
		if scrollID != "" {
			status.LastScrollID = scrollID
		}
		if timestamp > 0 {
			status.LastTimestamp = timestamp
		}
		status.Error = err
		if err != nil {
			status.IsRunning = false
		}
	}
}

// 添加支持FirstSend和WaitingAck的更新方法
func (tm *ESTaskManager) updateTaskStatusFull(jobId, scrollID string, timestamp int64, err error, isFirstSend bool, waitingAck bool) {
	tm.Mu.Lock()
	defer tm.Mu.Unlock()

	if status, exists := tm.Statuses[jobId]; exists {
		if scrollID != "" {
			status.LastScrollID = scrollID
		}
		if timestamp > 0 {
			status.LastTimestamp = timestamp
		}
		status.Error = err
		if err != nil {
			status.IsRunning = false
		}
		status.FirstSend = isFirstSend
		status.WaitingAck = waitingAck
	}
}

// sendSyncResult 发送同步结果
func (tm *ESTaskManager) sendSyncResult(conn net.Conn, jobId string, dbName string, result *model.SyncESResult) error {
	if len(result.Data) == 0 {
		return nil
	}

	// 准备行数据
	rowCount := uint32(len(result.Data))

	// 获取当前任务状态
	tm.Mu.Lock()
	isFirstSend := false
	if status, exists := tm.Statuses[jobId]; exists {
		isFirstSend = status.FirstSend
	}
	tm.Mu.Unlock()

	if isFirstSend {
		logger.Info("发送数据包(jobId=%s)：首次发送批量数据，不需要等待确认，本批次包含 %d 条记录", jobId, rowCount)
	} else {
		logger.Info("发送数据包(jobId=%s)：非首次发送批量数据，发送后需等待确认，本批次包含 %d 条记录", jobId, rowCount)
	}

	// 初始化消息buffer
	var message []byte

	// 1. 类型 (MReqrowmsg)
	messageType := byte(protocol.MReqrowmsg)
	message = append(message, messageType)

	// 2. 多少行 (uint32)
	message = append(message, byte(rowCount), byte(rowCount>>8), byte(rowCount>>16), byte(rowCount>>24))

	// 使用第一条数据的GTID
	firstData := result.Data[0]
	gtidBytes := []byte(fmt.Sprintf("'%s'", firstData.Gtid))
	gtidLen := uint32(len(gtidBytes))

	logger.Info("任务ID=%s: 使用GTID: %s", jobId, firstData.Gtid)

	// 3. gtid (len+str)
	message = append(message, byte(gtidLen), byte(gtidLen>>8), byte(gtidLen>>16), byte(gtidLen>>24))
	message = append(message, gtidBytes...)

	// 4. binlog_file (int32)，ES无binlog，直接写入0
	binlogFile := int32(0)
	message = append(message, byte(binlogFile), byte(binlogFile>>8), byte(binlogFile>>16), byte(binlogFile>>24))

	// 5. 数据类型 (insert=13/delete=15/update=14) - int8类型
	// 使用第一条数据的操作类型，与MongoDB保持一致
	var dataType int8
	if strings.Contains(firstData.Optype, "insert") {
		dataType = 13 // insert
	} else if strings.Contains(firstData.Optype, "update") {
		dataType = 14 // update
	} else if strings.Contains(firstData.Optype, "delete") {
		dataType = 15 // delete
	} else if strings.Contains(firstData.Optype, "index") {
		dataType = 13 // ES特有的index操作也映射为insert
	} else {
		dataType = 13 // 默认为插入
	}
	message = append(message, byte(dataType))

	// 6. db_name (len+str)
	dbNameBytes := []byte(dbName)
	dbNameLen := uint32(len(dbNameBytes))
	message = append(message, byte(dbNameLen), byte(dbNameLen>>8), byte(dbNameLen>>16), byte(dbNameLen>>24))
	message = append(message, dbNameBytes...)

	// 7. table_name (len+str)
	tbNameBytes := []byte(firstData.Index)
	tbNameLen := uint32(len(tbNameBytes))
	message = append(message, byte(tbNameLen), byte(tbNameLen>>8), byte(tbNameLen>>16), byte(tbNameLen>>24))
	message = append(message, tbNameBytes...)

	// 8. gtid 再次写入 (len+str)
	message = append(message, byte(gtidLen), byte(gtidLen>>8), byte(gtidLen>>16), byte(gtidLen>>24))
	message = append(message, gtidBytes...)

	// 9. sql (int32) - ES没有SQL，写入0
	sql := int32(0)
	message = append(message, byte(sql), byte(sql>>8), byte(sql>>16), byte(sql>>24))

	// 10. 列数量 (uint32) - 与MongoDB一致，使用3列: _id, data, optype
	columnNum := uint32(3)
	message = append(message, byte(columnNum), byte(columnNum>>8), byte(columnNum>>16), byte(columnNum>>24))

	// 11. 列名数组 (每个列名: len+str) - 与MongoDB保持一致的列名
	columnNames := []string{"__sys_obj_id__", "doc", "optype"}
	for _, colName := range columnNames {
		colNameBytes := []byte(colName)
		colNameLen := uint32(len(colNameBytes))
		message = append(message, byte(colNameLen), byte(colNameLen>>8), byte(colNameLen>>16), byte(colNameLen>>24))
		message = append(message, colNameBytes...)
	}

	// 12. 列属性数组 (每个属性: uint32, 都写0)
	for i := 0; i < 3; i++ {
		attr := uint32(0)
		message = append(message, byte(attr), byte(attr>>8), byte(attr>>16), byte(attr>>24))
	}

	// 13. 行数 (uint32, 发送所有行)
	message = append(message, byte(rowCount), byte(rowCount>>8), byte(rowCount>>16), byte(rowCount>>24))

	// 14. 所有行的内容
	for _, doc := range result.Data {
		// _id字段
		idBytes := []byte(fmt.Sprintf("'%s'", doc.ID))
		idLen := uint32(len(idBytes))
		message = append(message, byte(idLen), byte(idLen>>8), byte(idLen>>16), byte(idLen>>24))
		message = append(message, idBytes...)

		// doc字段 - 将整个文档数据转换为JSON字符串
		docBytes, err := json.Marshal(doc.Data)
		if err != nil {
			docBytes = []byte("{}")
		}
		docStr := fmt.Sprintf("'%s'", string(docBytes))
		docStrBytes := []byte(docStr)
		docStrLen := uint32(len(docStrBytes))
		message = append(message, byte(docStrLen), byte(docStrLen>>8), byte(docStrLen>>16), byte(docStrLen>>24))
		message = append(message, docStrBytes...)

		// optype字段
		optypeBytes := []byte(fmt.Sprintf("'%s'", doc.Optype))
		optypeLen := uint32(len(optypeBytes))
		message = append(message, byte(optypeLen), byte(optypeLen>>8), byte(optypeLen>>16), byte(optypeLen>>24))
		message = append(message, optypeBytes...)

		// 如果是更新操作，还需要添加旧值信息
		if dataType == 14 { // update
			// 旧的_id值
			oldIdBytes := []byte(doc.ID)
			oldIdLen := uint32(len(oldIdBytes))
			message = append(message, byte(oldIdLen), byte(oldIdLen>>8), byte(oldIdLen>>16), byte(oldIdLen>>24))
			message = append(message, oldIdBytes...)

			// 旧的data值 (简化处理，使用空对象)
			oldDataBytes := []byte("{}")
			oldDataLen := uint32(len(oldDataBytes))
			message = append(message, byte(oldDataLen), byte(oldDataLen>>8), byte(oldDataLen>>16), byte(oldDataLen>>24))
			message = append(message, oldDataBytes...)

			// 旧的optype值
			oldOptypeBytes := []byte("'old'")
			oldOptypeLen := uint32(len(oldOptypeBytes))
			message = append(message, byte(oldOptypeLen), byte(oldOptypeLen>>8), byte(oldOptypeLen>>16), byte(oldOptypeLen>>24))
			message = append(message, oldOptypeBytes...)
		}
	}

	// 发送完整的消息包
	logger.Info("发送完整的ES数据包(jobId=%s): 包含 %d 条记录，数据包大小=%d字节", jobId, rowCount, len(message))
	logger.Debug("任务ID=%s: 发送的GTID为: %s", jobId, string(gtidBytes))

	// 引入重试机制发送数据包
	maxSendRetries := 3
	for retry := 0; retry < maxSendRetries; retry++ {
		// 在发送消息时捕获错误
		err := protocol.SendMsg(conn, message)
		if err == nil {
			logger.Info("ES数据包发送完成(jobId=%s): 成功发送 %d 条记录", jobId, rowCount)
			break
		}

		// 连接错误，尝试重试
		if retry < maxSendRetries-1 {
			logger.Warn("数据包发送失败(尝试 %d/%d): %v，将在500ms后重试", retry+1, maxSendRetries, err)
			time.Sleep(500 * time.Millisecond)
			continue
		}

		// 最后一次尝试也失败
		return fmt.Errorf("发送消息失败(已尝试%d次): %v", maxSendRetries, err)
	}

	// 添加发送完成总结日志
	tm.Mu.Lock()
	var statusFirstSend bool = false
	if status, exists := tm.Statuses[jobId]; exists {
		statusFirstSend = status.FirstSend
	}
	tm.Mu.Unlock()

	if statusFirstSend {
		logger.Info("任务ID=%s：首次批量数据发送完成，本批次共发送%d条记录，将继续处理下一批数据，无需等待确认",
			jobId, rowCount)
	} else {
		logger.Info("任务ID=%s：非首次批量数据发送完成，本批次共发送%d条记录，进入永久等待确认状态，收到MRsprowmsg后才能继续，无超时限制",
			jobId, rowCount)
	}
	return nil
}

// 清理scroll上下文
func (tm *ESTaskManager) cleanupScroll(url, scrollID, username, password string) {
	client, err := tm.getESClient(url)
	if err != nil {
		logger.Error("获取ES客户端失败，无法清理scroll: %v", err)
		return
	}

	deleteScrollReq := map[string]interface{}{
		"scroll_id": scrollID,
	}
	requestBody, _ := json.Marshal(deleteScrollReq)
	requestURL := fmt.Sprintf("%s/_search/scroll", url)

	req, err := http.NewRequest("DELETE", requestURL, bytes.NewReader(requestBody))
	if err != nil {
		logger.Error("创建清理scroll请求失败: %v", err)
		return
	}

	req.Header.Set("Content-Type", "application/json")
	if username != "" && password != "" {
		req.SetBasicAuth(username, password)
	}

	resp, err := client.Do(req)
	if err != nil {
		logger.Error("清理scroll请求失败: %v", err)
		return
	}
	defer resp.Body.Close()

	logger.Debug("清理scroll完成，状态码: %d", resp.StatusCode)
}

// StopTask 停止任务
func (tm *ESTaskManager) StopTask(jobId string) bool {
	tm.Mu.Lock()
	defer tm.Mu.Unlock()

	cancel, exists := tm.Tasks[jobId]
	if exists {
		// 取消上下文
		cancel()
		// 从任务映射中删除
		delete(tm.Tasks, jobId)

		// 更新任务状态
		if status, ok := tm.Statuses[jobId]; ok {
			// 若任务有活跃的scroll ID，尝试清理
			if status.LastScrollID != "" {
				// 不需要锁，因为我们已经持有tm.Mu
				// 查找对应的客户端
				for url, client := range tm.clients {
					// 尝试清理scroll，但不要阻塞任务停止
					go func(url, scrollID, jobId string, client *http.Client) {
						// 尝试清理scroll上下文
						deleteScrollReq := map[string]interface{}{
							"scroll_id": scrollID,
						}
						requestBody, _ := json.Marshal(deleteScrollReq)
						requestURL := fmt.Sprintf("%s/_search/scroll", url)

						req, err := http.NewRequest("DELETE", requestURL, bytes.NewReader(requestBody))
						if err != nil {
							logger.Error("任务ID=%s: 创建清理scroll请求失败: %v", jobId, err)
							return
						}

						req.Header.Set("Content-Type", "application/json")
						// 不设置认证，因为不知道哪个连接是对应这个任务的

						// 执行清理
						resp, err := client.Do(req)
						if err != nil {
							logger.Error("任务ID=%s: 清理scroll请求失败: %v", jobId, err)
							return
						}
						defer resp.Body.Close()

						logger.Info("任务ID=%s: 清理scroll ID=%s完成，状态码: %d", jobId, scrollID, resp.StatusCode)
					}(url, status.LastScrollID, jobId, client)
				}
			}

			// 设置状态为非运行
			status.IsRunning = false
			status.Error = nil
			status.WaitingAck = false
			status.FirstSend = false
		}

		// 取消注册确认消息处理器
		tm.unregisterAckHandler(jobId)

		logger.Info("ES任务已成功停止并清理资源: %s", jobId)
		return true
	}
	return false
}

// GetTaskStatus 获取任务状态
func (tm *ESTaskManager) GetTaskStatus(jobId string) (*ESTaskStatus, bool) {
	tm.Mu.Lock()
	defer tm.Mu.Unlock()

	status, exists := tm.Statuses[jobId]
	return status, exists
}

// GetAllTaskIDs 获取所有活动的ES任务ID
func (tm *ESTaskManager) GetAllTaskIDs() []string {
	tm.Mu.Lock()
	defer tm.Mu.Unlock()

	var taskIDs []string
	for id := range tm.Tasks {
		taskIDs = append(taskIDs, id)
	}
	return taskIDs
}

// SyncESTask 外部调用的同步任务函数
func (tm *ESTaskManager) SyncESTask(ctx context.Context, es model.RecvData, conn net.Conn) {
	tm.syncESTask(ctx, es, conn)
}

// 辅助函数：将任意对象转换为JSON字符串
func toJsonString(data interface{}) string {
	jsonBytes, err := json.Marshal(data)
	if err != nil {
		return "{}"
	}
	return string(jsonBytes)
}

// 添加确认消息处理相关函数
// registerAckHandler 注册一个确认消息处理函数
func (tm *ESTaskManager) registerAckHandler(jobId string, handler func(byte, []byte)) {
	tm.ackHandlersMu.Lock()
	defer tm.ackHandlersMu.Unlock()
	tm.ackHandlers[jobId] = handler
}

// unregisterAckHandler 注销一个确认消息处理函数
func (tm *ESTaskManager) unregisterAckHandler(jobId string) {
	tm.ackHandlersMu.Lock()
	defer tm.ackHandlersMu.Unlock()
	delete(tm.ackHandlers, jobId)
}

// HandleAckMessage 处理来自main.go分发的确认消息
func (tm *ESTaskManager) HandleAckMessage(jobId string, msgType byte, msgData []byte) {
	tm.ackHandlersMu.RLock()
	handler, exists := tm.ackHandlers[jobId]
	tm.ackHandlersMu.RUnlock()

	if exists && handler != nil {
		handler(msgType, msgData)
	}
}

// getJsonNullOrString 辅助函数：将字符串转换为JSON格式的null或字符串
func getJsonNullOrString(s string) string {
	if s == "" {
		return "null"
	}
	return fmt.Sprintf(`"%s"`, s)
}
