/*-------------------------------------------------------------------------
 *
 * mysql_compress.h
 *		Kunlun Database MySQL protocol server side implementation.
 *		Compression related structs.
 *
 * Copyright (c) 2019-2022 ZettaDB inc. All rights reserved.
 *
 * This source code is licensed under Apache 2.0 License,
 * combined with Common Clause Condition 1.0, as detailed in the NOTICE file.
 *
 * IDENTIFICATION
 *	  src/include/libmysql/mysql_compress.h
 *
 *-------------------------------------------------------------------------
 */

#ifndef MY_COMPRESS_INCLUDED
#define MY_COMPRESS_INCLUDED

/* List of valid values for compression_algorithm */
typedef enum enum_compression_algorithm {
  MYSQL_UNCOMPRESSED = 1,
  MYSQL_ZLIB,
  MYSQL_ZSTD,
  MYSQL_INVALID
}enum_compression_algorithm ;

/**
  Compress context information. relating to zlib compression.
*/

typedef struct mysql_zlib_compress_context {
  /**
    Compression level to use in zlib compression.
  */
  unsigned int compression_level;
} mysql_zlib_compress_context;

typedef struct ZSTD_CCtx_s ZSTD_CCtx;
typedef struct ZSTD_DCtx_s ZSTD_DCtx;

/**
  Compress context information relating to zstd compression.
*/

typedef struct mysql_zstd_compress_context {
  /**
    Pointer to compressor context.
  */
  ZSTD_CCtx *cctx;
  /**
    Pointer to decompressor context.
  */
  ZSTD_DCtx *dctx;
  /**
    Compression level to use in zstd compression.
  */
  unsigned int compression_level;
} mysql_zstd_compress_context;

/**
  Compression context information.
  It encapsulate the context information based on compression method and
  presents a generic struct.
*/

typedef struct mysql_compress_context {
  enum enum_compression_algorithm algorithm;  ///< Compression algorithm name.
  union {
    mysql_zlib_compress_context zlib_ctx;  ///< Context information of zlib.
    mysql_zstd_compress_context zstd_ctx;  ///< Context information of zstd.
  } u;
} mysql_compress_context;

/**
  Get default compression level corresponding to a given compression method.

  @param algorithm Compression Method. Possible values are zlib or zstd.

  @return an unsigned int representing default compression level.
  6 is the default compression level for zlib and 3 is the
  default compression level for zstd.
*/

extern unsigned int mysql_default_compression_level(
    enum enum_compression_algorithm algorithm);

/**
  Initialize a compress context object to be associated with a NET object.

  @param cmp_ctx Pointer to compression context.
  @param algorithm Compression algorithm.
  @param compression_level Compression level corresponding to the compression
  algorithm.
*/

extern void mysql_compress_context_init(mysql_compress_context *cmp_ctx,
                                 enum enum_compression_algorithm algorithm,
                                 unsigned int compression_level);
/**
  Deinitialize the compression context allocated.

  @param mysql_compress_ctx Pointer to Compression context.
*/

extern void mysql_compress_context_deinit(mysql_compress_context *mysql_compress_ctx);
extern bool mysql_compress(mysql_compress_context *comp_ctx, unsigned char *packet,
	size_t *len, size_t *complen);
extern bool mysql_uncompress(mysql_compress_context *comp_ctx, unsigned char *packet, size_t len,
                   size_t *complen);
extern unsigned int mysql_default_compression_level(
    enum enum_compression_algorithm algorithm);


#define COMPRESSION_ALGORITHM_ZLIB "zlib"
#define COMPRESSION_ALGORITHM_ZSTD "zstd"
#define COMPRESSION_ALGORITHM_UNCOMPRESSED "uncompressed"
#define COMPRESSION_ALGORITHM_NAME_LENGTH_MAX 32
#define COMPRESSION_ALGORITHM_COUNT_MAX 3
#define COMPRESSION_ALGORITHM_NAME_BUFFER_SIZE                                 \
  ((COMPRESSION_ALGORITHM_NAME_LENGTH_MAX * COMPRESSION_ALGORITHM_COUNT_MAX) + \
   3)
#define PROTOCOL_COMPRESSION_DEFAULT_VALUE "zlib,zstd,uncompressed"
#define default_zstd_compression_level 3
struct List;
/* Helper functions to validate compression algorithm and level */
extern enum_compression_algorithm mysql_get_compression_algorithm(const char*name);
extern const char* mysql_get_compression_algorithm_name(enum_compression_algorithm);
extern struct List*mysql_parse_compression_algorithms(const char* name);
extern bool valid_zstd_compression_level(unsigned int level);
extern bool validate_compression_attributes(const char* algorithm_names);
#endif  // MY_COMPRESS_INCLUDED
