/*-------------------------------------------------------------------------
 *
 * mysql_socket.h
 *		Kunlun Database MySQL protocol server side implementation.
 *		A very thin wrapper of POSIX socket APIs.
 *
 * Copyright (c) 2019-2022 ZettaDB inc. All rights reserved.
 *
 * This source code is licensed under Apache 2.0 License,
 * combined with Common Clause Condition 1.0, as detailed in the NOTICE file.
 *
 * IDENTIFICATION
 *	  src/include/libmysql/mysql_socket.h
 *
 *-------------------------------------------------------------------------
 */

#ifndef MYSQL_SOCKET_H
#define MYSQL_SOCKET_H

#include <errno.h>
/* For strlen() */
#include <string.h>

/* For socket api */
#include <netinet/in.h>

#define SOCKBUF_T void

#include <unistd.h>
#include <fcntl.h>

#include "libmysql/mysql_com_sys.h"

#define MYSQL_START_SOCKET_WAIT(LOCKER, STATE, SOCKET, OP, COUNT)
#define MYSQL_END_SOCKET_WAIT(LOCKER, COUNT)
#define MYSQL_SOCKET_WAIT_VARIABLES(LOCKER, STATE)
#define MYSQL_SOCKET_SET_STATE(SOCKET, STATE)

#define INVALID_SOCKET -1

/**
  @defgroup psi_api_socket Socket Instrumentation (API)
  @ingroup psi_api
  @{
*/

/**
  MYSQL_SOCKET helper. Get socket descriptor.
  @param mysql_socket Instrumented socket
  @sa mysql_socket_setfd
*/
static inline my_socket mysql_socket_getfd(struct MYSQL_SOCKET mysql_socket) {
  return mysql_socket.fd;
}

/**
  MYSQL_SOCKET helper. Set socket descriptor.
  @param mysql_socket Instrumented socket
  @param fd Socket descriptor
  @sa mysql_socket_getfd
*/
static inline void mysql_socket_setfd(struct MYSQL_SOCKET *mysql_socket,
                                      my_socket fd) {
  if (likely(mysql_socket != NULL)) {
    mysql_socket->fd = fd;
  }
}




/** mysql_socket_socket */

static inline struct MYSQL_SOCKET mysql_socket_socket(
    int domain, int type, int protocol) {
  struct MYSQL_SOCKET mysql_socket = {INVALID_SOCKET};
  mysql_socket.fd = socket(domain, type, protocol);

  return mysql_socket;
}

/** mysql_socket_bind */

static inline int mysql_socket_bind(
    MYSQL_SOCKET mysql_socket, const struct sockaddr *addr, socklen_t len) {
  int result;

  result = bind(mysql_socket.fd, addr, len);

  return result;
}

/** mysql_socket_getsockname */

static inline int mysql_socket_getsockname(
    MYSQL_SOCKET mysql_socket, struct sockaddr *addr, socklen_t *len) {
  int result;

  result = getsockname(mysql_socket.fd, addr, len);

  return result;
}

/** mysql_socket_connect */

static inline int mysql_socket_connect(
    struct MYSQL_SOCKET mysql_socket, const struct sockaddr *addr, socklen_t len) {
  int result;

  result = connect(mysql_socket.fd, addr, len);

  return result;
}

/** mysql_socket_getpeername */

static inline int mysql_socket_getpeername(
    struct MYSQL_SOCKET mysql_socket, struct sockaddr *addr, socklen_t *len) {
  int result;

  result = getpeername(mysql_socket.fd, addr, len);

  return result;
}

/** mysql_socket_send */

static inline ssize_t mysql_socket_send(
    struct MYSQL_SOCKET mysql_socket, const SOCKBUF_T *buf, size_t n, int flags) {
  ssize_t result;

  result = send(mysql_socket.fd, buf, n, flags);

  return result;
}

/** mysql_socket_recv */

static inline ssize_t mysql_socket_recv(
    struct MYSQL_SOCKET mysql_socket, SOCKBUF_T *buf, size_t n, int flags) {
  ssize_t result;

  result = recv(mysql_socket.fd, buf, n, flags);

  return result;
}

/** mysql_socket_sendto */

static inline ssize_t mysql_socket_sendto(
    struct MYSQL_SOCKET mysql_socket, const SOCKBUF_T *buf, size_t n, int flags,
    const struct sockaddr *addr, socklen_t addr_len) {
  ssize_t result;

  result =
      sendto(mysql_socket.fd, buf, n, flags, addr, addr_len);

  return result;
}

/** mysql_socket_recvfrom */

static inline ssize_t mysql_socket_recvfrom(
    struct MYSQL_SOCKET mysql_socket, SOCKBUF_T *buf, size_t n, int flags,
    struct sockaddr *addr, socklen_t *addr_len) {
  ssize_t result;

  result =
      recvfrom(mysql_socket.fd, buf, n, flags, addr, addr_len);

  return result;
}

/** mysql_socket_getsockopt */

static inline int mysql_socket_getsockopt(
    struct MYSQL_SOCKET mysql_socket, int level, int optname, SOCKBUF_T *optval,
    socklen_t *optlen) {
  int result;

  result = getsockopt(mysql_socket.fd, level, optname, optval, optlen);

  return result;
}

/** mysql_socket_setsockopt */

static inline int mysql_socket_setsockopt(
    struct MYSQL_SOCKET mysql_socket, int level, int optname, const SOCKBUF_T *optval,
    socklen_t optlen) {
  int result;

  result = setsockopt(mysql_socket.fd, level, optname, optval, optlen);

  return result;
}

/** set_socket_nonblock */
static inline int set_socket_nonblock(my_socket fd) {
  int ret = 0;
  {
    int fd_flags;
    fd_flags = fcntl(fd, F_GETFL, 0);
    if (fd_flags < 0) {
      return errno;
    }
#if defined(O_NONBLOCK)
    fd_flags |= O_NONBLOCK;
#elif defined(O_NDELAY)
    fd_flags |= O_NDELAY;
#elif defined(O_FNDELAY)
    fd_flags |= O_FNDELAY;
#else
#error "No definition of non-blocking flag found."
#endif /* O_NONBLOCK */
    if (fcntl(fd, F_SETFL, fd_flags) == -1) {
      ret = errno;
    }
  }
  return ret;
}

/** mysql_socket_set_nonblocking */

static inline int inline_mysql_sock_set_nonblocking(
    struct MYSQL_SOCKET mysql_socket) {
  int result = 0;


  result = set_socket_nonblock(mysql_socket.fd);

  return result;
}

/** mysql_socket_listen */

static inline int mysql_socket_listen(
    struct MYSQL_SOCKET mysql_socket, int backlog) {
  int result;

  result = listen(mysql_socket.fd, backlog);

  return result;
}

/** mysql_socket_accept */

static inline MYSQL_SOCKET mysql_socket_accept(
    MYSQL_SOCKET socket_listen, struct sockaddr *addr, socklen_t *addr_len) {
  MYSQL_SOCKET socket_accept = {INVALID_SOCKET};
  socklen_t addr_length = (addr_len != NULL) ? *addr_len : 0;

  socket_accept.fd = accept(socket_listen.fd, addr, &addr_length);

  return socket_accept;
}

/** mysql_socket_close */

static inline int mysql_socket_close(
    struct MYSQL_SOCKET mysql_socket) {
  int result;

  result = closesocket(mysql_socket.fd);

  return result;
}

/** mysql_socket_shutdown */

static inline int mysql_socket_shutdown(
    struct MYSQL_SOCKET mysql_socket, int how) {
  int result;
    result = shutdown(mysql_socket.fd, how);

  return result;
}

#endif
