/*-------------------------------------------------------------------------
 *
 * big_endian.h
 *		Kunlun Database MySQL protocol server side implementation.
 *		Integer read/write functions for big endian architectures.
 *
 * Copyright (c) 2019-2022 ZettaDB inc. All rights reserved.
 *
 * This source code is licensed under Apache 2.0 License,
 * combined with Common Clause Condition 1.0, as detailed in the NOTICE file.
 *
 * IDENTIFICATION
 *	  src/include/libmysql/big_endian.h
 *
 *-------------------------------------------------------------------------
 */

#ifndef MY_BYTEORDER_INCLUDED
#error This file should never be #included directly; use my_byteorder.h.
#endif

#include <string.h>

static inline int16_t read_sint2(const unsigned char *A) {
  return (int16_t)(((int16_t)(A[0])) + ((int16_t)(A[1]) << 8));
}

static inline int32_t read_sint4(const unsigned char *A) {
  return (int32_t)(((int32_t)(A[0])) + (((int32_t)(A[1]) << 8)) +
                 (((int32_t)(A[2]) << 16)) + (((int32_t)(A[3]) << 24)));
}

static inline uint16_t read_uint2(const unsigned char *A) {
  return (uint16_t)(((uint16_t)(A[0])) + ((uint16_t)(A[1]) << 8));
}

static inline uint32_t read_uint4(const unsigned char *A) {
  return (uint32_t)(((uint32_t)(A[0])) + (((uint32_t)(A[1])) << 8) +
                  (((uint32_t)(A[2])) << 16) + (((uint32_t)(A[3])) << 24));
}

static inline unsigned long long read_uint8(const unsigned char *A) {
  return ((unsigned long long)(((uint32_t)(A[0])) + (((uint32_t)(A[1])) << 8) +
                      (((uint32_t)(A[2])) << 16) + (((uint32_t)(A[3])) << 24)) +
          (((unsigned long long)(((uint32_t)(A[4])) + (((uint32_t)(A[5])) << 8) +
                        (((uint32_t)(A[6])) << 16) + (((uint32_t)(A[7])) << 24)))
           << 32));
}

static inline longlong read_sint8(const unsigned char *A) {
  return (longlong)uint8korr(A);
}

static inline void write_int2(unsigned char *T, uint16_t A) {
  unsigned int def_temp = A;
  *(T) = (unsigned char)(def_temp);
  *(T + 1) = (unsigned char)(def_temp >> 8);
}

static inline void write_int4(unsigned char *T, uint32_t A) {
  *(T) = (unsigned char)(A);
  *(T + 1) = (unsigned char)(A >> 8);
  *(T + 2) = (unsigned char)(A >> 16);
  *(T + 3) = (unsigned char)(A >> 24);
}

static inline void write_int7(unsigned char *T, unsigned long long A) {
  *(T) = (unsigned char)(A);
  *(T + 1) = (unsigned char)(A >> 8);
  *(T + 2) = (unsigned char)(A >> 16);
  *(T + 3) = (unsigned char)(A >> 24);
  *(T + 4) = (unsigned char)(A >> 32);
  *(T + 5) = (unsigned char)(A >> 40);
  *(T + 6) = (unsigned char)(A >> 48);
}

static inline void write_int8(unsigned char *T, unsigned long long A) {
  unsigned int def_temp = (unsigned int )A, def_temp2 = (unsigned int )(A >> 32);
  write_int4(T, def_temp);
  write_int4(T + 4, def_temp2);
}

/*
  Data in big-endian format.
*/
static inline void write_float4(unsigned char *T, float A) {
  *(T) = ((unsigned char *)&A)[3];
  *((T) + 1) = (char)((unsigned char *)&A)[2];
  *((T) + 2) = (char)((unsigned char *)&A)[1];
  *((T) + 3) = (char)((unsigned char *)&A)[0];
}

static inline float read_float4(const unsigned char *M) {
  float def_temp;
  ((unsigned char *)&def_temp)[0] = (M)[3];
  ((unsigned char *)&def_temp)[1] = (M)[2];
  ((unsigned char *)&def_temp)[2] = (M)[1];
  ((unsigned char *)&def_temp)[3] = (M)[0];
  return def_temp;
}

static inline void write_float8(unsigned char *T, double V) {
  *(T) = ((unsigned char *)&V)[7];
  *((T) + 1) = (char)((unsigned char *)&V)[6];
  *((T) + 2) = (char)((unsigned char *)&V)[5];
  *((T) + 3) = (char)((unsigned char *)&V)[4];
  *((T) + 4) = (char)((unsigned char *)&V)[3];
  *((T) + 5) = (char)((unsigned char *)&V)[2];
  *((T) + 6) = (char)((unsigned char *)&V)[1];
  *((T) + 7) = (char)((unsigned char *)&V)[0];
}

static inline double read_float8(const unsigned char *M) {
  double def_temp;
  ((unsigned char *)&def_temp)[0] = (M)[7];
  ((unsigned char *)&def_temp)[1] = (M)[6];
  ((unsigned char *)&def_temp)[2] = (M)[5];
  ((unsigned char *)&def_temp)[3] = (M)[4];
  ((unsigned char *)&def_temp)[4] = (M)[3];
  ((unsigned char *)&def_temp)[5] = (M)[2];
  ((unsigned char *)&def_temp)[6] = (M)[1];
  ((unsigned char *)&def_temp)[7] = (M)[0];
  return def_temp;
}
