/*-------------------------------------------------------------------------
 *
 * my_stacktrace.h
 *		stack tracing and print
 *
 * Copyright (c) 2019-2022 ZettaDB inc. All rights reserved.
 *
 * This source code is licensed under Apache 2.0 License,
 * combined with Common Clause Condition 1.0, as detailed in the NOTICE file.
 *
 * src/include/tcop/my_stacktrace.h
 *
 *-------------------------------------------------------------------------
 */
#ifndef MY_STACKTRACE_INCLUDED
#define MY_STACKTRACE_INCLUDED


#include <stddef.h>
#include <sys/types.h>

#ifdef _WIN32
#include <windows.h>
#endif

#include "my_inttypes.h"
#include "port_config.h"

extern int test_flags;
#define TEST_CORE_ON_SIGNAL 0x1 /**< Give core if signal */
#define TEST_NO_STACKTRACE 0x2
#define TEST_SIGINT 0x4 /**< Allow sigint on threads */

/*
  HAVE_BACKTRACE - Linux, FreeBSD, OSX, Solaris
  _WIN32 - Windows
*/
#if defined(HAVE_BACKTRACE) || defined(_WIN32)
#define HAVE_STACKTRACE 1
void pg_init_stacktrace(void);
void pg_print_stacktrace(const uchar *stack_bottom, ulong thread_stack);
void pg_sane_puts_stderr(const char *val, size_t max_len);
char* get_stacktrace_str(const uchar *stack_bottom, ulong thread_stack);

#ifdef _WIN32
void my_set_exception_pointers(EXCEPTION_POINTERS *ep);
void my_create_minidump(const char *name, HANDLE process, DWORD pid);
#endif
#endif /* HAVE_BACKTRACE || _WIN32 */

void pg_write_core(int sig);

/**
  Async-signal-safe utility functions used by signal handler routines.
  Declared here in order to unit-test them.
  These are not general-purpose, but tailored to the signal handling routines.
*/
/**
  Converts a longlong value to string.
  @param   base 10 for decimal, 16 for hex values (0..9a..f)
  @param   val  The value to convert
  @param   buf  Assumed to point to the *end* of the buffer.
  @returns Pointer to the first character of the converted string.
		   Negative values:
		   for base-10 the return string will be prepended with '-'
		   for base-16 the return string will contain 16 characters
  Implemented with simplicity, and async-signal-safety in mind.
*/
char *pg_sane_itoa(int base, longlong val, char *buf);

/**
  Converts a ulonglong value to string.
  @param   base 10 for decimal, 16 for hex values (0..9a..f)
  @param   val  The value to convert
  @param   buf  Assumed to point to the *end* of the buffer.
  @returns Pointer to the first character of the converted string.
  Implemented with simplicity, and async-signal-safety in mind.
*/
char *pg_sane_utoa(int base, ulonglong val, char *buf);

/**
  A (very) limited version of snprintf.
  @param   to   Destination buffer.
  @param   n    Size of destination buffer.
  @param   fmt  printf() style format string.
  @returns Number of bytes written, including terminating '\0'
  Supports 'd' 'i' 'u' 'x' 'p' 's' conversion.
  Supports 'l' and 'll' modifiers for integral types.
  Does not support any width/precision.
  Implemented with simplicity, and async-signal-safety in mind.
*/
size_t pg_sane_snprintf(char *to, size_t n, const char *fmt, ...)
	MY_ATTRIBUTE((format(printf, 3, 4)));

/**
  A (very) limited version of snprintf, which writes the result to STDERR.
  @sa pg_sane_snprintf
  Implemented with simplicity, and async-signal-safety in mind.
  @note Has an internal buffer capacity of 512 bytes,
  which should suffice for our signal handling routines.
*/
size_t pg_sane_printf_stderr(const char *fmt, ...)
	MY_ATTRIBUTE((format(printf, 1, 2)));

/**
  Writes up to count bytes from buffer to STDERR.
  Implemented with simplicity, and async-signal-safety in mind.
  @param   buf   Buffer containing data to be written.
  @param   count Number of bytes to write.
  @returns Number of bytes written.
*/
size_t pg_write_stderr(const void *buf, size_t count);

/**
  Writes system time to STDERR without allocating new memory.
*/
void pg_sane_print_system_time(void);
void handle_fatal_signal(int sig);
#endif  // MY_STACKTRACE_INCLUDED
