/*-------------------------------------------------------------------------
 *
 * mysql_time.h
 *		Kunlun Database MySQL protocol server side implementation.
 *		MySQL date, time and timestamp internal representation.
 *
 * Copyright (c) 2019-2022 ZettaDB inc. All rights reserved.
 *
 * This source code is licensed under Apache 2.0 License,
 * combined with Common Clause Condition 1.0, as detailed in the NOTICE file.
 *
 * IDENTIFICATION
 *	  src/include/libmysql/mysql_time.h
 *
 *-------------------------------------------------------------------------
 */

#ifndef MYSQL_TIME_H
#define MYSQL_TIME_H

/*
  defines which fields of MYSQL_TIME are filled and valid, depending on
  different types of date and/or time values.
*/
typedef enum enum_mysql_timestamp_type {
  MYSQL_TIMESTAMP_NONE = -2,
  MYSQL_TIMESTAMP_ERROR = -1,

  /// Stores year, month and day components.
  MYSQL_TIMESTAMP_DATE = 0,

  /**
    Stores all date and time components.
    Value is in UTC for `TIMESTAMP` type.
    Value is in local time zone for `DATETIME` type.
  */
  MYSQL_TIMESTAMP_DATETIME = 1,

  /// Stores hour, minute, second and microsecond.
  MYSQL_TIMESTAMP_TIME = 2,

  /**
    A temporary type for `DATETIME` or `TIMESTAMP` types equipped with time
    zone information. After the time zone information is reconciled, the type is
    converted to MYSQL_TIMESTAMP_DATETIME.
  */
  MYSQL_TIMESTAMP_DATETIME_TZ = 3
} enum_mysql_timestamp_type;

/*
  Structure which is used to represent datetime values inside MySQL.

  We assume that values in this structure are normalized, i.e. year <= 9999,
  month <= 12, day <= 31, hour <= 23, hour <= 59, hour <= 59. Many functions
  in server such as my_system_gmt_sec() or make_time() family of functions
  rely on this (actually now usage of make_*() family relies on a bit weaker
  restriction). Also functions that produce MYSQL_TIME as result ensure this.
  There is one exception to this rule though if this structure holds time
  value (time_type == MYSQL_TIMESTAMP_TIME) days and hour member can hold
  bigger values.
*/
typedef struct MYSQL_TIME {
  unsigned int year, month, day, hour, minute, second;
  unsigned long second_part; /**< microseconds */
  bool neg;
  enum enum_mysql_timestamp_type time_type;
  /// The time zone displacement, specified in seconds.
  int time_zone_displacement;
} MYSQL_TIME;

inline static void
set_zero_time(MYSQL_TIME *tm, enum enum_mysql_timestamp_type time_type) {
  memset(tm, 0, sizeof(*tm));
  tm->time_type = time_type;
}

#endif /* MYSQL_TIME_H */
