/*-------------------------------------------------------------------------
 *
 * remotenodes.h
 *
 * Copyright (c) 2019-2021 ZettaDB inc. All rights reserved.
 *
 * This source code is licensed under Apache 2.0 License,
 * combined with Common Clause Condition 1.0, as detailed in the NOTICE file.
 *
 * IDENTIFICATION
 * 		src/include/nodes/remotenodes.h
 *
 *-------------------------------------------------------------------------
*/
#ifndef REMOTE_NODES_H
#define REMOTE_NODES_H

#include "nodes/nodes.h"
#include "lib/stringinfo.h"
//#include "nodes/execnodes.h"
struct EState;
struct AppendState;
struct RemoteScanState;
struct SortState;
struct Sort;
struct MergeAppendState;
struct LimitState;
struct UniqueState;
struct PlannedStmt;

#define REMOTE_JOIN_RELID 65535

enum PlanBranchType{ BT_NONE, OUTER = 1, INNER, OUTER_AND_INNER };

typedef struct Remote_OLD_state
{
	/*
	  Whether all QUALs have been pushed down. To push down DISTINCT, all
	  QUAL exprs must be pushed down, and this is an optional requirement
	  for ORDER BY.
	*/
	bool all_quals_down;

	/*
	  If there are QUAL exprs not pushed down and left in computing node to
	  execute row filter, do we push down ORDER BY or not.
	  If true, shard may sort a lot which will later be abandoned, if only a
	  small portion of them really qualify.
	*/
	bool push_down_orderby_comp_quals;

	/*
	  All projection nodes are pushed down. Required To push down DISTINCT.
	*/
	bool all_projs_down;

	/*
	  Traversing a subtree with Append as root. Don't count num_rels when so.
	*/
	bool under_append;

	/*
	  whether DISTINCT has been pushed down.
	  Set by RemoteScanState/RemoteJoinState code.
	*/
	bool unique_down;

	/*
	  Whether ORDER BY is pushed down.
	  Set by RemoteScanState/RemoteJoinState code.
	*/
	bool orderby_down;

	/*
	  whether the LIMIT&OFFSET clauses are pushed down.
	  Set by RemoteScanState/RemoteJoinState code.
	*/
	bool limits_down;

	enum PlanBranchType branch_type;

	/*
	  NO. of standalone relations accessed.
	*/
	uint16_t num_rels;

	/*
	  NO. of partitioned table counts 1 regardless of its leaf partitions.
	*/
	uint16_t num_appends;

	/*
	  The (first) standalone relations accessed. If there are more, the rest
	  are not recorded because we can not push down any of OLD nodes.
	  When we can push a join down, we may need to add a RemoteJoinState
	  member and a num_joins member to track this fact.
	*/
	struct RemoteScanState *rss;

	/*
	  The (first) partitioned relation accessed. If there are more, the rest
	  are not recorded because we can not push down any of OLD nodes.
	  OLD nodes of one partitioned table can be pushed down, with proper work
	  done at computing nodes.
	*/
	struct AppendState *appends;
	/*
	  The targetlist of the Append node may be used later when replacing the
	  Sort-Append edge with an MergeAppend node, so need to cache it in
	  InitAppend before initing its RemoteScan children.

	  The Sort/Append node's original targetlist's Var nodes will refer to
	  RemoteScan's result type after RemoteScan inited, such tl can't be used
	  again to init another set of RemoteScan nodes which will be used as the
	  MergeAppend node's children.
	*/
	struct List *append_targetlist;
	/*
	  List of targetlist ptrs for each RS node of the Append node.
	*/
	struct List *rs_tls;

	/*
	  -1 if no such clause.
	*/
	int64_t	nlimits;
	int64_t	noffsets;

	/*
	  Valid if currently working on a descendant of a Limit node.
	*/
	struct	LimitState*limit_state;

	/*
	  Valid if currently working on a descendant of a Sort node.
	*/
	struct	SortState*sort_state;

	/*
	  Valid if currently working on a descendant of a Unique node.
	*/
	struct UniqueState *unique_state;

	/*
	  Valid if currently working on a descendant of a JoinState node.
	*/
	struct JoinState *join_state;
	struct Remote_OLD_state *next;
} Remote_OLD_state;

struct SortSupportData;
typedef struct AggColMeta
{
	Oid aggfnoid;
	int rpresno;
	// below fields set at start of execution
	// used by max/min
	struct SortSupportData *sortKey;
	// add func for sum()
	FmgrInfo *flinfo_add;
	FunctionCallInfoData *fcinfo_add;
} AggColMeta;
extern void reset_remote_OLD_state(struct EState *estate, bool reset_all);
extern void free_remote_OLD_states(struct EState *estate);
extern struct MergeAppendState*
merge_append_state_from_sort_append_states(struct EState *estate, int eflags, struct SortState*ss);
extern bool
expression_tree_modifier(Node **node,
					   bool (*handler) (),
					   bool (*walker) (),
					   void *context);
inline static bool is_agg_count(Oid funcid)
{
	return (funcid ==  2803 || funcid == 2147);
}

extern void pushdown_agg_tl_grouping_qual(Plan *remote_plan, List *tlist,
	List *havingQuals, int*num_aggrefs, Index *raw_groupcols);
extern bool
is_mysql_agg_func(Aggref *aggref, const char **pfuncname, bool *pdistince,
	bool *porderby);
extern void convert_aggrefs_to_funcexprs(List *aggref_pptrs, Index scanrelid);
extern bool is_mysql_agg_funcid(Oid funcid, List *args);
#endif // !REMOTE_NODES_H
