/*-------------------------------------------------------------------------
 *
 * remote_dml.h
 *	  POSTGRES remote access method DML statements processing code.
 *
 *
 * Copyright (c) 2019-2021 ZettaDB inc. All rights reserved.
 *
 * This source code is licensed under Apache 2.0 License,
 * combined with Common Clause Condition 1.0, as detailed in the NOTICE file.
 *
 * src/include/access/remote_dml.h
 *
 *-------------------------------------------------------------------------
 */
#ifndef REMOTE_DML_H
#define REMOTE_DML_H
#include "utils/relcache.h"
#include "nodes/nodes.h"
#include "nodes/pg_list.h"
#include "nodes/primnodes.h"
#include "nodes/execnodes.h"

typedef struct VarPickerCtx
{
	/*
	 * Target columns of target scanned relation that we will need to execute
	 * the query. Iff !has_alien_cols and nvars > 0 and snprint_expr() can serialize the expr
	 * will we use the serialized text as target, otherwise if (nvars > 0) we use the Vars in
	 * target_cols as target list; otherwise the target isn't sent to remote.
	 * */
	Var **target_cols;
	MemoryContext mctx;
	int nvars;
	int nvar_buf;

	/*
	  The expr has a agg func call without column reference, e.g. 
	  count(*), count(1), sum(1), etc. Such a call has no args list and
	  no vars but it should be serialized.
	*/
	bool has_agg;
	/*
	 * If there are no vars to reference target rel, and all Nodes of the
	 * targetEntry->expr are locally
	 * evaluable, then the target can be processed locally without sending it
	 * to remote.
	 *
	 * Currently only Const and SQLValueFunction are known to be locally evaluable.
	 * */
	int local_evaluables;

	/*
	 * There are base/leaf primitive nodes that we can't handle, have to
	 * return error to client in this case.
	 * */
	int local_unevaluables;

	/*
	  PlaceHolderVar::phexpr found from an expression, we want to simply
	  extract such PHV's exprs.
	*/
	List *phv_expr_list;
	
	/* List of phvs which assigned alias and should be treated as normal vars */
	List *nonrecursive_exprs;

	/* Do not recursively walk with aggregate function */
	bool nonrecursive_agg;

	List *aggref_pptr_list;
} VarPickerCtx;

struct RemotePrintExprContext;

/*
 * Used to construct sql for pushdownable update&&delete 
 */
typedef struct RemoteUD
{
	ModifyTableState *mtstate;
	ResultRelInfo *relinfo;
	int planIndex;
	int index;		 /* Index of current result relation */
	Relation rel;	 /* Current result relation */
	int rti;		 /* RTE number of current result relation */
	List *tlist;	 /* Update target list */
	List *qual;		 /* Where clause of current result relation */
	PlanState *from; /* From clause */
	List *sortlist;	 /* Sort clause */
	List *sortop;	 /* Sort operator */
	List *sortnullfirst; 
	int64_t limit;	 /* Limit clause */
	List *rellist;	 /* All of the result relation */
	List *quallist;	 /* All of the qual of result relation */
} RemoteUD;

#define RemoteUDRelations(p) \
	(list_length((p)->rellist))

extern void RemoteUDSetup(PlanState *planstate, RemoteUD *remote_updel);
extern bool RemoteUDNext(RemoteUD *remote_updel, struct RemotePrintExprContext *rpec, StringInfo sql, Oid *shardid);
extern bool RemoteUDEOF(RemoteUD *remote_updel);

extern int remote_param_fetch_threshold;

extern bool CheckPartitionKeyModified(Relation,  Index attrno);

extern bool CanPushdownRemoteUD(PlanState *state, List *unused_tl, int *leafs, const char **reaseon);
extern bool var_picker(Node **ppnode, struct VarPickerCtx*ctx);
extern bool var_picker_hdlr(Node **ppnode, struct VarPickerCtx*ctx);
extern void reset_var_picker_ctx(VarPickerCtx *vpc);
#endif // !REMOTE_DML_H
