/*-------------------------------------------------------------------------
 *
 * sharding.h
 *	  definitions of types, and declarations of functions, that are used in
 *	  table sharding functionality.
 *
 *
 * Copyright (c) 2019-2021 ZettaDB inc. All rights reserved.
 *
 * This source code is licensed under Apache 2.0 License,
 * combined with Common Clause Condition 1.0, as detailed in the NOTICE file.
 *
 * src/include/sharding/sharding.h
 *
 *-------------------------------------------------------------------------
 */
#ifndef SHARDING_H
#define SHARDING_H

#include "catalog/pg_shard_node.h"
#include "catalog/pg_shard.h"
#include "utils/hsearch.h"
#include "utils/rel.h" // Relation


#define MAX_NODES_PER_SHARD 7

/*
 * If one modifies this number, do modify ERRORDATA_STACK_SIZE to the
 * same value.
 * */
#define MAX_SHARDS 512
#define METASHARD_REQ 0x1
#define STORAGE_SHARD_REQ 0x2
#define MAX_STORAGE_NODES 65535 // can NOT be greater than UINT16_MAX
#define MAX_CONNS_STORAGE_NODE_PER_SESSION 255

extern Oid Invalid_shard_node_id;
extern Oid First_shard_node_id;
extern Oid Invalid_shard_id;
extern Oid First_shard_id;

/*
 * A shard has one master node and multiple slaves nodes. they contain
 * identical data, and each shard node is stored as a mysql db instance
 * on a computer server.
 * A slave node replicates binlogs from master node to keep data in sync.
 * A computing node mostly sent queries to master node to execute, but it
 * can also send read only queries to slave nodes to execute.
 * */
typedef struct Shard_node_t
{
	NameData user_name;

	uint32 shard_id; /* The owner shard, references pg_shard(id). BKI can't specify
					 foreign key though. */
	uint32 svr_node_id;
	uint16_t port;
	/*
	 * When dispatching read only select stmts to a slave, choose the one with
	 * max ro_weight of a shard.
	 */
	int16_t ro_weight;
	Oid id;
	char *hostaddr;
	/*
	 * This field can be very long because we want to store encrypted password
	 * instead of plain text ones. pg_shard_node's 'passwd' column data type is
	 * text, so is the 'passwd' column of shard_nodes table on meta server.
	 * So we don't use char[N].
	 * */
	char *passwd;
} Shard_node_t;

extern bool ShardExists(Oid shardid, bool sync_onfail);
extern List *GetAllShardIds(bool fence);
extern List *GetDatabaseShardIds(bool fence);
extern Oid GetRandomShard(void);
extern Oid GetDatabaseRandomShard(void);
extern int NumberOfShards(void);
extern Oid GetShardId(Name name);
extern bool GetShardName(Oid shardid, NameData *name);
extern Oid GetShardMasterNodeId(Oid shardid);
extern List *GetShardAllNodes(Oid shardid);
extern List *GetAllShardNodes(void);
extern Oid FindShardNode(Oid shardid, const char *ip, uint16_t port);
extern bool FindCachedShardNode(Oid shardid, Oid nodeid, Shard_node_t *node) ;
extern Oid FindBestShardForTable(int which, Relation rel);
extern bool SetNewShardMaster(Oid shardid, Oid master_nodeid);

#endif /* !SHARDING_H */
