/*-------------------------------------------------------------------------
 *
 * stmt_hdl.h
 *
 *
 * Copyright (c) 2019-2022 ZettaDB inc. All rights reserved.
 *
 * This source code is licensed under Apache 2.0 License,
 * combined with Common Clause Condition 1.0, as detailed in the NOTICE file.
 *
 * src/include/sharding/stmt_hdl.h
 *
 *-------------------------------------------------------------------------
 */
#ifndef STMT_HDL_H
#define STMT_HDL_H

#include "executor/rwconfict_check.h"
#include "nodes/nodes.h"
#include "sharding/mat_cache.h"
#include "sharding/mysql/server/private/sql_cmd.h"

extern int64_t remote_rows_read;
extern int64_t remote_rows_affected;

typedef char **MYSQL_ROW;
struct AsyncStmtInfo;
struct st_mysql_res;
struct st_mysql_stmt;

typedef struct MysqlDeserializeInfo MysqlDeserializeInfo;

typedef Datum (*DeserializeFun)(char *str, size_t len, int type, bool *isnull, void *param);

/**
 *  Contain the informations used to deserialize mysql result into pg datum
 */
struct MysqlDeserializeInfo
{
	Oid typid;		/* The expected type of output datum*/
	void *param; 	      /* The private param used by deserialize function */
	DeserializeFun func; /* Deserialize function */
};

/* The parameters for binary query */
typedef struct
{
	Oid typid;
	bool null;
	Datum value;
} ShardStmtParam;

typedef struct StmtHandle
{
	struct AsyncStmtInfo *asi;
	MemoryContext mem;
	SubTransactionId subxactid;
	int refcount;
	char *stmt;
	char *stmt_original; // the original sql (only for not owned)
	size_t stmt_len;
	bool owns_stmt_mem;
	CmdType cmd;
	enum enum_sql_command sqlcom;
	bool is_dml_write;
	bool support_rewind;
	bool start_of_txn;
	bool update_snapshot;

	/* Fields used by datum deserialization */
	MysqlDeserializeInfo *deserialize_info;
	int nr_deserialize_info;

	int status;
	int status_req;
	bool first_packet;  // true if the first packet recved
	bool prepared; // true if the stmt was prepared in binary protocal
	bool binded; // true if the parameters was binded in binary protocal
	bool executed; // true if the stmt was executed in binary protocal
	bool fetch;			// true if the stmt is going to fetch results?
	bool nextres;		// true if need get the next result
	bool finished;		// true if finish read from socket
	bool cancel;
	bool start_free;
	bool read_cache;	// true if should read from matcache
	
	int ignore_errno;
	bool got_ignored_error;
	uint32_t affected_rows;
	uint32_t warnings_count;

	struct st_mysql_res *res;
	size_t *lengths;
	size_t *bind_lengths;
	enum enum_field_types *types;
	char **fieldnames;
	MYSQL_ROW row;
	int field_count;
	
	/* binary protocal specific fields */
	bool binary_protocol;
	bool cursor_mode;
	bool cursor_suspend; /* prefetched rows consumed */
	uint32 hashkey;
	struct st_mysql_stmt *prepared_stmt;
	char *nulls;
	char **result_buffer;
	List *param_list;
	bool data_truncated;

	/* Used by materialize */
	StringInfoData read_buff;
	StringInfoData buff;
	MatCache *cache;
	/* True if reset the next start position to read */
	bool reset_cache_pos;
	/* Read position of the cache */
	MatCachePos cache_pos;

	/* read/write conflict checker */
	bool conflict;
	RemoteRWCheckContext *rwcheck_context;
} StmtHandle;

/**
 * @brief Stmthandle with epoch information  
 */
typedef struct
{
	StmtHandle *handle;
	int32_t epoch;
} StmtSafeHandle;

#define RAW_HANDLE(h) (h.handle)

#define INVALID_STMT_HANLE ((StmtSafeHandle){NULL, 0})

static inline bool stmt_handle_valid(StmtSafeHandle handle) { return handle.handle != NULL; }

extern bool handle_expired(StmtSafeHandle handle);

/**
 * @brief Sets the ignored errors for the next statement to execute;
 */
extern int set_stmt_ignored_error(int mysql_errno);

/**
 * @brief Check if mysql binary protocol can safely bind the pg type
 */
extern bool can_bind_as_bin_param(Oid pgtypeid);

/**
 * @brief Set the remote rwcheck context to do conflict check at the begin or end of statements
 */
extern void set_remote_rwcheck_context(StmtSafeHandle handle, RemoteRWCheckContext *context);

/**
 * @brief Append 'stmt' into asi's job queue. 'stmt' will be sent later when its turn comes.
 * 
 * @return A handle of the stmt. t can be used by the caller to receive results from stmt,
 *  which must be released when used up.
 *  
 * NOTE: Handles are automatically released after the (sub)transaction ends, so don't share
 *  handles across (sub)transactions.
 *  If 'ownsit' set to true, caller transfers the responsibility of freeing the memory of stmt to
 *  this function, which automatically frees the memory after the stmt is sent. In this case, 
 *  the function restricts that stmt must be allocated from TopMemoryContext. 
 */
extern StmtSafeHandle send_stmt_async(struct AsyncStmtInfo *asi, char *stmt, size_t stmt_len,
				      CmdType cmd, bool ownsit, enum enum_sql_command sqlcom, bool materialize) __attribute__((warn_unused_result));

/**
 * @brief Similar to send_stmt_async(), but send statement in binary protocal
 */
extern StmtSafeHandle send_stmt_async_bin(struct AsyncStmtInfo *asi,
					 char *stmt, size_t stmt_len,
					 List *param_list,
					 MysqlDeserializeInfo *de, int nr_de,
					 CmdType cmd,
					 bool owns_stmt_mem,
					 enum enum_sql_command sqlcom,
					 bool materialize,
					 bool cursor) __attribute__((warn_unused_result));
/**
 * @brief Wait until the result of some stmts can be read
 */
extern StmtSafeHandle wait_for_readable_stmt(StmtSafeHandle *handles, int size);

/**
 * @brief Get the next row of the statement
 * 
 * @param handle 		handle returned from send_stmt_async
 * @return MYSQL_ROW  	Null if EOF
 */
extern MYSQL_ROW get_stmt_next_row(StmtSafeHandle handle);

/**
 * @brief Same as get_stmt_next_row(), but no wait if the result is not ready yet 
 */
extern MYSQL_ROW try_get_stmt_next_row(StmtSafeHandle handle);

/**
 * @brief Similar to get_stmt_next_row(), but return current row in pg datum style instead of string.
 */
bool
get_stmt_next_deserialized_row(StmtSafeHandle handle, Datum *output, bool *isnull, size_t nattrs);

/**
 * @brief Get the number of fields of stmt results
 */
extern int get_stmt_field_count(StmtSafeHandle handle);

/**
 * @brief Get the field types of stmt result
 */
extern enum enum_field_types *get_stmt_field_types(StmtSafeHandle handle);

/**
 * @brief Get the field names of stmt result
 */
extern char ** get_stmt_field_names(StmtSafeHandle handle);

/**
 * @brief Get the row lengths array of the current row
 * 
 * @param handle 		handle returned from send_stmt_async
 */
size_t *get_stmt_row_lengths(StmtSafeHandle handle);

/**
 * @brief Rewind the result of the statement
 * 
 * @param handle 		handle returned from send_stmt_async
 */
extern void stmt_rewind(StmtSafeHandle handle);

/**
 * @brief Check if the statement is rewindable
 * 
 * @param handle 		handle returned from send_stmt_async
 */
extern bool is_stmt_rewindable(StmtSafeHandle handle);

/**
 * @brief Check EOF the the statement
 * 
 * @param handle 		handle returned from send_stmt_async
 */
extern bool is_stmt_eof(StmtSafeHandle handle);

/**
 * @brief Get the affected rows of the statement
 */
extern int32 get_stmt_affected_rows(StmtSafeHandle handle);

/**
 * @brief Mark the statement to be canceled, and no longer cares whether it executes
 *    or its returned result (which may still be executed).
 */
extern void cancel_stmt_async(StmtSafeHandle handle);

/**
 * @brief Marked statement to be canceled, and free all the results remained in the sockets
 */
extern void cancel_stmt(StmtSafeHandle handle);

/**
 * @brief Release the statement handle if no longer use it
 */
extern void release_stmt_handle(StmtSafeHandle handle);

/**
 * @brief Wait util a new transaction started on the given channel(shardid, nodeid)
 */
extern void force_start_transaction(Oid shardid, Oid nodeid, bool master);

#endif // STMT_HDL_H
