/*-------------------------------------------------------------------------
 *
 * string_utils.h
 *		Kunlun Database MySQL protocol server side implementation.
 *		String manipulation routines, types and symbols used by MySQL protocol.
 *
 * Copyright (c) 2019-2022 ZettaDB inc. All rights reserved.
 *
 * This source code is licensed under Apache 2.0 License,
 * combined with Common Clause Condition 1.0, as detailed in the NOTICE file.
 *
 * IDENTIFICATION
 *	  src/include/libmysql/string_utils.h
 *
 *-------------------------------------------------------------------------
 */
#ifndef STRING_UTILS_INCLUDED
#define STRING_UTILS_INCLUDED

#include "postgres.h"
#include <float.h>
#include <limits.h>
#include <stdbool.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "libmysql/mysql_com_sys.h"

#ifndef STRING_WITH_LEN
#define STRING_WITH_LEN(s) (s), (sizeof(s) - 1)
#endif

extern const char _dig_vec_upper[];

inline static char *strmake(char *dst, const char *src, size_t length)
{
  while (length--)
    if (!(*dst++ = *src++)) return dst - 1;
  *dst = 0;
  return dst;
}


static inline char *strend(char *s) {
  while (*s++)
    ;
  return s - 1;
}

/*
  my_stpmov(dst, src) moves all the  characters  of  src  (including  the
  closing NUL) to dst, and returns a pointer to the new closing NUL in
  dst.  The similar UNIX routine strcpy returns the old value of dst,
  which I have never found useful.  my_stpmov(my_stpmov(dst,a),b) moves a//b
  into dst, which seems useful.
*/
static inline char *my_stpmov(char *dst, const char *src) {
  while ((*dst++ = *src++))
    ;
  return dst - 1;
}

/*
  my_stpnmov(dst,src,length) moves length characters, or until end, of src to
  dst and appends a closing NUL to dst if src is shorter than length.
  The result is a pointer to the first NUL in dst, or is dst+n if dst was
  truncated.
*/
static inline char *my_stpnmov(char *dst, const char *src, size_t n) {
  while (n-- != 0) {
    if (!(*dst++ = *src++)) return (char *)dst - 1;
  }
  return dst;
}

/**
   Copy a string from src to dst until (and including) terminating null byte.

   @param dst   Destination
   @param src   Source

   @note src and dst cannot overlap.
         Use my_stpmov() if src and dst overlaps.

   @note Unsafe, consider using my_stpnpy() instead.

   @return pointer to terminating null byte.
*/
static inline char *my_stpcpy(char *dst, const char *src) {
#if defined(HAVE_BUILTIN_STPCPY)
  return __builtin_stpcpy(dst, src);
#elif defined(HAVE_STPCPY)
  return stpcpy(dst, src);
#else
  /* Fallback to implementation supporting overlap. */
  return my_stpmov(dst, src);
#endif
}

static inline char *my_strtok_r(char *str, const char *delim, char **saveptr) {
  return strtok_r(str, delim, saveptr);
}


/* Conversion routines */
typedef enum { SERIALIZE_FLOAT, SERIALIZE_DOUBLE } mysql_float_serialize_type;

extern char *ll10_to_str(int64_t val, char *dst, int radix_sign);



#endif  // STRING_UTILS_INCLUDED
