/*-------------------------------------------------------------------------
 *
 * remote_dml.h
 *	  POSTGRES remote access method DML statements processing code.
 *
 *
 * Copyright (c) 2019-2021 ZettaDB inc. All rights reserved.
 *
 * This source code is licensed under Apache 2.0 License,
 * combined with Common Clause Condition 1.0, as detailed in the NOTICE file.
 *
 * src/include/access/remote_dml.h
 *
 *-------------------------------------------------------------------------
 */
#ifndef REMOTE_DML_H
#define REMOTE_DML_H

#include "executor/tuptable.h"
#include "nodes/execnodes.h"
#include "nodes/nodes.h"
#include "nodes/pg_list.h"
#include "nodes/primnodes.h"
#include "nodes/print.h"
#include "utils/relcache.h"

typedef Node *(FixFn)(PlannerInfo *, Node *, int);

struct RemotePrintExprContext;

struct AsyncStmtInfo;
/*
 * Used to construct sql for un pushedownable update
 */
typedef struct RemoteModifyTupleCommon
{
	MemoryContext mem;
	CommandId cid;
	Oid relid;
	Oid identifier_typid;
	Oid identifier_typmod;
	TupleDesc identifier_tupdesc;
	FmgrInfo *identifier_finfo;
	FmgrInfo *value_finfo;
	Bitmapset *updatedCols;
	Bitmapset *identifier_pkcols;

	// struct AsyncStmtInfo *asi;
	List *asis;
	StringInfoData sql;

	RemoteRWCheckContext *rw_context;
	List *emit_handles;
	uint64 processed;

	bool can_recovery_table_migration;
	int table_migration_recovery_retries;
} RemoteModifyTupleCommon;

/**
 *  @brief Used by Un-pushdownable  DML to update tuples one by one 
 */
extern TupleTableSlot *
ExecRemoteDirectUpdate(EState *estate, ResultRelInfo *resultRelInfo, HeapTuple oldtuple, TupleTableSlot *slot);
/**
 * @brief Used by Un-pushdownable DML to delete tuples one by one
 */
extern void ExecRemoteDirectDelete(EState *estate, ResultRelInfo *resultRelInfo, HeapTuple oldtuple);

/**
 * @brief Used by Un-pushdownable DML to insert one tuple in async manner
 */
extern void ExecRemoteDirectInsert(EState *estate, ResultRelInfo *resultRelInfo, TupleTableSlot *slot,
				   OnConflictAction onConflict, bool only_insert);
/**
 * @brief Used by Un-pushdownable delete to construct result relation tuples from child plan slot
 */
extern void StoreRemoteTupleFromSubPlanSlot(TupleTableSlot *slot,
				     TupleTableSlot *planslot,
				     List *positionList);

/**
 * If the length of the sql is greater than throttle, or no longer grows, then emit the generated remote sql.
 */
extern void EmitRemoteModifyTupleSQL(ResultRelInfo *resultRelInfo, bool end);

extern void EndRemoteModifyTupleCommon(ResultRelInfo *resultRelinfo, uint64 *processed);

/**
 * struct RemoteUDPushdown 
 *     用于存放UPDATE/DELETE 下推的信息
 */
struct RemoteUDPushdown;

/**
 * struct RemoteUDPushdownState
 *     UPDATE/DELETE 下推的运行时状态
 */
struct RemoteUDPushdownState;

extern int numberOfPushdownRemoteStmts(struct RemoteUDPushdown *remote_ud);

/**
 * @brief 将RemoteUDPushdown序列化成List，便于拷贝
 */
extern List * serializeRemoteUDPushdown(struct RemoteUDPushdown *remote_ud);
/**
 * @brief 将List反序列化成RemoteUDPushdown，便于拷贝
 */
extern struct RemoteUDPushdown *deserializeRemoteUDPushdown(List *data);

/**
 * @brief 拷贝RemoteUDPushdown
 */
extern struct RemoteUDPushdown *copyRemoteUDPushdown(struct RemoteUDPushdown *remote_ud);

/**
 * @brief 用于set_plan_refs()，调整表达式的rti
 */
extern void fixRemoteUDPushdown(PlannerInfo *root, struct RemoteUDPushdown *remote_ud, int rtoffset, FixFn fn);

/**
 * @brief 展示第*index个下推的sql，以及目标shard
 */
extern bool nextRemoteUDStmt(struct RemoteUDPushdownState *state, List *rtlist, int *index, Oid *shard, StringInfo sql);

/**
 * @brief 根据rte的updatedCols判断是否更新了某分区表（可能是rte，也可能是其父表或者子表）的分区列
 */
extern bool CheckPartitionKeyUpdated(Oid relid, Bitmapset *updateCols, bool check_children);

/**
 * @brief 能否为ModifyTable计划构建下推的UPDATE/DELETE计划
 */
extern bool CanPlanRemotePushdownUD(PlannerInfo *root, ModifyTable *node);

/**
 * @brief 为ModifyTable计划构建下推的UPDATE/DELETE计划
 */
extern void PlanRemotePushdownUD(PlannerInfo *root, ModifyTable *node, bool hasUpdateIgnore);

/**
 * @brief 初始化UPDATE/DELETE下推计划的运行时状态
 */
extern void ExecInitRemoteUDPushdown(ModifyTableState *node);

/**
 * @brief 执行UDPATE/DELETE下推计划
 */
extern TupleTableSlot * ExecRemoteUDPushdown(ModifyTableState *node);

/**
 * @brief 清理UPDATE/DELETE下推计划的运行时状态
 */
extern void ExecEndRemoteUDPushdown(ModifyTableState *node);


/**
 * @brief 输出被更新列上的约束条件
 */
extern bool print_remote_tuple_constraint(RangeTblEntry *rte, Relation relation, StringInfo sql);

extern bool print_remote_tuple_rls(RangeTblEntry *rte, Relation rel, List *withCheckOptions, StringInfo sql);

#endif // !REMOTE_DML_H
