/* Copyright (c) 2000, 2019, Oracle and/or its affiliates. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License, version 2.0,
   as published by the Free Software Foundation.

   This program is also distributed with certain software (including
   but not limited to OpenSSL) that is licensed under separate terms,
   as designated in a particular file or component or in included license
   documentation.  The authors of MySQL hereby grant you an additional
   permission to link the program and your derivative works with the
   separately licensed software that they have included with MySQL.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License, version 2.0, for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301  USA */

#ifndef MY_DBUG_INCLUDED
#define MY_DBUG_INCLUDED

/**
  @file include/my_dbug.h
*/

#ifdef MY_MSCRT_DEBUG
#include <crtdbg.h>
#endif
#include <stdlib.h>
#include "port_config.h"

//#include "my_compiler.h"

#include <string.h>

#if defined(ENABLE_DEBUG)
#include <assert.h>  // IWYU pragma: keep
#include <stdio.h>
#endif

#include "tcop/debug_sync.h"

#if defined(ENABLE_DEBUG)

struct _db_stack_frame_ {
  const char *func;   /* function name of the previous stack frame       */
  int func_len;       /* how much to print from func */
  const char *file;   /* filename of the function of previous frame      */
  unsigned int level; /* this nesting level, highest bit enables tracing */
  struct _db_stack_frame_ *prev; /* pointer to the previous frame */
};

struct CODE_STATE;

extern int dbug_contains_switch(struct CODE_STATE *, const char *, int);
extern int dbug_show_switches(struct CODE_STATE *cs, char *buf, size_t len);
extern int dbug_show_switches_global(char *buf, size_t len);
extern int dbug_is_pushed(void);
extern void dbug_set_proc_name(const char *name);
extern void dbug_push(const char *control);
extern void dbug_pop(void);
extern void dbug_set_switch(const char *control);
extern void dbug_set_switch_global(const char *control);
extern void dbug_enter(const char *_func_, int func_len, const char *_file_,
					   unsigned int _line_,
					   struct _db_stack_frame_ *_stack_frame_);
extern void dbug_return(unsigned int _line_,
						struct _db_stack_frame_ *_stack_frame_);
extern void dbug_pargs(unsigned int _line_, const char *keyword);
extern int dbug_enabled(void);
extern void dbug_doprnt(const char *format, ...)
	MY_ATTRIBUTE((format(printf, 1, 2)));
extern void dbug_dump(unsigned int _line_, const char *keyword,
					  const unsigned char *memory, size_t length);
extern void dbug_end(void);
extern void dbug_lock_file(void);
extern void dbug_unlock_file(void);
extern FILE *dbug_fp(void);
extern void dbug_flush(void);


#define DBUG_ENTER(a)                       \
  struct _db_stack_frame_ _db_stack_frame_; \
  dbug_enter(a, ::strlen(a), __FILE__, __LINE__, &_db_stack_frame_)

#define DBUG_RETURN(a1)                       \
  do {                                        \
	dbug_return(__LINE__, &_db_stack_frame_); \
	return (a1);                              \
  } while (0)
#define DBUG_VOID_RETURN                      \
  do {                                        \
	dbug_return(__LINE__, &_db_stack_frame_); \
	return;                                   \
  } while (0)
#define DEBUG_INJECT(keyword, a1)        \
  do {                                   \
	if (dbug_contains_switch(0, (keyword), 0)) { \
	  a1                                 \
	}                                    \
  } while (0)
#define DEBUG_INJECT_IF(keyword, a1)     \
  do {                                   \
	if (dbug_contains_switch(0, (keyword), 1)) { \
	  a1                                 \
	}                                    \
  } while (0)
#define DEBUG_EVAL(keyword, a1, a2) \
  (dbug_contains_switch(0, (keyword), 0) ? (a1) : (a2))
#define DEBUG_EVAL_IF(keyword, a1, a2) \
  (dbug_contains_switch(0, (keyword), 1) ? (a1) : (a2))
#define PRINT_DEBUG_INFO(keyword, arglist) \
  do {                               \
	dbug_pargs(__LINE__, keyword);   \
	if (dbug_enabled()) {            \
	  dbug_doprnt arglist;           \
	}                                \
  } while (0)

#define DBUG_PUSH(a1) dbug_push(a1)
#define DBUG_POP() dbug_pop()
#define DBUG_SET(a1) dbug_set_switch(a1)
#define DBUG_SET_INITIAL(a1) dbug_set_switch_global(a1)
#define DBUG_PROCESS(a1) dbug_set_proc_name(a1)
#define DBUG_FILE dbug_fp()
#define DBUG_DUMP(keyword, a1, a2) dbug_dump(__LINE__, keyword, a1, a2)
#define DBUG_END() dbug_end()
#define DBUG_LOCK_FILE dbug_lock_file()
#define DBUG_UNLOCK_FILE dbug_unlock_file()
#define DBUG_ASSERT(A) assert(A)
#define DBUG_EXPLAIN(buf, len) dbug_show_switches(0, (buf), (len))
#define DBUG_EXPLAIN_INITIAL(buf, len) dbug_show_switches_global((buf), (len))
#ifndef _WIN32
#define DBUG_ABORT() (dbug_flush(), abort())
#define DBUG_EXIT() (dbug_flush(), exit(2))
#else
#include <crtdbg.h>

#define DBUG_ABORT()                                                     \
  (dbug_flush(), (void)_CrtSetReportMode(_CRT_ERROR, _CRTDBG_MODE_FILE), \
   (void)_CrtSetReportFile(_CRT_ERROR, _CRTDBG_FILE_STDERR), abort())
#define DBUG_EXIT()                                                      \
  (dbug_flush(), (void)_CrtSetReportMode(_CRT_ERROR, _CRTDBG_MODE_FILE), \
   (void)_CrtSetReportFile(_CRT_ERROR, _CRTDBG_FILE_STDERR), _exit(2))
#endif

/*
  Make the program fail, without creating a core file.
  abort() will send SIGABRT which (most likely) generates core.
  Use SIGKILL instead, which cannot be caught.
  We also pause the current thread, until the signal is actually delivered.
  An alternative would be to use _exit(EXIT_FAILURE),
  but then valgrind would report lots of memory leaks.
 */
#ifdef _WIN32
#define DBUG_SUICIDE() DBUG_EXIT()
#else
extern void dbug_suicide(void) MY_ATTRIBUTE((noreturn));
extern void dbug_flush_gcov(void);
#define DBUG_SUICIDE() (dbug_flush(), dbug_suicide())
#endif

#else /* No debugger */

#ifdef __cplusplus
#define DEBUG_FUNC_ENTER \
  do {             \
  } while (false)
#endif
#define DBUG_ENTER(a1)
#define DBUG_RETURN(a1) \
  do {                  \
	return (a1);        \
  } while (0)
#define DBUG_VOID_RETURN \
  do {                   \
	return;              \
  } while (0)
#define DEBUG_INJECT(keyword, a1) \
  do {                            \
  } while (0)
#define DEBUG_INJECT_IF(keyword, a1) \
  do {                               \
  } while (0)
#define DEBUG_EVAL(keyword, a1, a2) (a2)
#define DEBUG_EVAL_IF(keyword, a1, a2) (a2)
#define PRINT_DEBUG_INFO(keyword, arglist) \
  do {                               \
  } while (0)
#define DBUG_PUSH(a1) \
  do {                \
  } while (0)
#define DBUG_SET(a1) \
  do {               \
  } while (0)
#define DBUG_SET_INITIAL(a1) \
  do {                       \
  } while (0)
#define DBUG_POP() \
  do {             \
  } while (0)
#define DBUG_PROCESS(a1) \
  do {                   \
  } while (0)
#define DBUG_DUMP(keyword, a1, a2) \
  do {                             \
  } while (0)
#define DBUG_END() \
  do {             \
  } while (0)
#define DBUG_ASSERT(A) \
  do {                 \
  } while (0)
#define DBUG_LOCK_FILE \
  do {                 \
  } while (0)
#define DBUG_FILE (stderr)
#define DBUG_UNLOCK_FILE \
  do {                   \
  } while (0)
#define DBUG_EXPLAIN(buf, len)
#define DBUG_EXPLAIN_INITIAL(buf, len)
#define DBUG_ABORT() \
  do {               \
  } while (0)
#define DBUG_SUICIDE() \
  do {                 \
  } while (0)

#endif


#endif /* MY_DBUG_INCLUDED */
