/*-------------------------------------------------------------------------
 *
 * pg_shard.h
 *	  definition of the "shard (db shards)" system catalog (pg_shard)
 *
 *
 * Copyright (c) 2019-2021 ZettaDB inc. All rights reserved.
 *
 * This source code is licensed under Apache 2.0 License,
 * combined with Common Clause Condition 1.0, as detailed in the NOTICE file.
 *
 * src/include/catalog/pg_shard.h
 *
 * NOTES
 *	  The Catalog.pm module reads this file and derives schema
 *	  information.
 *
 *-------------------------------------------------------------------------
 */

#ifndef PG_SHARD_H
#define PG_SHARD_H

#include "catalog/genbki.h"
#include "catalog/pg_shard_d.h"
#include "datatype/timestamp.h"

/*
 * A shard has one master node(i.e. db instance) and multiple slaves nodes,
 * which contain identical data, and each shard node is stored as a mysql db
 * instance on a computer server.
 * A slave node replicates binlogs from master node to keep data in sync.
 * A computing node mostly sent queries to master node to execute, but it
 * can also send read only queries to slave nodes to execute.
 *
 * Here the 12346 and ShardRelationId is hand written, the oid number is
 * chosen at arbitrarily and in the header file pg_shard_d.h generated by
 * genbki.pl, the macro ShardRelationId will be defined as 12346.
 *
 * We don't want to use system 'oid' column in order to use globally consistent
 * id for each shard and shard node. so we have to use Oid(i.e. uint32),
 * we can't use uint32 directly because pg's bki and type system doesn't
 * support uint8/16/32/64 types;
 * */
CATALOG(pg_shard,12346,ShardRelationId) BKI_SHARED_RELATION BKI_WITHOUT_OIDS
{
	NameData name;
	Oid id;
	Oid master_node_id; /* 0 if no master node known to be chosen now. */
	int32 num_nodes;    /* number of nodes, including master; */
	int64 space_volumn;
	int32 num_tablets BKI_DEFAULT(0);
	Oid db_cluster_id BKI_DEFAULT(-1);

	/*
	 * All above fixed length fields have ' not null' constraints in pg_shard
	 * metadata table, no need for extra BKI_FORCE_NOT_NULL notation.
	 * */
#ifdef CATALOG_VARLEN
	timestamptz when_created BKI_DEFAULT(0);
#endif
}
FormData_pg_shard;

typedef FormData_pg_shard*Form_pg_shard;

extern void pg_shard_add_one(NameData *name,
			       Oid shardid,
			       Oid clusterid,
			       int32 num_nodes,
			       int64 spaces_volumn,
			       int32 num_tables,
			       Datum create_tz);

extern void pg_shard_del_one(Oid shardid);

#endif /* !PG_SHARD_H */
