/*-------------------------------------------------------------------------
 *
 * mysql_conn.h
 *		Kunlun Database MySQL protocol server side implementation.
 *		Network IO related types and symbols.
 *
 * Copyright (c) 2019-2022 ZettaDB inc. All rights reserved.
 *
 * This source code is licensed under Apache 2.0 License,
 * combined with Common Clause Condition 1.0, as detailed in the NOTICE file.
 *
 * IDENTIFICATION
 *	  src/include/libmysql/mysql_conn.h
 *
 *-------------------------------------------------------------------------
 */
#ifndef MYSQL_CONN_H
#define MYSQL_CONN_H

#include <stdatomic.h>
#include <stdint.h>
#include "libmysql/mysql_socket.h"

typedef enum enum_virtio_type {
  /**
    Type of the connection is unknown.
  */
  NO_VIRTIO_TYPE = 0,
  /**
    Used in case of TCP/IP connections.
  */
  VIRTIO_TYPE_TCPIP = 1,
  /**
    Used for Unix Domain socket connections. Unix only.
  */
  VIRTIO_TYPE_SOCKET = 2,
  /**
    Used for named pipe connections. Windows only.
  */
  VIRTIO_TYPE_NAMEDPIPE = 3,
  /**
    Used in case of SSL connections.
  */
  VIRTIO_TYPE_SSL = 4,
  /**
    Used for shared memory connections. Windows only.
  */
  VIRTIO_TYPE_SHARED_MEMORY = 5,
  /**
    Used internally by the prepared statements
  */
  VIRTIO_TYPE_LOCAL = 6,
  /**
    Implicitly used by plugins that doesn't support any other VIRTIO_TYPE.
  */
  VIRTIO_TYPE_PLUGIN = 7,

  FIRST_VIRTIO_TYPE = VIRTIO_TYPE_TCPIP,
  /*
    If a new type is added, please update LAST_VIRTIO_TYPE. In addition, please
    change get_virtio_type_name() to return correct name for it.
  */
  LAST_VIRTIO_TYPE = VIRTIO_TYPE_PLUGIN
} enum_virtio_type ;

/**
  Convert a vio type to a printable string.
  @param virtio_type the type
  @param[out] str the string
  @param[out] len the string length
*/
extern void get_virtio_type_name(enum enum_virtio_type virtio_type, const char **str, int *len);

/**
  VIO I/O events.
*/
typedef enum enum_virtio_io_event {
  VIRTIO_IO_EVENT_READ,
  VIRTIO_IO_EVENT_WRITE,
  VIRTIO_IO_EVENT_CONNECT
}enum_virtio_io_event ;

#define VIRTIO_SOCKET_ERROR ((size_t)-1)
#define VIRTIO_SOCKET_WANT_READ ((size_t)-2)
#define VIRTIO_SOCKET_WANT_WRITE ((size_t)-3)

#define VIRTIO_LOCALHOST 1            /* a localhost connection */
#define VIRTIO_BUFFERED_READ 2        /* use buffered read */
#define VIRTIO_READ_BUFFER_SIZE 16384 /* size of read buffer */
#define OPENSSL_ERROR_LENGTH 512   /* Openssl error code max length */

struct MySQLConn;
extern struct MySQLConn* virtio_new(my_socket sd, enum enum_virtio_type type, unsigned int flags);
extern struct MySQLConn* mysql_socket_virtio_new(MYSQL_SOCKET mysql_socket,
                               enum enum_virtio_type type, unsigned int flags);

extern void virtio_delete(struct MySQLConn* myconn);
extern int virtio_shutdown(struct MySQLConn* myconn, int how);
extern int virtio_cancel(struct MySQLConn* myconn, int how);
extern bool virtio_reset(struct MySQLConn* myconn, enum enum_virtio_type type, my_socket sd, void *ssl,
               unsigned int flags);
extern bool virtio_is_blocking(struct MySQLConn *myconn);
extern int virtio_set_blocking(struct MySQLConn *myconn, bool set_blocking_mode);
extern int virtio_set_blocking_flag(struct MySQLConn *myconn, bool set_blocking_flag);
extern size_t virtio_read(struct MySQLConn* myconn, unsigned char *buf, size_t size);
extern size_t virtio_read_buff(struct MySQLConn* myconn, unsigned char *buf, size_t size);
extern size_t virtio_write(struct MySQLConn* myconn, const unsigned char *buf, size_t size);

typedef struct st_virtio_network {
  union {
    struct in_addr in;
    struct in6_addr in6;
  } addr;
  union {
    struct in_addr in;
    struct in6_addr in6;
  } mask;
  sa_family_t family;
}st_virtio_network ;

extern void virtio_proxy_protocol_add(const st_virtio_network *net) ;
extern void virtio_proxy_cleanup() ;
extern void virtio_force_skip_proxy(struct MySQLConn* myconn);
/* setsockopt TCP_NODELAY at IPPROTO_TCP level, when possible */
extern int virtio_fastsend(struct MySQLConn* myconn);
/* setsockopt SO_KEEPALIVE at SOL_SOCKET level, when possible */
extern int virtio_keepalive(struct MySQLConn* myconn, bool onoff);
/* Whenever we should retry the last read/write operation. */
extern bool virtio_should_retry(struct MySQLConn* myconn);
/* Check that operation was timed out */
extern bool virtio_was_timeout(struct MySQLConn* myconn);
#ifdef ENABLE_DEBUG
/* Short text description of the socket for those, who are curious.. */
#define VIRTIO_DESCRIPTION_SIZE 30 /* size of description */
extern void virtio_description(struct MySQLConn* myconn, char *buf);
#endif
/* Return the type of the connection */
extern enum enum_virtio_type virtio_type(const struct MySQLConn* myconn);
/* Return last error number */
extern int virtio_errno(struct MySQLConn* myconn);
/* Get socket number */
extern my_socket virtio_fd(struct MySQLConn* myconn);
/* Remote peer's address and name in text form */
extern bool virtio_peer_addr(struct MySQLConn* myconn, char *buf, uint16_t *port, size_t buflen);
/* Wait for an I/O event notification. */
extern int virtio_io_wait(struct MySQLConn* myconn, enum enum_virtio_io_event event, int timeout);
extern bool virtio_is_connected(struct MySQLConn* myconn);
#ifdef ENABLE_DEBUG
extern ssize_t virtio_pending(struct MySQLConn* myconn);
#endif
/* Set timeout for a network operation. */
extern int virtio_timeout(struct MySQLConn* myconn, unsigned int which, int timeout_sec);
extern void virtio_set_wait_callback(void (*before_wait)(void),
                                  void (*after_wait)(void));

/* Connect to a peer. */
extern bool virtio_socket_connect(struct MySQLConn* myconn, struct sockaddr *addr, socklen_t len,
                        bool nonblocking, int timeout,
                        bool *connect_done);

extern bool virtio_get_normalized_ip_string(const struct sockaddr *addr,
                                  size_t addr_length, char *ip_string,
                                  size_t ip_string_size);

extern bool virtio_is_no_name_error(int err_code);


#include <openssl/opensslv.h>
#if OPENSSL_VERSION_NUMBER < 0x0090700f
#define DES_cblock des_cblock
#define DES_key_schedule des_key_schedule
#define DES_set_key_unchecked(k, ks) des_set_key_unchecked((k), *(ks))
#define DES_ede3_cbc_encrypt(i, o, l, k1, k2, k3, iv, e) \
  des_ede3_cbc_encrypt((i), (o), (l), *(k1), *(k2), *(k3), (iv), (e))
#endif

#if OPENSSL_VERSION_NUMBER >= 0x10100000L
#define HAVE_OPENSSL11 1
#endif  // OPENSSL_VERSION_NUMBER

#define HEADER_DES_LOCL_H dummy_something

#include <openssl/err.h>
#include <openssl/ssl.h>

typedef enum enum_ssl_init_error {
  SSL_INITERR_NOERROR = 0,
  SSL_INITERR_CERT,
  SSL_INITERR_KEY,
  SSL_INITERR_NOMATCH,
  SSL_INITERR_BAD_PATHS,
  SSL_INITERR_CIPHERS,
  SSL_INITERR_MEMFAIL,
  SSL_INITERR_NO_USABLE_CTX,
  SSL_INITERR_DHFAIL,
  SSL_TLS_VERSION_INVALID,
  SSL_FIPS_MODE_INVALID,
  SSL_FIPS_MODE_FAILED,
  SSL_INITERR_ECDHFAIL,
  SSL_INITERR_X509_VERIFY_PARAM,
  SSL_INITERR_LASTERR
}enum_ssl_init_error ;
const char *sslGetErrString(enum enum_ssl_init_error err);

typedef struct st_VirtioSSLFd {
  SSL_CTX *ssl_context;
}st_VirtioSSLFd ;

extern int sslaccept(struct st_VirtioSSLFd *, struct MySQLConn*, long timeout,
              unsigned long *errptr);
extern int sslconnect(struct st_VirtioSSLFd *, struct MySQLConn*, long timeout,
               unsigned long *errptr, SSL **ssl);

extern struct st_VirtioSSLFd *new_VirtioSSLConnectorFd(
    const char *key_file, const char *cert_file, const char *ca_file,
    const char *ca_path, const char *cipher, const char *ciphersuites,
    enum enum_ssl_init_error *error, const char *crl_file, const char *crl_path,
    const long ssl_ctx_flags, const char *server_host);

extern long process_tls_version(const char *tls_version);

extern int set_fips_mode(const unsigned int fips_mode, char *err_string);

extern unsigned int get_fips_mode(void );

extern struct st_VirtioSSLFd *new_VirtioSSLAcceptorFd(
    const char *key_file, const char *cert_file, const char *ca_file,
    const char *ca_path, const char *cipher, const char *ciphersuites,
    enum enum_ssl_init_error *error, const char *crl_file, const char *crl_path,
    const long ssl_ctx_flags);
extern void free_virtio_ssl_acceptor_fd(struct st_VirtioSSLFd *fd);

extern void virtio_ssl_end(void );

extern void ssl_start(void);
extern void virtio_end(void);

#if !defined(DONT_MAP_VIRTIO)
#define virtio_delete(myconn) (myconn)->viodelete(myconn)
#define virtio_errno(myconn) (myconn)->vioerrno(myconn)
#define virtio_read(myconn, buf, size) ((myconn)->read)(myconn, buf, size)
#define virtio_write(myconn, buf, size) ((myconn)->write)(myconn, buf, size)
#define virtio_fastsend(myconn) (myconn)->fastsend(myconn)
#define virtio_keepalive(myconn, set_keep_alive) \
  (myconn)->viokeepalive(myconn, set_keep_alive)
#define virtio_should_retry(myconn) (myconn)->should_retry(myconn)
#define virtio_was_timeout(myconn) (myconn)->was_timeout(myconn)
#define virtio_shutdown(myconn, how) ((myconn)->vioshutdown)(myconn, how)
#define virtio_cancel(myconn, how) ((myconn)->viocancel)(myconn, how)
#define virtio_peer_addr(myconn, buf, prt, buflen) \
  (myconn)->peer_addr(myconn, buf, prt, buflen)
#define virtio_io_wait(myconn, event, timeout) (myconn)->io_wait(myconn, event, timeout)
#define virtio_is_connected(myconn) (myconn)->is_connected(myconn)
#define virtio_is_blocking(myconn) (myconn)->is_blocking(myconn)
#define virtio_set_blocking(myconn, val) (myconn)->set_blocking(myconn, val)
#define virtio_set_blocking_flag(myconn, val) (myconn)->set_blocking_flag(myconn, val)
#endif /* !defined(DONT_MAP_VIRTIO) */

#define virtio_set_thread_id(myconn, tid)

#define USE_PPOLL_IN_VIRTIO

/* This enumerator is used in parser - should be always visible */
typedef enum SSL_type {
  SSL_TYPE_NOT_SPECIFIED = -1,
  SSL_TYPE_NONE,
  SSL_TYPE_ANY,
  SSL_TYPE_X509,
  SSL_TYPE_SPECIFIED
} SSL_type;
/*
  MySQL network connection state.
*/
typedef struct MySQLConn
{
  MYSQL_SOCKET mysql_socket;          /* Instrumented socket */
  bool localhost;           /* Are we from localhost? */
  enum_virtio_type type; /* Type of connection */

  int read_timeout;  /* Timeout value (ms) for read ops. */
  int write_timeout; /* Timeout value (ms) for write ops. */
  int retry_count;    /* Retry count */
  bool inactive;  /* Connection has been shutdown */
  bool force_skip_proxy;

  struct sockaddr_storage local;  /* Local internet address */
  struct sockaddr_storage remote; /* Remote internet address */
  size_t addrLen;           /* Length of remote address */
  char *read_buffer;  /* buffer for virtio_read_buff */
  char *read_pos;     /* start of unfetched data in the
                                     read buffer */
  char *read_end;     /* end of unfetched data */

#ifdef USE_PPOLL_IN_VIRTIO
  my_thread_t thread_id;  // Thread PID
  sigset_t signal_mask;         // Signal mask
  /*  
    Flag to indicate whether we are in poll or shutdown.
    A true value of flag indicates either the socket
    has called  shutdown or it is sleeping on a poll call.
    False value of this flag means that the socket is
    not sleeping on a poll call.
    This flag provides synchronization between two threads
    one entering virtio_io_wait and another entering virtio_shutdown
    for the same socket. If the other thread is waiting on poll
    sleep, it wakes up the thread by sending a signal via
    pthread_kill. Also it ensures that no other thread enters in
    to a poll call if it's socket has undergone shutdown.

  */
  volatile atomic_flag poll_shutdown_flag;
#elif defined HAVE_KQUEUE
  int kq_fd;
  volatile atomic_flag kevent_wakeup_flag;
#endif

#ifdef HAVE_SETNS
  /**
    Socket network namespace.
  */
  char network_namespace[256];
#endif
  void *ssl_arg;
  /* Indicates whether socket or SSL based communication is blocking or not. */
  bool is_blocking_flag;
  struct MemoryContextData * memctx;

  void (*viodelete)(struct MySQLConn *);
  int (*vioerrno)(struct MySQLConn *);
  size_t (*read)(struct MySQLConn *, unsigned char *, size_t);
  size_t (*write)(struct MySQLConn *, const unsigned char *, size_t);
  int (*timeout)(struct MySQLConn *, uint, bool);
  int (*viokeepalive)(struct MySQLConn *, bool);
  int (*fastsend)(struct MySQLConn *);
  bool (*peer_addr)(struct MySQLConn *, char *, uint16_t *, size_t);
  void (*in_addr)(struct MySQLConn *, struct sockaddr_storage *);
  bool (*should_retry)(struct MySQLConn *);
  bool (*was_timeout)(struct MySQLConn *);
  /*
     vioshutdown is resposnible to shutdown/close the channel, so that no
     further communications can take place, however any related buffers,
     descriptors, handles can remain valid after a shutdown.
  */
  int (*vioshutdown)(struct MySQLConn *, int);
  /**
    Partial shutdown. All the actions performed which shutdown performs,
    but descriptor remains open and valid.
  */
  int (*viocancel)(struct MySQLConn *, int);
  bool (*is_connected)(struct MySQLConn *);
  bool (*has_data)(struct MySQLConn *);
  int (*io_wait)(struct MySQLConn *, enum enum_virtio_io_event, int);
  bool (*connect)(struct MySQLConn *, struct sockaddr *, socklen_t, int);
  bool (*is_blocking)(struct MySQLConn *myconn);
  int (*set_blocking)(struct MySQLConn *myconn, bool val);
  int (*set_blocking_flag)(struct MySQLConn *myconn, bool val);

} MySQLConn;

extern void virtio_proxy_cleanup(void );
extern void virtio_force_skip_proxy(MySQLConn *myconn);
extern void virtio_assign(MySQLConn *this_myconn, MySQLConn *myconn);
#endif // !MYSQL_CONN_H
