/*-------------------------------------------------------------------------
 *
 * mysql_prepare.h
 *		Kunlun Database MySQL protocol server side implementation.
 *		MySQL command struct definitions.
 *
 * Copyright (c) 2019-2022 ZettaDB inc. All rights reserved.
 *
 * This source code is licensed under Apache 2.0 License,
 * combined with Common Clause Condition 1.0, as detailed in the NOTICE file.
 *
 * IDENTIFICATION
 *	  src/include/libmysql/mysql_prepare.h
 *
 *-------------------------------------------------------------------------
 */
#ifndef MYSQL_PREPARE_H
#define MYSQL_PREPARE_H
#include "postgres.h"
#include <stdint.h>
#include "libmysql/mysql_time.h"
#include "libmysql/mysql_session.h"
#include "libmysql/field_types.h"
#include "lib/stringinfo.h"
#include "nodes/pg_list.h"

typedef struct StringRef
{
  char *str;
  size_t len;
} StringRef;

union COM_DATA;
struct MemoryContextData;
struct List;
struct CachedPlanSource;
struct ParamListInfoData;
struct PS_PARAM;
struct MySQLSession;

typedef enum PSParamState {
    PSParam_NO_VALUE,
    PSParam_NULL_VALUE,
    PSParam_INT_VALUE,
    PSParam_REAL_VALUE,
    PSParam_STRING_VALUE,
    PSParam_TIME_VALUE,  ///< holds TIME, DATE, DATETIME
    PSParam_LONG_DATA_VALUE,
    PSParam_DECIMAL_VALUE
} PSParamState;

typedef void (*set_param_func_t)(struct PS_PARAM *param, unsigned char **pos,
		unsigned long len);

typedef struct PS_PARAM {
  bool isnull;
  bool is_unsigned; // bound param value is unsigned or not
  enum_field_types actual_type; // type of parameter value supplied for binding
  PSParamState param_state;
  
  unsigned short param_num; // param number starting from 1.

  // parameter value supplied for binding, its type is defined in actual_type.
  union {
	int8_t i8v;
	uint8_t ui8v;
    int16_t i16v;
	uint16_t ui16v;
    int32_t i32v;
	uint32_t ui32v;
	int64_t i64v;
	uint64_t ui64v;
	float fv;
	double dv;
	// decimal?
  } native_val;// value of native types
  MYSQL_TIME time_val;
  StringInfoData strbuf;// stores string value

  Oid typid; // param value's type id in pg deducted during prepare phase

  // charset encoding of the string in strbuf. user may change client encoding via
  // 'SET NAMES' stmt between bind and execute, so we must store the string's
  // encoding with it instead of using the session state.
  Oid charset_encid;

  // callback func to store parameter value from protocol memory buffer.
  set_param_func_t set_param_func;

  // the name is only used as query attribute names, which we don't support here.
  StringRef name;

  // references to raw bytes transferred from client and stored in protocol
  // buffer, while above fields stores typed value in this object.
  unsigned char *value;
  unsigned long length;
}PS_PARAM ;

typedef struct Prepared_statement
{
  // session wide uniquely identifies a prepared stmt. multiple stmts can
  // be active at the same time and they can be executed interleaved.
  unsigned long stmt_id;
  
  unsigned int param_count;
  unsigned int flags;
  
  // stores above stmt_id in str because pg identifies PS objects by string name
  char stmt_id_str[16];

  // alloced from thd->memctx, valid through lifetime of the prepared stmt
  MemoryContext ps_memctx;

  StringRef m_name; // refers to above stmt_id_str
  PS_PARAM *param_array; // alloc'ed from ps_memctx, valid thourghout the prepared stmt
  struct CachedPlanSource *cached_plan_src;
  long nrows_fetch; // the NO. of rows to fetch in a COM_STMT_FETCH cmd.

  // mysql params in param_array are converted to pg_params which will be
  // used during multiple fetches of the same BIND's EXECUTE result
  // this is alloc'ed from portal's memctx
  struct ParamListInfoData * pg_params;

  /**
  The query associated with this statement.
  */
  StringInfoData m_query_string;
  
  MySQLSession *thd;
  unsigned int last_errno;
  StringInfoData message_text;
  
} Prepared_statement;

/*
TODO:
  An interface that is used to take an action when
  the locking module notices that a table version has changed
  since the last execution. "Table" here may refer to any kind of
  table -- a base table, a temporary table, a view or an
  information schema table.

  When we open and lock tables for execution of a prepared
  statement, we must verify that they did not change
  since statement prepare. If some table did change, the statement
  parse tree *may* be no longer valid, e.g. in case it contains
  optimizations that depend on table metadata.
*/
// reprepare when referenced tables altered

extern bool mysql_stmt_precheck(struct MySQLSession *thd,
						 const union COM_DATA *com_data,
                         enum enum_server_command cmd,
                         Prepared_statement **stmt);
extern void mysqld_stmt_prepare(struct MySQLSession *thd, const char *query,
	unsigned int length, Prepared_statement *stmt);
extern void mysqld_stmt_execute(MySQLSession *thd, Prepared_statement *stmt,
                         PS_PARAM *parameters);
extern void mysqld_stmt_close(struct MySQLSession *thd, Prepared_statement *stmt);
extern void mysql_sql_stmt_prepare(struct MySQLSession *thd);
extern void mysql_sql_stmt_execute(struct MySQLSession *thd);
extern void mysql_sql_stmt_close(struct MySQLSession *thd);
extern void mysqld_stmt_fetch(struct MySQLSession *thd, Prepared_statement *stmt,
 	unsigned long num_rows);
extern void mysqld_stmt_reset(struct MySQLSession *thd, Prepared_statement *stmt);
extern void mysql_stmt_get_longdata(struct MySQLSession *thd, Prepared_statement *stmt,
	unsigned int param_number, unsigned char *longdata, unsigned long length);

extern bool mysql_ps_insert_params(Prepared_statement *pstmt, PS_PARAM *parameters,
	unsigned int paramcnt, bool refresh_types);
/** One result set record. */

typedef struct  PS_cached_row
{
  StringRef *m_column_array;
  size_t m_column_count;
}PS_cached_row ;

/**
  Cached_resultset -- a container with result set rows.
*/
typedef struct Cached_resultset {


  struct MemoryContextData * memctx;
  size_t m_column_count;
  struct List*m_rows; // <PS_cached_row> 
  PS_cached_row *m_fields;
  struct Cached_resultset *m_next_rset;
}Cached_resultset ;

inline static unsigned int
Cached_resultset_size(Cached_resultset*rs) { return list_length(rs->m_rows); }
inline static PS_cached_row *
Cached_resultset_get_fields(Cached_resultset*rs) { return rs->m_fields; }
inline static size_t
get_field_count(Cached_resultset*rs) { return rs->m_column_count; }


inline static const StringRef *
get_column_from_cached_row(PS_cached_row *row, const unsigned int column_index)
{
  Assert(column_index < row->m_column_count);
  return row->m_column_array + column_index;
}

inline static void
init_ps_cached_row(PS_cached_row*row, StringRef *column_array_arg,
	size_t column_count_arg)
{
  row->m_column_array=column_array_arg;
  row->m_column_count=column_count_arg;
}

/*
  we immediately convert the mysql params to pg params when EXECUTE command is executed,
  so no need to duplicate another copy.
*/
inline static void
copy_bind_parameter_values(Prepared_statement *stmt, PS_PARAM *parameters, unsigned int paramcnt)
{
  stmt->param_array = parameters;
  stmt->param_count = paramcnt;
}


#endif  // MYSQL_PREPARE_H
