/*-------------------------------------------------------------------------
 *
 * mysql_proto.h
 *		Kunlun Database MySQL protocol server side implementation.
 *		Proto types and symbols.
 *
 * Copyright (c) 2019-2022 ZettaDB inc. All rights reserved.
 *
 * This source code is licensed under Apache 2.0 License,
 * combined with Common Clause Condition 1.0, as detailed in the NOTICE file.
 *
 * IDENTIFICATION
 *	  src/include/libmysql/mysql_proto.h
 *
 *-------------------------------------------------------------------------
 */
#ifndef MYSQL_PROTO_H
#define MYSQL_PROTO_H

#include "libmysql/mysql_com.h"
#include "libmysql/mysql_conn.h"
#include "libmysql/field_types.h"
#include "lib/stringinfo.h"

struct CommBuffer;
struct MySQLSession;
struct MySQLConn;
union COM_DATA;
struct List;
struct MySQLProto;
struct TupleTableSlot;
struct FormData_pg_attribute;
struct TargetEntry;
struct MySQLFieldMeta;

typedef enum MySQLProtoSendResultType
{ MySQLProto_SEND_NUM_ROWS = 1, MySQLProto_SEND_DEFAULTS = 2, MySQLProto_SEND_EOF = 4 }
MySQLProtoSendResultType;

typedef enum enum_protocol_type {
    PROTOCOL_TEXT = 0,    // text Protocol type used mostly
                          // for the old (MySQL 4.0 protocol)
    PROTOCOL_BINARY = 1,  // binary protocol type
    PROTOCOL_LOCAL = 2,   // local protocol type(intercepts communication)
    PROTOCOL_ERROR = 3,   // error protocol instance
    PROTOCOL_PLUGIN = 4   // pluggable protocol type
} enum_protocol_type ;

typedef void (*proto_start_row_t)(struct MySQLProto *proto) ;
typedef bool (*proto_store_null_t)(struct MySQLProto *proto) ;
typedef bool (*proto_store_tiny_t)(struct MySQLProto *proto, long long from) ;
typedef bool (*proto_store_short_t)(struct MySQLProto *proto, long long from) ;
typedef bool (*proto_store_long_t)(struct MySQLProto *proto, long long from) ;
typedef bool (*proto_store_longlong_t)(struct MySQLProto *proto, long long from, bool unsigned_flag);
typedef bool (*proto_store_datetime_t)(struct MySQLProto *proto, const Datum time, unsigned int precision) ;
typedef bool (*proto_store_date_t)(struct MySQLProto *proto, const Datum time) ;
typedef bool (*proto_store_time_t)(struct MySQLProto *proto, const Datum time, unsigned int precision) ;
typedef bool (*proto_store_float_t)(struct MySQLProto *proto, Datum d) ;
typedef bool (*proto_store_double_t)(struct MySQLProto *proto, Datum d) ;

typedef bool (*proto_send_parameters_t)(struct MySQLProto *proto, struct List*parameters,// list of Item_param
                     bool is_sql_prepare) ;

typedef struct MySQLProto
{
  unsigned long m_client_capabilities;
  // refers to MySQLSession::packet, protocol reads from here & write to here
  StringInfo packet;
  struct MySQLSession *thd;
  // all stuff alloced here are released after a request is handled and results sent to client.
  MemoryContext memctx;

  uint32_t field_pos;
  unsigned int fld_meta_pos;
  enum enum_field_types *field_types;
  // true if sending field metadata; false if sending row&field data if not idle.
  bool sending_metadata;
  // packet format illegal
  bool bad_packet;
  enum enum_protocol_type proto_type;
  uint32_t field_count;
  uint32_t sending_flags;
  unsigned int bit_fields;// for binary protocol only
  unsigned long input_packet_length;
  unsigned char *input_raw_packet;
  //const CHARSET_INFO *result_cs;
  struct CommBuffer *net;

  // set text or binary funcs according to proto_type
  proto_start_row_t start_row;
  proto_store_null_t store_null;
  proto_store_tiny_t store_tiny;
  proto_store_short_t store_short;
  proto_store_long_t store_long;
  proto_store_longlong_t store_longlong;
  proto_store_datetime_t store_datetime;
  proto_store_date_t store_date;
  proto_store_time_t store_time;
  proto_store_float_t store_float;
  proto_store_double_t store_double;
  proto_send_parameters_t send_parameters;

} MySQLProto;

inline static unsigned long
proto_get_client_capabilities(MySQLProto*proto)
{ return proto->m_client_capabilities; }

/* Sets the client capabilities */
inline static void
proto_set_client_capabilities(MySQLProto*proto,
	unsigned long client_capabilities) {
  proto->m_client_capabilities = client_capabilities;
}
/* Adds  client capability */
inline static void
proto_add_client_capability(MySQLProto*proto,
	unsigned long client_capability) {
  proto->m_client_capabilities |= client_capability;
}
/* Removes a client capability*/
inline static void
proto_remove_client_capability(MySQLProto*proto, unsigned long capability) {
  proto->m_client_capabilities &= ~capability;
}
/* Returns true if the client has the capability and false otherwise*/
inline static bool
proto_has_client_capability(MySQLProto*proto, unsigned long client_capability) {
  return (bool)(proto->m_client_capabilities & client_capability);
}

inline static bool proto_flush(MySQLProto*proto)
{ return comm_flush(proto->net); }

inline static bool
proto_write(MySQLProto *proto, const unsigned char *ptr, size_t len) {
  return comm_write(proto->net, ptr, len);
}

inline static unsigned long long
proto_get_packet_length(MySQLProto *proto)
{ return proto->input_packet_length; }

extern int proto_read_packet(MySQLProto *proto);

struct Port;
extern int SendInitialHandShakePacket(struct Port*port);

extern bool thd_send_error(struct MySQLSession *thd, unsigned int sql_errno, const char *err);
extern bool comm_send_error(CommBuffer *net, unsigned int sql_errno, const char *err);
extern bool proto_store_strs(MySQLProto*, struct List *strlist);

extern bool proto_send_ok(MySQLProto*, unsigned int server_status, unsigned int statement_warn_count,
              unsigned long long affected_rows,unsigned long long last_insert_id,
               const char *message) ;

extern bool proto_send_eof(MySQLProto*, unsigned int server_status, unsigned int statement_warn_count) ;

extern bool proto_send_error(MySQLProto*, unsigned int sql_errno, const char *err_msg,
                const char *sql_state) ;
extern bool proto_store_ps_status(MySQLProto*, unsigned long stmt_id, unsigned int column_count, unsigned int param_count,
                     unsigned long cond_count) ;

extern void proto_init(MySQLProto*, struct MySQLSession *thd_arg, enum_protocol_type prottype);
extern bool proto_store_string(MySQLProto*, const char *from, size_t length);
extern int proto_get_command(MySQLProto*, union COM_DATA *com_data, enum_server_command *cmd) ;
extern enum_field_types type_conv_pg2mysql(Oid pgtypid);
extern Oid type_conv_mysql2pg(enum_field_types fldtype);

/**
  Parses the passed parameters and creates a command

  @param com_data  out parameter
  @param cmd       in  parameter
  @param pkt       packet to be parsed
  @param length    size of the packet

  @retval false   ok
  @retval true    error
*/
extern bool proto_create_command(MySQLProto*, union COM_DATA *com_data,
	enum_server_command cmd, unsigned char *pkt, size_t length);

/*
  In server mode(non-embedded mode), this call always writes current result row
  into net buffer, may or may not trigger a socket write.
*/
inline static bool proto_end_row(MySQLProto*proto)
{
  return comm_write(proto->net, (unsigned char *)(proto->packet->data),
                      proto->packet->len);
}

inline static unsigned int proto_get_rw_status(MySQLProto*proto)
{ return proto->net->reading_or_writing; }

inline static bool proto_get_compression(MySQLProto*proto)
{ return proto->net->compress; }
extern char *proto_get_compression_algorithm(MySQLProto*) ;
extern unsigned int proto_get_compression_level(MySQLProto*) ;

extern bool proto_start_result_metadata(MySQLProto*,
	unsigned int num_cols, unsigned int flags);
extern bool proto_end_result_metadata(MySQLProto*) ;

extern bool proto_send_field_metadata(MySQLProto*proto, struct MySQLFieldMeta *fldmeta);
extern bool makeMySQLFieldMeta(struct MySQLFieldMeta*fld, struct TargetEntry *te);
extern void makeMySQLFieldMetaByAttr(struct MySQLFieldMeta*fld, struct FormData_pg_attribute* attr);
inline static void proto_abort_row(MySQLProto*proto)  {}

/**
  Returns the type of the connection

  @return
    enum enum_virtio_type
*/
inline static enum enum_virtio_type proto_connection_type(MySQLProto*proto) {
  const struct MySQLConn *v = proto->net->myconn;
  return v ? virtio_type(v) : NO_VIRTIO_TYPE;
}

inline static my_socket proto_get_socket(MySQLProto*proto)
{ return proto->net->myconn->mysql_socket.fd; }

// NET interaction functions
/* Return last error from NET */
inline static unsigned char proto_get_error(MySQLProto*proto)
{ return proto->net->error; }

/* Set max allowed packet size */
inline static void
proto_set_max_packet_size(MySQLProto*proto, unsigned long max_packet_size)
{ proto->net->max_packet_size = max_packet_size;}
/* Deinitialize VIO */
extern int proto_shutdown(MySQLProto*, bool server_shutdown ) ;

/* Check whether VIO is healhty */
inline static bool proto_connection_alive(MySQLProto*proto)
{ return proto->net->myconn != NULL; }

inline static void proto_start_row(MySQLProto*proto) 
{
  proto->start_row(proto);
}

inline static bool proto_store_null(MySQLProto*proto) 
{
  return proto->store_null(proto);
}

inline static bool proto_store_tiny(MySQLProto*proto, long long from) 
{
  return proto->store_tiny(proto, from);
}

inline static bool proto_store_short(MySQLProto*proto, long long from) 
{
  return proto->store_short(proto, from);
}

inline static bool proto_store_long(MySQLProto*proto, long long from) 
{
  return proto->store_long(proto, from);
}

inline static bool proto_store_longlong(MySQLProto*proto,
	long long from, bool unsigned_flag)
{
  return proto->store_longlong(proto, from, unsigned_flag);
}



inline static bool
proto_store_datetime(MySQLProto*proto, const Datum time, unsigned int precision)
{
  return proto->store_datetime(proto, time, precision);
}

inline static bool
proto_store_date(MySQLProto*proto, const Datum time) 
{
  return proto->store_date(proto, time);
}

inline static bool
proto_store_time(MySQLProto*proto, const Datum time, unsigned int precision)
{
  return proto->store_time(proto, time, precision);
}

inline static bool proto_store_float(MySQLProto*proto, Datum nr) 
{
  return proto->store_float(proto, nr);
}

inline static bool proto_store_double(MySQLProto*proto, Datum from) 
{
  return proto->store_double(proto, from);
}


inline static bool proto_send_parameters(MySQLProto*proto,
	struct List*parameters,// list of Item_param
	bool is_sql_prepare)
{
  return proto->send_parameters(proto, parameters, is_sql_prepare);
}

inline static bool proto_store_int(MySQLProto*proto, int from)
{ return proto_store_long(proto, (long long)(from)); }

inline static bool proto_store_uint32(MySQLProto*proto, uint32_t from)
{ return proto_store_long(proto, (long long)(from)); }

inline static bool proto_store_ll(MySQLProto*proto, long long from)
{ return proto_store_longlong(proto, from, false); }

inline static bool proto_store_ull(MySQLProto*proto,unsigned long long from) {
  return proto_store_longlong(proto, (long long)(from), true);
}

inline static bool proto_store_tiny_no0fill(MySQLProto*proto, long long from)
{ return proto_store_tiny(proto, from); }

inline static bool proto_store_short_no0fill(MySQLProto*proto, long long from)
{ return proto_store_short(proto, from); }

inline static bool proto_store_long_no0fill(MySQLProto*proto, long long from)
{ return proto_store_long(proto, from); }

inline static bool
proto_store_ll_no0fill(MySQLProto*proto, long long from, bool unsigned_flag) {
  return proto_store_longlong(proto, from, unsigned_flag);
}

inline static unsigned int proto_get_output_pkt_nr(MySQLProto*proto)
{ return proto->net->pkt_nr; }

inline static StringInfo proto_get_output_packet(MySQLProto*proto)
{ return proto->packet; }


inline static void proto_wipe_net(MySQLProto*proto) {
  memset(proto->net, 0, sizeof(*proto->net));
}

inline static struct CommBuffer *
proto_get_net(MySQLProto*proto) { return proto->net; }

inline static struct MySQLConn *proto_get_vio(MySQLProto*proto)
{ return proto->net->myconn; }

inline static void proto_set_vio(MySQLProto*proto, struct MySQLConn *myconn)
{ proto->net->myconn = myconn; }

inline static void
proto_set_output_pkt_nr(MySQLProto*proto, unsigned int pkt_nr) {
  proto->net->pkt_nr = pkt_nr;
}

inline static void
proto_set_read_timeout(MySQLProto*proto, unsigned long read_timeout) {
  comm_set_read_timeout(proto->net, read_timeout);
}

inline static void
proto_set_write_timeout(MySQLProto*proto, unsigned long write_timeout) {
  comm_set_write_timeout(proto->net, write_timeout);
}

inline static void proto_end_net(MySQLProto*proto) {
  Assert(proto->net->buff);
  comm_end(proto->net);
  proto->net->myconn = NULL;
}

#endif // !MYSQL_PROTO_H
