/*-------------------------------------------------------------------------
 *
 * mysql_session.h
 *		Kunlun Database MySQL protocol server side implementation.
 *		Session management types and symbols.
 *
 * Copyright (c) 2019-2022 ZettaDB inc. All rights reserved.
 *
 * This source code is licensed under Apache 2.0 License,
 * combined with Common Clause Condition 1.0, as detailed in the NOTICE file.
 *
 * IDENTIFICATION
 *	  src/include/libmysql/mysql_session.h
 *
 *-------------------------------------------------------------------------
 */
#ifndef MYSQL_SESSION_H
#define MYSQL_SESSION_H

#include "postgres.h"
#include "lib/stringinfo.h"
#include "utils/memutils.h"
#include "libmysql/mysql_com.h"
#include "libmysql/mysql_conn.h"
#include "libmysql/my_command.h"
#include "libmysql/com_data.h"

struct MySQLConn;
struct MySQLAuth;
struct CommBuffer;
struct MySQLProto;
struct MemoryContextData;
struct Port;
struct TupleTableSlot;
struct List;
struct MYSQL_TIME;
struct Prepared_statement;
struct tupleDesc;
struct pg_tz;

#define SPECIAL_NO_RESOLVE 64     /* Don't use gethostname */
#define SPECIAL_NO_HOST_CACHE 512 /* Don't cache hosts */

typedef enum enum_StmtStat{
    /** The area is cleared at start of a statement. */
    SS_EMPTY = 0,
    /** Set whenever one calls my_ok(). */
    SS_OK,
    /** Set whenever one calls my_eof(). */
    SS_EOF,
    /** Set whenever one calls my_error() or my_message(). */
    SS_ERROR,
    /** Set in case of a custom response, such as one from COM_STMT_PREPARE. */
    SS_DISABLED
}enum_StmtStat;

typedef struct StmtStat
{
  bool is_sent;
  enum_StmtStat status;
  /**
    Message buffer. It is used only when SS is in OK or ERROR status.
    If SS status is ERROR, it's the MESSAGE_TEXT attribute of SQL-condition.
    If SS status is OK, it's the OK-message to be sent.

	This must be alloced in ErrorContext.
  */
  StringInfoData message_text;// MYSQL_ERRMSG_SIZE

  /**
    SQL RETURNED_SQLSTATE condition item.
    This member is always NUL terminated.
  */
  StringInfoData returned_sqlstate;//[SQLSTATE_LENGTH + 1];

  /**
    SQL error number. One of ER_ codes from share/errmsg.txt.
    Set by set_error_status.
  */
  unsigned int mysql_errno;
  unsigned int num_warnings; // NO. of warnings.
  /**
    The number of rows affected by the last statement. This is
    semantically close to thd->row_count_func, but has a different
    life cycle. thd->row_count_func stores the value returned by
    function ROW_COUNT() and is cleared only by statements that
    update its value, such as INSERT, UPSSTE, DELETE and few others.
    This member is cleared at the beginning of the next statement.

    We could possibly merge the two, but life cycle of thd->row_count_func
    can not be changed.
  */
  unsigned long long affected_rows;

  /**
    Similarly to the previous member, this is a replacement of
    thd->first_successful_insert_id_in_prev_stmt, which is used
    to implement LAST_INSERT_ID().
  */
  unsigned long long last_insert_id;
  unsigned long long sent_row_count;
  unsigned long long examined_row_count;
} StmtStat;

typedef struct Security_context
{
  StringInfoData ip, host, user, host_or_ip;
  uint16_t port;
} Security_context;

typedef struct TxnState
{
  bool is_autocommit;
} TxnState;

typedef enum enum_killed_state {
    NOT_KILLED = 0,
    KS_KILL_CONNECTION,
    KS_KILL_QUERY,
    KS_KILL_TIMEOUT,
    KS_KILLED_NO_VALUE /* means neither of the states */
} enum_killed_state;

typedef struct MySQLSession
{
  struct MemoryContextData* memctx; // all objects of this session should be alloced here.
  struct MemoryContextData* stmt_memctx; // objects valid only during one stmt execution should be alloced here.
  struct MySQLConn *conn;// network connection to recv commands and send results
  struct CommBuffer *comm_buffer;// com_buf->conn and this->conn always points to the same object.
  struct MySQLAuth *auth;

  /*
	We use text protocol by default and for all and only PREAPRE&EXECUTE stmts
	we use binary protocol. This is exactly the same as MySQL.
	We don't support MySQL stored procs (and thus nor cursors) so no need for
	protocol stacking or other types of protocols.
  */
  struct MySQLProto*protocol; // current protocol
  struct MySQLProto*protocol_text, *protocol_binary;

  StringInfoData m_connection_attributes;
  StringInfoData scramble;
  unsigned int server_status;
  pid_t thread_id;//getpid()
  Security_context security_context;
  struct rand_struct *rand;

  /*
    the db and schema specified at connection packet.
	note that current schema is namespace_search_path, which may differ from conn_schema.
  */
  StringInfoData conn_db;
  StringInfoData conn_schema;

  StringInfoData packet; // network packet sent to the connected client
  size_t bytes_sent, bytes_rcvd;
  enum_server_command usr_cur_cmd; // current user command
  uint16_t peer_port, this_port;
  uint32_t current_statement_cond_count;

  // config vars
  unsigned long long net_wait_timeout,
  		net_read_timeout, net_write_timeout;
  unsigned long net_buffer_length;

  enum_resultset_metadata resultset_metadata;
  enum_killed_state killed_state;
  uint16_t client_charset_num;
  unsigned long max_client_packet_length;
  unsigned long prepared_stmt_count; // also used to generate prepared stmt id
  bool return_tuples;// does current stmt return tuples to client?

  /* True if we should not send OK packet to mysql client(on error the error
    packet should be sent as before). This stmt is translated and returned by
	ddl2kunlun from one utility (DDL) stmt from a mysql client. and this field is
	set to true if the current translated stmt is not last one of the returned stmt(s)
	so that we won't send multiple OK packets for one DDL stmt in a mysql session. */
  bool mysql_ddl_substmts_no_reply;

  StmtStat stmt_stat;
  TxnState txn_state;
  COM_DATA com_data;// command read from protocol
  // Prepared_statement ptrs received and prepared earlier in this session.
  struct List *prepared_stmts;
  struct Port *pg_port;
  struct List *warning_msgs;
  struct List *error_msgs;

} MySQLSession;

extern MySQLSession g_mysql_session;
#define current_thd (&g_mysql_session)

extern void init_thd_common(void);

inline static enum_server_command
thd_get_command(MySQLSession *thd) { return thd->usr_cur_cmd; }

inline static enum_virtio_type thd_get_virtio_type(MySQLSession*thd)
{
  return thd->conn->type;
}

inline static void thd_increment_bytes_sent(MySQLSession*s, size_t length)
{
  s->bytes_sent += length;
}

inline static void thd_increment_bytes_received(MySQLSession*s, size_t length)
{
  s->bytes_rcvd += length;
}

inline static void thd_inc_examined_rows(MySQLSession *thd, unsigned int by)
{
  thd->stmt_stat.examined_row_count += by;
}

inline static void thd_inc_sent_rows(MySQLSession *thd, unsigned int by)
{
  thd->stmt_stat.sent_row_count += by;
}

inline static void thd_set_sent_rows(MySQLSession *thd, unsigned int to)
{
  thd->stmt_stat.sent_row_count = to;
}

inline static void thd_inc_warnings(MySQLSession *thd, unsigned int by)
{
  thd->stmt_stat.num_warnings += by;
}

inline static bool thd_is_ok_status(MySQLSession *thd)
{ return thd->stmt_stat.status == SS_OK; }
inline static bool thd_is_disabled_status(MySQLSession *thd)
{ return thd->stmt_stat.status == SS_DISABLED; }
inline static bool thd_is_eof_status(MySQLSession *thd)
{ return thd->stmt_stat.status == SS_EOF; }
inline static bool thd_is_error_status(MySQLSession *thd)
{ return thd->stmt_stat.status == SS_ERROR; }

inline static bool thd_status_is_set(MySQLSession *thd)
{
  return thd->stmt_stat.status != SS_EMPTY;
}

inline static void thd_disable_status(MySQLSession *thd)
{
  thd->stmt_stat.status = SS_DISABLED;
}

inline static StmtStat*
thd_get_stmt_status(MySQLSession*thd) { return &thd->stmt_stat; }

extern void thd_set_affected_rows(MySQLSession *thd, unsigned long long nrows);

inline static void *thd_alloc(size_t sz)
{ return MemoryContextAlloc(current_thd->memctx, sz); }

inline static void session_assign_user(MySQLSession*thd, StringInfo usrname)
{
  StringInfo user = &thd->security_context.user;
  resetStringInfo(user);
  appendBinaryStringInfo3(user, usrname);
}

extern int thd_prepare_connection(struct Port *port);

inline static void thd_set_multi_results(MySQLSession*thd)
{	thd->server_status |= SERVER_MORE_RESULTS_EXISTS; }

inline static unsigned long long thd_get_net_wait_timeout(MySQLSession *thd)
{ return thd->net_wait_timeout; }

inline static bool thd_util_substmts_no_reply(MySQLSession*thd)
{
	return thd->mysql_ddl_substmts_no_reply;
}

inline static void thd_util_substmts_set_no_reply(MySQLSession*thd, bool b)
{
	thd->mysql_ddl_substmts_no_reply = b;
}

extern  bool thd_set_client_charset(MySQLSession *thd, uint16_t charset_code);

struct DR_printtup;
extern bool
thd_send_result_set_row(MySQLSession *thd, struct TupleTableSlot *tts, struct DR_printtup *myState);
extern bool thd_send_result_metadata(MySQLSession *thd,
	struct List *targetlist, struct tupleDesc* typinfo, unsigned int flags);
extern struct Prepared_statement *
thd_find_prepared_stmt(MySQLSession *thd, unsigned long stmtid);
extern bool HandleMySQLCommand(void );
extern bool thd_connection_alive(MySQLSession *thd);
extern void send_mysql_error(ErrorData *edata);
extern void thd_set_error_status(MySQLSession *thd, unsigned int mysql_errno,
	const char *message_text, const char *returned_sqlstate);
extern void
thd_set_ok_status(MySQLSession*thd, unsigned long long affected_rows,
	unsigned long long id, const char *message);
extern void thd_set_eof_status(MySQLSession *thd);
extern const char *thd_serialize_secctx(MySQLSession *thd);
extern void thd_reply_connected(void );
extern void thd_set_net_io_retry_count(int num_retries);
extern void thd_txn_start(void );
extern void thd_txn_end(bool is_commit/*true: commit; false: abort*/);

extern struct MYSQL_TIME* DatumGetMYSQL_TIME(struct MySQLProto*proto,
	enum enum_field_types fldtype, Datum flddata);
extern Datum MYSQL_TIMEGetDatum(struct MYSQL_TIME* mt);
extern void thd_use_binary_proto(MySQLSession *thd);
extern void thd_use_text_proto(MySQLSession *thd);
extern void thd_add_diagnosis(int level, int mysqlerrcode, int sqlerrcode,
	const char*errmsg);
extern MySQLSession* get_current_thd(void);
extern void thd_set_affected_rows(MySQLSession *thd, unsigned long long nrows);
extern Datum MYSQL_TIMEGetTimestampDatum(struct MYSQL_TIME* mt, bool has_tz);
extern Datum MYSQL_TIMEGetTimeDatum(struct MYSQL_TIME* mt);
extern Datum MYSQL_TIMEGetDateDatum(struct MYSQL_TIME* mt);
extern void thd_end_middle_command(void );
extern void clear_stmt_diagnosis(MySQLSession*thd);
struct _DestReceiver;
extern void ExecShowDiagnoStmt(struct _DestReceiver *dest, int level);
extern void MySQLBeginQuery(int nstmts);
extern void clear_stmt_diagnosis2(void );
extern void PostInitMySQLSession(void);
extern void ExecuteUseStmt(MySQLSession*thd, const char *db_name, size_t length);
#endif // !MYSQL_SESSION_H
