/*-------------------------------------------------------------------------
 *
 * cluster_meta.h
 *	  definitions of types, and declarations of functions, that are used in
 *	  meta data cluster SQL statement send and result receive.
 *
 *
 * Copyright (c) 2019-2021 ZettaDB inc. All rights reserved.
 *
 * This source code is licensed under Apache 2.0 License,
 * combined with Common Clause Condition 1.0, as detailed in the NOTICE file.
 *
 * src/include/sharding/cluster_meta.h
 *
 *-------------------------------------------------------------------------
 */

#ifndef CLUSTER_META_H
#define CLUSTER_META_H

#include "postgres.h"
#include "sharding/mysql/mysql.h"
#include "nodes/nodes.h"
#include "postmaster/bgworker.h"

extern Oid comp_node_id;

#define METADATA_SHARDID 0xFFFFFFFF
#define MAX_META_SHARD_NODES 19

/*
 * What status to check, flag bits for check_mysql_instance_status.
 * */
#define CHECK_KEEP_ALIVE 0x1
#define CHECK_NOT_READONLY 0x2
#define CHECK_IS_READONLY 0x4
#define CHECK_SET_NAMES 0x8
#define CHECK_MASTER 0x10
#define CHECK_HA_ROLE 0x20

#define MAX_HOSTADDR_LEN 128 // align with def in metadata tables.
/*
 * A mysql connection for synchronous communication, currently only used to
 * communicate with mysql metadata cluster. We don't cache connection
 * parameters(user name, password, host, port) here in case of cluster
 * metadata server master switch. We always query syscache for such fields.
 * */
typedef struct MYSQL_CONN
{
	bool connected;
	bool inside_err_hdlr;
	char node_type; // type of connected mysql instance. 1: primary; 0: replica; -1: unknown
	CmdType cmd;
	MYSQL_RES *result;
	int nrows_affected;
	int nwarnings;
	/*
	 * If mysql returns this error number, ignore it, don't report or throw.
	 * for now we only need to ignore one error, make it an array if need more
	 * in future.
	 * */
	int ignore_err;
	MYSQL conn;
	Oid nodeid;
	char addr[MAX_HOSTADDR_LEN];
	int port;
} MYSQL_CONN;

typedef enum enum_rcr_role {
	RCR_ROLE_NONE,
	RCR_ROLE_PRIMARY,
	RCR_ROLE_REPLICA,
	RCR_ROLE_BOTH
} enum_rcr_role;

static inline bool check_rcr_role_valid(int role)
{
	Assert(role == RCR_ROLE_NONE || role == RCR_ROLE_PRIMARY ||
			role == RCR_ROLE_REPLICA || role == RCR_ROLE_BOTH);
	if (role == RCR_ROLE_NONE || role == RCR_ROLE_PRIMARY ||
			role == RCR_ROLE_REPLICA || role == RCR_ROLE_BOTH) return true;
	return false;
}

static inline bool is_rcr_replica(int role)
{
	check_rcr_role_valid(role);
	return role == RCR_ROLE_REPLICA;
}

static inline bool is_rcr_primary(int role)
{
	check_rcr_role_valid(role);
	return role & RCR_ROLE_PRIMARY;
}

extern const char *cluster_rcr_role_str(int rcr_role);
extern int cluster_rcr_role;

typedef enum enum_storage_node_type {
	STORAGE_NODE_TYPE_INVALID,
	STORAGE_NODE_TYPE_KUNLUN_MYSQL,
	STORAGE_NODE_TYPE_KUNLUN_MARIADB,
	STORAGE_NODE_TYPE_KUNLUN_MYSQL_CANTIAN,
	STORAGE_NODE_TYPE_KUNLUN_MARIADB_CANTIAN
} enum_storage_node_type;

struct AsyncStmtInfo;
extern enum_storage_node_type get_storage_node_type(struct AsyncStmtInfo *asi);
extern enum_storage_node_type init_storage_node_type(void);

extern enum_storage_node_type g_storage_node_type;
inline static bool storage_node_is_mysql()
{
	init_storage_node_type();
	return g_storage_node_type == STORAGE_NODE_TYPE_KUNLUN_MYSQL ||
		g_storage_node_type == STORAGE_NODE_TYPE_KUNLUN_MYSQL_CANTIAN;
}
inline static bool storage_node_is_mariadb()
{
	init_storage_node_type();
	return g_storage_node_type == STORAGE_NODE_TYPE_KUNLUN_MARIADB ||
		g_storage_node_type == STORAGE_NODE_TYPE_KUNLUN_MARIADB_CANTIAN;
}
inline static bool storage_node_is_cantian()
{
	init_storage_node_type();
	return g_storage_node_type == STORAGE_NODE_TYPE_KUNLUN_MYSQL_CANTIAN ||
		g_storage_node_type == STORAGE_NODE_TYPE_KUNLUN_MARIADB_CANTIAN;
}

extern const char *default_collation_string(void);

/**
 * @brief Connect to the meta cluster master, and return the internal connection singleton.
 * 
 * @param conn the connection to initialize. NULL if want to use the internal connection to cluster meta.
 * @param isbg  true if in the background process, and through no exception.
 */
extern MYSQL_CONN* connect_to_meta_cluster_master(MYSQL_CONN *conn, bool isbg);
/**
 * @brief Close internal connection singleton
 */
extern void disconnect_metadata_shard(void);

/**
 * @brief  Send SQL statement [stmt, stmt_len) to cluster meta node in sync,
 * 
 * @param isbg true if called by a background process(xidsender), false if 
 * 			called by backend worker processes.
 * @retval if 'isbg' is true, true on error, false on success.
 *         if !isbg, false on success; exception is thrown on error.
 */
extern bool send_stmt_to_cluster_meta(MYSQL_CONN*conn, const char *stmt,
	size_t len, CmdType cmd, bool isbg);
extern void free_metadata_cluster_result(MYSQL_CONN *conn);
extern bool handle_metadata_cluster_result(MYSQL_CONN *conn, bool isbg);
extern void close_metadata_cluster_conn(MYSQL_CONN* conn);

extern int send_stmt_to_cluster_meta_start(MYSQL_CONN *conn,
					   const char *stmt,
					   size_t len,
					   CmdType cmd,
					   bool *err);

extern int send_stmt_to_cluster_meta_cont(MYSQL_CONN *conn,
					  int status,
					  bool *err);
/**
 * @brief Similar to connect_to_meta_cluster_master(), but when isbg is true, this api
 *  will retry again and again until the connection is established. 
 *  If the current master is unavailible, it also ask other nodes who the current master is,
 *  and update the pg_meta_cluster/pg_meta_cluster_nodes, if the master is changed.
 * 
 * @param term  pointer to a flag indating whether current process catch a term signal.
 *  when the it is set to true, the api quit the loop, and return the unfinished connection.
 */
extern MYSQL_CONN* get_meta_cluster_conn(bool isbg, bool (*term)());

extern bool UpdateCurrentMetaShardMasterNodeId(void);
extern void KillMetaShardConn(char type, Oid nodeid, uint32_t connid);

extern void sync_shard_from_meta_cluster(void);
extern void sync_shard_node_from_meta_cluster(void);
extern void sync_meta_nodes_from_meta_cluster(void);
extern void sync_comp_nodes_from_meta_cluster(void);
extern void set_meta_conn_stmt_id(MYSQL_CONN *conn);
#endif // !CLUSTER_META_H
