/*-------------------------------------------------------------------------
 *
 * sharding_conn.h
 *	  definitions of types, and declarations of functions, that are used in
 *	  table sharding connection functionality.
 *
 *
 * Copyright (c) 2019-2021 ZettaDB inc. All rights reserved.
 *
 * This source code is licensed under Apache 2.0 License,
 * combined with Common Clause Condition 1.0, as detailed in the NOTICE file.
 *
 * src/include/sharding/sharding_conn.h
 *
 *-------------------------------------------------------------------------
 */
#ifndef SHARDING_CONN_H
#define SHARDING_CONN_H

#include "sharding/mysql/mysql.h"
#include "sharding/mysql/server/private/sql_cmd.h"
#include "sharding/sharding.h"
#include "sharding/stmt_hdl.h"
#include "nodes/nodes.h"

#include <sys/time.h>

/* GUC options. */
extern int mysql_connect_timeout;
extern int mysql_read_timeout;
extern int mysql_write_timeout;
extern int mysql_max_packet_size;
extern bool mysql_transmit_compress;
extern bool enable_sql_log;
extern bool enable_remote_cursor;
extern int remote_cursor_prefetch_rows;
struct MYSQL;
/**
 * CONN_VALID	: Connection is valid. if not, need to reconnect at next use of the connection.
 * CON_RESET	: Connection is reset. if so, need to resend SET NAMES and cached.
 * 				  mysql session vars before sending any stmt through it
 */
#define CONN_VALID 0x1
#define CONN_RESET 0x2
#define CONN_MASTER 0x4
#define CONN_MVCC 0x8

typedef struct ShardConnection
{
	Oid shard_id;	   // Connections to one shard's all nodes.
	uint8_t num_nodes; // number of payload slots in the 3 arrays below.
	/*
	 * Whenever a connection to storage node is made once, the nodeids[i],
	 * conns[i] and conn_flags[i] always belong to the same storage node.
	 * */
	Oid nodeids[MAX_NODES_PER_SHARD];  // shard node ids. insert in order
	MYSQL *conns[MAX_NODES_PER_SHARD]; // conn[i] is nodeids[i]'s connection.

	char conn_flags[MAX_NODES_PER_SHARD];
	MYSQL conn_objs[MAX_NODES_PER_SHARD]; // append only
} ShardConnection;


/*
 * A communication port with one storage node, which mostly is a master.
 * It should be reset/cleared at start of each statement.
 * */
typedef struct AsyncStmtInfo
{
	Oid shard_id, node_id;
	Oid donor_shard_id, donor_node_id; /* the real shard/node id */
	int status; // mysql async API wait&cont status.
	uint64 client_query_id;

	/*
	 * In this channel, the NO. of stmts executed correctly and got its/their
	 * results.
	 * */
	int executed_stmts;

	/* The mysql client conn */
	MYSQL *conn;

	/*
	 * Inserted/Deleted/Modified rows of INSERT/DELETE/UPDATE stmts executed in
	 * current txn and NOT SET for returned NO. of rows for SELECT.
	 * */
	uint32_t txn_wrows;

	/*
	 * Inserted/Deleted/Modified rows of INSERT/DELETE/UPDATE stmts executed in
	 * current user stmt and NOT SET for returned NO. of rows for SELECT.
	 * */
	uint32_t stmt_wrows;

	/*
	 * NO. of warnings from storage node query execution. this field should
	 * be returned to pg's client, and in pg's equivalent of 'show warnings'
	 * we should fetch warnings from each storage node that executed part of
	 * the last stmt, and assemble them together as final result to client. TODO
	 */
	uint32_t nwarnings;

	/*
	 * Whether write(INSERT/UPDATE/DELETE) and read(SELECT) commands were
	 * executed in current pg stmt. Transaction mgr should accumulate the
	 * group of storage nodes written and read-only by collecting them from
	 * this object at end of each stmt, in order to do 2PC.
	 * */
	bool did_write;
	bool did_read;
	/*
	 * Set if an DDL is executed in this shard in current txn. Note that we use
	 * CMD_UTILITY to denote DDLs as pg does, but CMD_UTILITY includes many
	 * other types of commands, including CALL stmt. Maybe in future we need
	 * to distinguish stored proc/func CALL stmts from DDL stmts.
	 * */
	bool did_ddl;
	
	/* Indicate if a xa in progress on the conn */
	bool txn_in_progress;

	/* True if current conn is not the leader of parallel vnodes */
	bool snapshot_channel;

	/* 
	 * The thread id of the leader connection which generate the snapshot used by
	 * parallel worker process.
	 */
	uint64 snapshot_threadid;

	uint64 visible_version;

	/* The current statment work on */
	StmtHandle *curr_stmt;

	/* The pending statements to send */
	List *stmt_queue;

	/* The handle still in use */
	List *stmt_inuse;

	/* ----- sql logger -------- */
	struct timeval 	sqllog_starttime;
	char sqllog_buf[256];
} AsyncStmtInfo;

inline static bool ASIAccessed(AsyncStmtInfo *asi)
{
	return asi->did_write || asi->did_ddl || asi->did_read;
}

inline static bool ASIReadOnly(AsyncStmtInfo *asi)
{
	return !asi->did_write && !asi->did_ddl && asi->did_read;
}

inline static bool ASIConnected(AsyncStmtInfo *asi)
{
	return asi && asi->conn != NULL;
}

inline static bool ASITxnInProgress(AsyncStmtInfo *asi)
{
	return asi && asi->txn_in_progress;
}

extern void InitShardingSession(void);

extern void ResetCommunicationHub(void);
extern void ResetCommunicationHubStmt(bool ended_clean);

extern AsyncStmtInfo *GetAsyncStmtInfoByIndex(int i);
extern int GetAsyncStmtInfoUsed(void);
extern AsyncStmtInfo *GetAsyncStmtInfo(Oid shardid);
extern AsyncStmtInfo *GetAsyncStmtInfoNode(Oid shardid, Oid shardNodeId, bool req_chk_onfail, bool want_master);

/*
 * In RC mode, reset the global visible version at the start of the command.
 */
extern void ResetAsiGlobalVisibleVersion(void);

/**
 * @brief Append 'stmt' into asi's job queue. 'stmt' will be sent later when its turn comes.
 * 
 *  Same as send_stmt_async(), but not return a handle. Typically used to execute sql 
 *  that does not return results.
 */
extern void send_stmt_async_nowarn(AsyncStmtInfo *asi, char *stmt,
			    size_t stmt_len, CmdType cmd, bool ownsit, enum enum_sql_command sqlcom);

/**
 * @brief Append 'stmt' into asi's job queue. and wait for the completion of 'stmt'
 */
extern void send_remote_stmt_sync(AsyncStmtInfo *asi, char *stmt, size_t len,
				  CmdType cmdtype, bool owns_it, enum enum_sql_command sqlcom, int ignore_err);

/**
 * @brief Send statement to all the shard currently in use, and wait for the completion of the statement
 *
 *   @param written_only Only send to shards which is written is current transaction
 */
extern void send_stmt_to_all_inuse_sync(char *stmt, size_t len, CmdType cmdtype, bool owns_it,
					enum enum_sql_command sqlcom, bool written_only);
/**
 * @brief Send statement to all of the shards in current cluster, and wait for the completion of the statement
 */
extern void send_stmt_to_all_shards_sync(char *stmt, size_t len, CmdType cmdtype, bool owns_it,
					 enum enum_sql_command sqlcom);

/**
 * @brief Wait for all of the statements in queue to be completed
 */
extern void flush_all_stmts(void);

extern AsyncStmtInfo *FindAnyShard(void);

/**
 * @brief Cancel all of the statements in queue, and wait for the completion of the running statements
 * 
 *  Note: 
 *  This is used internally and is mainly used when the asi is finally cleaned up, so even if
 *  the running stmt eventually reports an error, or even if the connection is disconnected,
 *  no exception will be thrown.
 */
extern void cancel_all_stmts(void);

// extern uint64_t GetRemoteAffectedRows(void);
extern bool MySQLQueryExecuted(void);

extern Oid GetCurrentNodeOfShard(Oid shard);

extern bool IsConnMaster(AsyncStmtInfo *asi);
extern bool IsConnReset(AsyncStmtInfo *asi);
extern void disconnect_storage_shards(void);
extern void request_topo_checks_used_shards(void);

#endif // !SHARDING_CONN_H
