#ifndef DEBUG_SYNC_INCLUDED
#define DEBUG_SYNC_INCLUDED

/* Copyright (c) 2009, 2017, Oracle and/or its affiliates. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License, version 2.0,
   as published by the Free Software Foundation.

   This program is also distributed with certain software (including
   but not limited to OpenSSL) that is licensed under separate terms,
   as designated in a particular file or component or in included license
   documentation.  The authors of MySQL hereby grant you an additional
   permission to link the program and your derivative works with the
   separately licensed software that they have included with MySQL.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License, version 2.0, for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301  USA */

/**
  @file

  Declarations for the Debug Sync Facility. See debug_sync.cc for details.
*/

#include <stddef.h>
#include <sys/types.h>

#include "my_inttypes.h"
#include "port_config.h"

#if defined(ENABLE_DEBUG_SYNC)

/* Macro to be put in the code at synchronization points. */
#define DEBUG_SYNC(_sync_point_name_)                 \
  do {                                                       \
	if (unlikely(opt_debug_sync_timeout))                    \
	  debug_sync(STRING_WITH_LEN(_sync_point_name_)); \
  } while (0)

/* Command line options */
extern int opt_debug_sync_timeout;
extern char *debug_sync_cmd;

/* Default WAIT_FOR timeout if command line option is given without argument. */
#define DEBUG_SYNC_DEFAULT_WAIT_TIMEOUT 300

extern Size DebugSyncShmemSize(void);
extern void CreateDebugSyncShmem(void);
/* Debug Sync prototypes. See debug_sync.cc. */
extern void debug_sync_end_global(void);
extern int debug_sync_init(void);
extern void debug_sync_end(int code, Datum arg);
extern void debug_sync(const char *sync_point_name, size_t name_len);
extern bool debug_sync_define_action(const char *action_str, size_t len);
extern void update_debug_sync(const char *val_str, void *extra);
extern const char *show_debug_sync(void);

/**
  This macro simplifies when a DEBUG_INJECT_IF will generate a given
  signal and then will wait for another signal to continue.
*/
#define DBUG_SIGNAL_WAIT_FOR(A, B, C)                          \
  DEBUG_INJECT_IF(A, {                                            \
	const char act[] = "now SIGNAL " B " WAIT_FOR " C;            \
	DBUG_ASSERT(!debug_sync_define_action(STRING_WITH_LEN(act))); \
  };)

#else /* defined(ENABLE_DEBUG_SYNC) */

#define DEBUG_SYNC(_sync_point_name_) /* disabled DEBUG_SYNC */
#define DBUG_SIGNAL_WAIT_FOR(A, B, C) \
  do {                                   \
  } while (0)

#endif /* defined(ENABLE_DEBUG_SYNC) */

#endif /* DEBUG_SYNC_INCLUDED */
