/*-------------------------------------------------------------------------
 *
 * byte_order.h
 *		Kunlun Database MySQL protocol server side implementation.
 *		Architecture independent integer read and write functions.
 *
 * Copyright (c) 2019-2022 ZettaDB inc. All rights reserved.
 *
 * This source code is licensed under Apache 2.0 License,
 * combined with Common Clause Condition 1.0, as detailed in the NOTICE file.
 *
 * IDENTIFICATION
 *	  src/include/libmysql/byte_order.h
 *
 *-------------------------------------------------------------------------
 */
#ifndef MY_BYTEORDER_INCLUDED
#define MY_BYTEORDER_INCLUDED

#include <string.h>
#include <sys/types.h>

#ifdef HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif

#if defined(_MSC_VER)
#include <stdlib.h>
#endif

#if defined(_WIN32) && defined(WIN32_LEAN_AND_MEAN)
#include <winsock2.h>
#endif

#include <stdint.h>

#ifdef WORDS_BIGENDIAN
#include "big_endian.h"  // IWYU pragma: export
#else
#include "little_endian.h"  // IWYU pragma: export
#endif

static inline int32_t read_sint3(const unsigned char *A) {
  return ((int32_t)(((A[2]) & 128)
                      ? (((uint32_t)255L << 24) | (((uint32_t)A[2]) << 16) |
                         (((uint32_t)A[1]) << 8) | ((uint32_t)A[0]))
                      : (((uint32_t)A[2]) << 16) | (((uint32_t)A[1]) << 8) |
                            ((uint32_t)A[0])));
}

static inline uint32_t read_uint3(const unsigned char *A) {
  return (uint32_t)(((uint32_t)(A[0])) + (((uint32_t)(A[1])) << 8) +
                  (((uint32_t)(A[2])) << 16));
}

static inline unsigned long long read_uint5(const unsigned char *A) {
  return ((unsigned long long)(((uint32_t)(A[0])) + (((uint32_t)(A[1])) << 8) +
                      (((uint32_t)(A[2])) << 16) + (((uint32_t)(A[3])) << 24)) +
          (((unsigned long long)(A[4])) << 32));
}

static inline unsigned long long read_uint6(const unsigned char *A) {
  return ((unsigned long long)(((uint32_t)(A[0])) + (((uint32_t)(A[1])) << 8) +
                      (((uint32_t)(A[2])) << 16) + (((uint32_t)(A[3])) << 24)) +
          (((unsigned long long)(A[4])) << 32) + (((unsigned long long)(A[5])) << 40));
}

/**
  int3store

  Stores an unsinged integer in a platform independent way

  @param T  The destination buffer. Must be at least 3 bytes long
  @param A  The integer to store.

  _Example:_
  A @ref a_protocol_type_int3 "int \<3\>" with the value 1 is stored as:
  ~~~~~~~~~~~~~~~~~~~~~
  01 00 00
  ~~~~~~~~~~~~~~~~~~~~~
*/
static inline void write_int3(unsigned char *T, unsigned int A) {
  *(T) = (unsigned char)(A);
  *(T + 1) = (unsigned char)(A >> 8);
  *(T + 2) = (unsigned char)(A >> 16);
}

static inline void write_int5(unsigned char *T, unsigned long long A) {
  *(T) = (unsigned char)(A);
  *(T + 1) = (unsigned char)(A >> 8);
  *(T + 2) = (unsigned char)(A >> 16);
  *(T + 3) = (unsigned char)(A >> 24);
  *(T + 4) = (unsigned char)(A >> 32);
}

static inline void write_int6(unsigned char *T, unsigned long long A) {
  *(T) = (unsigned char)(A);
  *(T + 1) = (unsigned char)(A >> 8);
  *(T + 2) = (unsigned char)(A >> 16);
  *(T + 3) = (unsigned char)(A >> 24);
  *(T + 4) = (unsigned char)(A >> 32);
  *(T + 5) = (unsigned char)(A >> 40);
}


#endif /* MY_BYTEORDER_INCLUDED */
