/*-------------------------------------------------------------------------
 *
 * little_endian.h
 *		Kunlun Database MySQL protocol server side implementation.
 *		Integer read/write functions for little endian architectures.
 *
 * Copyright (c) 2019-2022 ZettaDB inc. All rights reserved.
 *
 * This source code is licensed under Apache 2.0 License,
 * combined with Common Clause Condition 1.0, as detailed in the NOTICE file.
 *
 * IDENTIFICATION
 *	  src/include/libmysql/little_endian.h
 *
 *-------------------------------------------------------------------------
 */
#ifndef LITTLE_ENDIAN_INCLUDED
#define LITTLE_ENDIAN_INCLUDED

#ifndef MY_BYTEORDER_INCLUDED
#error This file should never be #included directly; use my_byteorder.h.
#endif
#include <stdint.h>
#include <string.h>

/*
  Since the pointers may be misaligned, we cannot do a straight read out of
  them. (It usually works-by-accident on x86 and on modern ARM, but not always
  when the compiler chooses unusual instruction for the read, e.g. LDM on ARM
  or most SIMD instructions on x86.) memcpy is safe and gets optimized to a
  single operation, since the size is small and constant.
*/

static inline int16_t read_sint2(const unsigned char *A) {
  int16_t ret;
  memcpy(&ret, A, sizeof(ret));
  return ret;
}

static inline int32_t read_sint4(const unsigned char *A) {
  int32_t ret;
  memcpy(&ret, A, sizeof(ret));
  return ret;
}

static inline uint16_t read_uint2(const unsigned char *A) {
  uint16_t ret;
  memcpy(&ret, A, sizeof(ret));
  return ret;
}

static inline uint32_t read_uint4(const unsigned char *A) {
  uint32_t ret;
  memcpy(&ret, A, sizeof(ret));
  return ret;
}

static inline unsigned long long read_uint8(const unsigned char *A) {
  unsigned long long ret;
  memcpy(&ret, A, sizeof(ret));
  return ret;
}

static inline int64_t read_sint8(const unsigned char *A) {
  int64_t ret;
  memcpy(&ret, A, sizeof(ret));
  return ret;
}

static inline void write_int2(unsigned char *T, uint16_t A) { memcpy(T, &A, sizeof(A)); }

static inline void write_int4(unsigned char *T, uint32_t A) { memcpy(T, &A, sizeof(A)); }

static inline void write_int7(unsigned char *T, unsigned long long A) { memcpy(T, &A, 7); }

static inline void write_int8(unsigned char *T, unsigned long long A) {
  memcpy(T, &A, sizeof(A));
}

static inline float read_float4(const unsigned char *M) {
  float V;
  memcpy(&V, (M), sizeof(float));
  return V;
}

static inline void write_float4(unsigned char *V, float M) {
  memcpy(V, (&M), sizeof(float));
}

static inline double read_float8(const unsigned char *M) {
  double V;
  memcpy(&V, M, sizeof(double));
  return V;
}

static inline void write_float8(unsigned char *V, double M) {
  memcpy(V, &M, sizeof(double));
}

#endif /* LITTLE_ENDIAN_INCLUDED */
