/*-------------------------------------------------------------------------
 *
 * plugin_auth.h
 *		Kunlun Database MySQL protocol server side implementation.
 *		Authentication plugin.
 *
 * Copyright (c) 2019-2022 ZettaDB inc. All rights reserved.
 *
 * This source code is licensed under Apache 2.0 License,
 * combined with Common Clause Condition 1.0, as detailed in the NOTICE file.
 *
 * IDENTIFICATION
 *	  src/include/libmysql/plugin_auth.h
 *
 *-------------------------------------------------------------------------
 */
#ifndef MYSQL_PLUGIN_AUTH_INCLUDED
#define MYSQL_PLUGIN_AUTH_INCLUDED
/*
  Authentication Plugin API.

  This file defines the API for server authentication plugins.
*/
#include "lib/stringinfo.h"

/* defines for MYSQL_SERVER_AUTH_INFO.password_used */

#define PASSWORD_USED_NO 0
#define PASSWORD_USED_YES 1
#define PASSWORD_USED_NO_MENTION 2

#define MYSQL_AUTHENTICATION_INTERFACE_VERSION 0x0200

/* Authentication flags */

#define AUTH_FLAG_PRIVILEGED_USER_FOR_PASSWORD_CHANGE (1L << 0)
#define AUTH_FLAG_USES_INTERNAL_STORAGE (1L << 1)
struct MYSQL_PLUGIN_VIRTIO;
/**
  Provides server plugin access to authentication information
*/
typedef struct MYSQL_SERVER_AUTH_INFO {
  /**
    User name as sent by the client and shown in USER().
    NULL if the client packet with the user name was not received yet.
  */
  StringInfoData user_name;
  /**
    A corresponding column value from the mysql.user table for the
    matching account name
  */
  StringInfoData auth_string;

  /**
    This only affects the "Authentication failed. Password used: %s"
    error message. has the following values :
    0 : %s will be NO.
    1 : %s will be YES.
    2 : there will be no %s.
    Set it as appropriate or ignore at will.
  */
  int password_used;

  /**
    Set to the name of the connected client host, if it can be resolved,
    or to its IP address otherwise.
  */
  StringInfoData host_or_ip;

} MYSQL_SERVER_AUTH_INFO;

/**
  Function provided by the plugin which should perform authentication (using
  the virtual IO functions if necessary) and return 0 if successful. The plugin can
  also fill the info.authenticated_as field if a different username should be
  used for authorization.
*/
typedef int (*authenticate_user_t)(struct MYSQL_PLUGIN_VIRTIO *myconn,
                                   MYSQL_SERVER_AUTH_INFO *info);

/**
  New plugin API to generate password digest out of authentication string.
  This function will first invoke a service to check for validity of the
  password based on the policies defined and then generate encrypted hash

  @param[out]   outbuf      A buffer provided by server which will hold the
                            authentication string generated by plugin.
  @param[in,out] outbuflen   Length of server provided buffer as IN param and
                            length of plugin generated string as OUT param.
  @param[in]    inbuf       auth string provided by user.
  @param[in]    inbuflen    auth string length.

  @retval  0 OK
  @retval  1 ERROR
*/
typedef int (*generate_authentication_string_t)(char *outbuf,
                                                unsigned int *outbuflen,
                                                const char *inbuf,
                                                unsigned int inbuflen);

/**
  Plugin API to validate password digest.

  @param[in]    inbuf     hash string to be validated.
  @param[in]    buflen    hash string length.

  @retval  0 OK
  @retval  1 ERROR
  */
typedef int (*validate_authentication_string_t)(const char *inbuf,
                                                unsigned int buflen);

/**
  Plugin API to convert scrambled password to binary form
  based on scramble type.

  @param[in]    password          The password hash containing the salt.
  @param[in]    password_len      The length of the password hash.
  @param[in,out] salt             Used as password hash based on the
                                  authentication plugin.
  @param[in,out] salt_len         The length of salt.

  @retval  0 OK
  @retval  1 ERROR
*/
typedef int (*set_salt_t)(const char *password, unsigned int password_len,
                          unsigned char *salt, unsigned char *salt_len);

/**
  Plugin API to compare a clear text password with a stored hash

  @arg hash              pointer to the hashed data
  @arg hash_length       length of the hashed data
  @arg cleartext         pointer to the clear text password
  @arg cleartext_length  length of the cleat text password
  @arg[out] is_error     non-zero in case of error extracting the salt
  @retval 0              the hash was created with that password
  @retval non-zero       the hash was created with a different password
*/
typedef int (*compare_password_with_hash_t)(const char *hash,
                                            unsigned long hash_length,
                                            const char *cleartext,
                                            unsigned long cleartext_length,
                                            int *is_error);
typedef int (*validate_password_t)(const char *inbuf,
                                         unsigned int buflen);
/**
  Server authentication plugin descriptor
*/
typedef struct st_mysql_auth {
  int interface_version; /** version plugin uses */
  /**
    Authentication plugin capabilities
  */
  const unsigned long authentication_flags;
  authenticate_user_t authenticate_user;
  generate_authentication_string_t generate_authentication_string;
  validate_authentication_string_t validate_authentication_string;
  set_salt_t set_salt;
  validate_password_t validate_password;

  compare_password_with_hash_t compare_password_with_hash;
  /**
    A plugin that a client must use for authentication with this server
    plugin. Can be NULL to mean "any plugin".
  */
  StringInfoData plugin_name;
} st_mysql_auth;
#endif
