<?php
defined('BASEPATH') or exit('No direct script access allowed');

class AlarmHandler {
    public $CI;

    public function __construct() {
        $this->CI =& get_instance();
        $this->CI->load->model('Cluster_model');
    }

    /**
     * 获取告警级别
     */
    public function getAlarmLevel($type, $content = '') {
        $levels = $this->getConfig('alarm_levels', "SELECT id, level FROM cluster_alarm_user");

        if ($this->isSpecialCase($type, $content)) {
            return $this->getSpecialAlarmLevel($type);
        }

        return $levels[$type] ?? 'WARNING';
    }

    protected function isSpecialCase($type, $content) {
        return in_array($type, ['comp_node_exception', 'storage_node_exception']) || 
               (strpos($content, 'Exception') !== false);
    }

    protected function getSpecialAlarmLevel($type) {
        return $type === 'comp_node_exception' ? 'FATAL' : 'ERROR';
    }

    /**
     * 获取告警阈值
     */
    protected function getAlarmThreshold($alarm_type) {
        $thresholds = $this->getConfig('alarm_thresholds', "SELECT id, threshold FROM cluster_alarm_user");
        return $thresholds[$alarm_type] ?? null;
    }

    protected function getConfig($cache_key, $sql) {
        $result = $this->CI->Cluster_model->getList($sql);
        $config = array_column($result, 'level', 'id');
        return $config;
    }

    /**
     * 检查是否需要告警升级
     */
    protected function needEscalation($alarm) {
        // 检查未处理时间是否超过阈值
        $sql = "SELECT TIMESTAMPDIFF(MINUTE, occur_timestamp, NOW()) as duration 
                FROM cluster_alarm_info 
                WHERE id = ? AND status = 'unhandled'";
        $result = $this->CI->Cluster_model->getList($sql, [$alarm['id']]);
        
        if (!empty($result)) {
            $duration = $result[0]['duration'];
            $threshold = $this->getAlarmThreshold($alarm['type']);
            
            return $threshold && $duration > $threshold;
        }
        
        return false;
    }

    /**
     * 获取告警升级阈值
     */
    protected function getEscalationThreshold($alarm_type) {
        $thresholds = [
            'FATAL' => 15,  // 15分钟
            'ERROR' => 30,  // 30分钟
            'WARNING' => 60 // 60分钟
        ];
        return $thresholds[$this->getAlarmLevel($alarm_type)] ?? 60;
    }

    /**
     * 升级告警
     */
    protected function upgradeAlarm($alarm) {
        // 更新告警级别
        $levels = ['WARNING' => 'ERROR', 'ERROR' => 'FATAL'];
        $current_level = $alarm['alarm_level'];
        $new_level = $levels[$current_level] ?? $current_level;

        if ($new_level !== $current_level) {
            $sql = "UPDATE cluster_alarm_info SET alarm_level = ? WHERE id = ?";
            $this->CI->Cluster_model->updateList($sql, [$new_level, $alarm['id']]);

            // 记录升级日志
            $this->logAlarmEscalation($alarm, $current_level, $new_level);
        }
    }

    /**
     * 记录告警升级日志
     */
    protected function logAlarmEscalation($alarm, $old_level, $new_level) {
        $log_sql = "INSERT INTO cluster_alarm_push_log 
                    (alarm_type, push_type, content, content_res, create_at) 
                    VALUES (?, 'escalation', ?, ?, ?)";
        $content = json_encode([
            'alarm_id' => $alarm['id'],
            'old_level' => $old_level,
            'new_level' => $new_level
        ]);
        $this->CI->Cluster_model->updateList($log_sql, [
            $alarm['alarm_type'],
            $content,
            'success',
            time()
        ]);
    }

    /**
     * 监控告警性能
     */
    protected function monitorAlarmPerformance($alarm_id) {
        // 记录告警处理延迟
        $sql = "SELECT 
                    occur_timestamp,
                    COALESCE(handle_time, NOW()) as end_time,
                    TIMESTAMPDIFF(SECOND, occur_timestamp, COALESCE(handle_time, NOW())) as duration
                FROM cluster_alarm_info 
                WHERE id = ?";
        $result = $this->CI->Cluster_model->getList($sql, [$alarm_id]);
        
        if (!empty($result)) {
            $performance = $result[0];
            // 记录性能数据
            $this->logPerformanceMetrics('alarm_latency', [
                'alarm_id' => $alarm_id,
                'duration' => $performance['duration'],
                'timestamp' => time()
            ]);
        }
    }

    /**
     * 记录性能指标
     */
    protected function logPerformanceMetrics($metric_type, $data) {
        $log_sql = "INSERT INTO cluster_performance_metrics 
                    (metric_type, metric_data, create_at) 
                    VALUES (?, ?, ?)";
        $this->CI->Cluster_model->updateList($log_sql, [
            $metric_type,
            json_encode($data),
            time()
        ]);
    }
} 