<?php

// 检查 NullLogger 类是否存在，如果不存在则定义
if (!class_exists('Psr\\Log\\NullLogger')) {
    // 先定义LoggerInterface接口
    if (!interface_exists('Psr\\Log\\LoggerInterface')) {
        /**
         * Describes a logger instance.
         */
        interface Psr_Log_LoggerInterface
        {
            /**
             * System is unusable.
             *
             * @param string $message
             * @param array  $context
             *
             * @return void
             */
            public function emergency($message, array $context = array());

            /**
             * Action must be taken immediately.
             *
             * @param string $message
             * @param array  $context
             *
             * @return void
             */
            public function alert($message, array $context = array());

            /**
             * Critical conditions.
             *
             * @param string $message
             * @param array  $context
             *
             * @return void
             */
            public function critical($message, array $context = array());

            /**
             * Runtime errors that do not require immediate action but should typically
             * be logged and monitored.
             *
             * @param string $message
             * @param array  $context
             *
             * @return void
             */
            public function error($message, array $context = array());

            /**
             * Exceptional occurrences that are not errors.
             *
             * @param string $message
             * @param array  $context
             *
             * @return void
             */
            public function warning($message, array $context = array());

            /**
             * Normal but significant events.
             *
             * @param string $message
             * @param array  $context
             *
             * @return void
             */
            public function notice($message, array $context = array());

            /**
             * Interesting events.
             *
             * @param string $message
             * @param array  $context
             *
             * @return void
             */
            public function info($message, array $context = array());

            /**
             * Detailed debug information.
             *
             * @param string $message
             * @param array  $context
             *
             * @return void
             */
            public function debug($message, array $context = array());

            /**
             * Logs with an arbitrary level.
             *
             * @param mixed  $level
             * @param string $message
             * @param array  $context
             *
             * @return void
             */
            public function log($level, $message, array $context = array());
        }
    }

    // 定义 NullLogger 类
    if (!class_exists('Psr\\Log\\AbstractLogger')) {
        /**
         * This is a simple Logger implementation that other Loggers can inherit from.
         */
        class Psr_Log_AbstractLogger implements Psr_Log_LoggerInterface
        {
            /**
             * System is unusable.
             *
             * @param string $message
             * @param array  $context
             *
             * @return void
             */
            public function emergency($message, array $context = array())
            {
                $this->log('emergency', $message, $context);
            }

            /**
             * Action must be taken immediately.
             *
             * @param string $message
             * @param array  $context
             *
             * @return void
             */
            public function alert($message, array $context = array())
            {
                $this->log('alert', $message, $context);
            }

            /**
             * Critical conditions.
             *
             * @param string $message
             * @param array  $context
             *
             * @return void
             */
            public function critical($message, array $context = array())
            {
                $this->log('critical', $message, $context);
            }

            /**
             * Runtime errors that do not require immediate action but should typically
             * be logged and monitored.
             *
             * @param string $message
             * @param array  $context
             *
             * @return void
             */
            public function error($message, array $context = array())
            {
                $this->log('error', $message, $context);
            }

            /**
             * Exceptional occurrences that are not errors.
             *
             * @param string $message
             * @param array  $context
             *
             * @return void
             */
            public function warning($message, array $context = array())
            {
                $this->log('warning', $message, $context);
            }

            /**
             * Normal but significant events.
             *
             * @param string $message
             * @param array  $context
             *
             * @return void
             */
            public function notice($message, array $context = array())
            {
                $this->log('notice', $message, $context);
            }

            /**
             * Interesting events.
             *
             * @param string $message
             * @param array  $context
             *
             * @return void
             */
            public function info($message, array $context = array())
            {
                $this->log('info', $message, $context);
            }

            /**
             * Detailed debug information.
             *
             * @param string $message
             * @param array  $context
             *
             * @return void
             */
            public function debug($message, array $context = array())
            {
                $this->log('debug', $message, $context);
            }

            /**
             * Logs with an arbitrary level.
             *
             * @param mixed  $level
             * @param string $message
             * @param array  $context
             *
             * @return void
             */
            public function log($level, $message, array $context = array())
            {
                // NOOP
            }
        }
    }

    /**
     * This Logger can be used to avoid conditional log calls.
     *
     * Logging should always be optional, and if no logger is provided to your
     * library creating a NullLogger instance to have something to throw logs at
     * is a good way to avoid littering your code with `if ($this->logger) { }`
     * blocks.
     */
    class Psr_Log_NullLogger extends Psr_Log_AbstractLogger
    {
        /**
         * Logs with an arbitrary level.
         *
         * @param mixed  $level
         * @param string $message
         * @param array  $context
         *
         * @return void
         */
        public function log($level, $message, array $context = array())
        {
            // noop
        }
    }

    // 创建别名
    class_alias('Psr_Log_LoggerInterface', 'Psr\\Log\\LoggerInterface');
    class_alias('Psr_Log_AbstractLogger', 'Psr\\Log\\AbstractLogger');
    class_alias('Psr_Log_NullLogger', 'Psr\\Log\\NullLogger');
}

require 'vendor/autoload.php'; // 加载 Composer 自动加载器

use Elastic\Elasticsearch\ClientBuilder;


class Es_model extends CI_Model
{


    public function __construct()
    {
        parent::__construct();

    }


    /**
     * 获取系统以外的全部索引列表
     * 
     * @param string $db_url ES服务器URL（可以包含协议，如https://example.com:9200）
     * @param string $db_name 已弃用参数，保留是为了兼容性
     * @param string $username ES服务器用户名（可选）
     * @param string $password ES服务器密码（可选）
     * @param bool $use_https 是否使用HTTPS（如果URL中未指定协议）
     * @param bool $verify_ssl 是否验证SSL证书
     * @return array 包含所有非系统索引的数组
     */
    public function get_db_list($db_url, $db_name = null, $username = null, $password = null, $use_https = false, $verify_ssl = true)
    {
        // 检查URL是否已包含协议
        if (!preg_match('/^https?:\/\//i', $db_url)) {
            $protocol = $use_https ? 'https://' : 'http://';
            $db_url = $protocol . $db_url;
        }

        $hosts = [$db_url];

        $clientBuilder = ClientBuilder::create()->setHosts($hosts);
        
        // 如果提供了用户名和密码，设置基本认证
        if (!empty($username) && !empty($password)) {
            $clientBuilder->setBasicAuthentication($username, $password);
        }
        
        // 如果是HTTPS但不验证SSL证书
        if (strpos($db_url, 'https://') === 0 && !$verify_ssl) {
            $clientBuilder->setSSLVerification(false);
        }
        
        $client = $clientBuilder->build();

        $indexData = [];
        
        try {
            // 获取所有索引
            $response = $client->indices()->get(['index' => '*']);
            $indices = $response->asArray();
            
            // 遍历所有索引，过滤掉系统索引（通常以 . 开头）
            foreach (array_keys($indices) as $indexName) {
                // 跳过系统索引（以.开头的索引）
                if (substr($indexName, 0, 1) === '.') {
                    continue;
                }
                
                // 获取索引统计信息
                $statsResponse = $client->indices()->stats(['index' => $indexName]);
                $indexStats = $statsResponse->asArray();
                
                // 尝试获取最小的_seq_no（如果可用）
                $minSeqNo = 0; // 默认从0开始
                if (isset($indexStats['indices'][$indexName]['primaries']['seq_no'])) {
                    $minSeqNo = $indexStats['indices'][$indexName]['primaries']['seq_no']['min_seq_no'] - 1;
                    // 减1确保比最小的序列号还小，作为起始点
                    $minSeqNo = max(0, $minSeqNo); // 确保不小于0
                }
                
                // 构建索引同步数据
                $syncData = [
                    'id' => '',
                    'seq_no' => $minSeqNo,
                    'scroll_id' => null,
                    'timestamp' => time()
                ];
                
                $indexData[] = [
                    'binlog_file' =>  $indexName,
                    'binlog_pos' =>"_doc",
                    'gtid_set' => json_encode($syncData)
                ];
            }
        } catch (\Exception $e) {
            log_message('error', 'Elasticsearch error: ' . $e->getMessage());
            // 记录异常，返回空数组
            return [];
        }

        return $indexData;
    }
    
    /**
     * 创建scroll请求并返回第一批结果
     * 
     * @param string $db_url ES服务器URL（可以包含协议，如https://example.com:9200）
     * @param string $db_name 索引名
     * @param mixed $scrollPosition 上次同步位置信息
     * @param string $username ES服务器用户名（可选）
     * @param string $password ES服务器密码（可选）
     * @param bool $use_https 是否使用HTTPS（如果URL中未指定协议）
     * @param bool $verify_ssl 是否验证SSL证书
     * @return array 包含scroll_id和查询结果
     */
    public function create_scroll_request($db_url, $db_name, $scrollPosition = null, $username = null, $password = null, $use_https = false, $verify_ssl = true)
    {
        // 检查URL是否已包含协议
        if (!preg_match('/^https?:\/\//i', $db_url)) {
            $protocol = $use_https ? 'https://' : 'http://';
            $db_url = $protocol . $db_url;
        }

        $hosts = [$db_url];

        $clientBuilder = ClientBuilder::create()->setHosts($hosts);
        
        // 如果提供了用户名和密码，设置基本认证
        if (!empty($username) && !empty($password)) {
            $clientBuilder->setBasicAuthentication($username, $password);
        }
        
        // 如果是HTTPS但不验证SSL证书
        if (strpos($db_url, 'https://') === 0 && !$verify_ssl) {
            $clientBuilder->setSSLVerification(false);
        }
        
        $client = $clientBuilder->build();
            
        $scrollParams = [
            'index' => $db_name,
            'scroll' => '1m', // 保持scroll上下文1分钟
            'body' => [
                'size' => 1000, // 每批次返回1000条
                'sort' => [
                    '_seq_no' => 'asc' // 按序列号排序
                ],
                'query' => [
                    'match_all' => (object)[]
                ]
            ]
        ];
        
        // 如果有上次位置，设置查询条件
        if ($scrollPosition) {
            $position = is_array($scrollPosition) ? $scrollPosition : json_decode($scrollPosition, true);
            if (is_array($position) && isset($position['seq_no']) && $position['seq_no'] >= 0) {
                $scrollParams['body']['query'] = [
                    'range' => [
                        '_seq_no' => [
                            'gt' => intval($position['seq_no']) // 确保是整数
                        ]
                    ]
                ];
            }
        }
        
        try {
            $searchResponse = $client->search($scrollParams);
            $response = $searchResponse->asArray();
            
            return [
                'scroll_id' => '',
                'hits' => $response['hits']['hits'],
                'total' => $response['hits']['total']['value']
            ];
        } catch (\Exception $e) {
            log_message('error', 'Elasticsearch scroll error: ' . $e->getMessage());
            return [
                'scroll_id' => '',
                'hits' => [],
                'total' => 0,
                'error' => $e->getMessage()
            ];
        }
    }



}


