package main

import (
	"bytes"
	"consistent_sql/protocol"
	"consistent_sql/testutils"
	"encoding/binary"
	"encoding/json"
	"flag"
	"fmt"
	"io"
	"io/ioutil"
	"net"
	"os"
	"path/filepath"
	"strings"
	"testing"
	"time"
)

// 保存数据包的函数
func saveESDataPacket(data []byte, messageType string) error {
	// 创建日志目录
	logDir := "elasticsearch_data_packets"
	if err := os.MkdirAll(logDir, 0755); err != nil {
		return fmt.Errorf("创建日志目录失败: %v", err)
	}

	// 生成文件名，包含时间戳和类型
	timestamp := time.Now().Format("20060102_150405.000")
	filename := filepath.Join(logDir, fmt.Sprintf("es_packet_%s_%s.bin", messageType, timestamp))

	// 保存二进制数据
	if err := ioutil.WriteFile(filename, data, 0644); err != nil {
		return fmt.Errorf("保存数据包失败: %v", err)
	}

	// 保存十六进制视图
	hexFilename := filepath.Join(logDir, fmt.Sprintf("es_packet_%s_%s.hex", messageType, timestamp))
	hexView := testutils.FormatPacketHex(data)
	if err := ioutil.WriteFile(hexFilename, []byte(hexView), 0644); err != nil {
		return fmt.Errorf("保存十六进制视图失败: %v", err)
	}

	// 保存JSON视图（如果数据是有效的JSON）
	if testutils.IsValidJSON(data) {
		jsonFilename := filepath.Join(logDir, fmt.Sprintf("es_packet_%s_%s.json", messageType, timestamp))
		var prettyJSON bytes.Buffer
		if err := json.Indent(&prettyJSON, data, "", "  "); err == nil {
			if err := ioutil.WriteFile(jsonFilename, prettyJSON.Bytes(), 0644); err != nil {
				fmt.Printf("保存JSON视图失败: %v\n", err)
			} else {
				fmt.Printf("JSON视图已保存: %s\n", jsonFilename)
			}
		}
	}

	fmt.Printf("数据包已保存: %s (二进制) 和 %s (十六进制视图)\n", filename, hexFilename)
	return nil
}

// 解析并打印ES数据包字段的函数
func analyzeESPacket(data []byte) string {
	if len(data) < 20 {
		return "数据包太短，无法完整解析"
	}

	var result strings.Builder
	result.WriteString("Elasticsearch数据包字段详细解析:\n")
	result.WriteString("================================\n\n")

	offset := 0

	// 1. 解析消息类型
	if offset >= len(data) {
		return result.String() + "数据包不完整，无法解析消息类型"
	}
	messageType := data[offset]
	result.WriteString(fmt.Sprintf("消息类型: 1 字节 (偏移量: %d) 值: %d\n\n", offset, messageType))
	offset++

	// 2. 解析行数量
	if offset+4 > len(data) {
		return result.String() + "数据包不完整，无法解析行数量"
	}
	rowCount := binary.LittleEndian.Uint32(data[offset : offset+4])
	result.WriteString(fmt.Sprintf("行数量: 4 字节 (偏移量: %d-%d) 值: %d\n\n", offset, offset+3, rowCount))
	offset += 4

	// 3. 解析索引ID
	if offset+4 > len(data) {
		return result.String() + "数据包不完整，无法解析索引ID长度"
	}
	idLen := int(binary.LittleEndian.Uint32(data[offset : offset+4]))
	result.WriteString(fmt.Sprintf("索引ID长度: %d 字节 (偏移量: %d-%d) 值: %d\n", 4, offset, offset+3, idLen))
	offset += 4

	if offset+idLen > len(data) {
		return result.String() + "数据包不完整，无法读取索引ID内容"
	}
	indexId := string(data[offset : offset+idLen])
	result.WriteString(fmt.Sprintf("索引ID内容: (偏移量: %d-%d) 值: '%s'\n\n", offset, offset+idLen-1, indexId))
	offset += idLen

	// 4. 解析版本号/序列号
	if offset+4 > len(data) {
		return result.String() + "数据包不完整，无法解析版本号"
	}
	versionNum := int32(binary.LittleEndian.Uint32(data[offset : offset+4]))
	result.WriteString(fmt.Sprintf("版本号: 4 字节 (偏移量: %d-%d) 值: %d\n\n", offset, offset+3, versionNum))
	offset += 4

	// 5. 解析数据类型
	if offset >= len(data) {
		return result.String() + "数据包不完整，无法解析数据类型"
	}
	dataType := int8(data[offset])
	var typeDesc string
	switch dataType {
	case 13:
		typeDesc = "插入"
	case 14:
		typeDesc = "更新"
	case 15:
		typeDesc = "删除"
	default:
		typeDesc = "未知"
	}
	result.WriteString(fmt.Sprintf("数据类型: 1 字节 (偏移量: %d) 值: %d (%s)\n\n", offset, dataType, typeDesc))
	offset++

	// 6. 解析索引名
	if offset+4 > len(data) {
		return result.String() + "数据包不完整，无法解析索引名长度"
	}
	indexNameLen := int(binary.LittleEndian.Uint32(data[offset : offset+4]))
	result.WriteString(fmt.Sprintf("索引名长度: %d 字节 (偏移量: %d-%d) 值: %d\n", 4, offset, offset+3, indexNameLen))
	offset += 4

	if offset+indexNameLen > len(data) {
		return result.String() + "数据包不完整，无法读取索引名内容"
	}
	indexName := string(data[offset : offset+indexNameLen])
	result.WriteString(fmt.Sprintf("索引名内容: (偏移量: %d-%d) 值: '%s'\n\n", offset, offset+indexNameLen-1, indexName))
	offset += indexNameLen

	// 7. 解析文档类型
	if offset+4 > len(data) {
		return result.String() + "数据包不完整，无法解析文档类型长度"
	}
	docTypeLen := int(binary.LittleEndian.Uint32(data[offset : offset+4]))
	result.WriteString(fmt.Sprintf("文档类型长度: %d 字节 (偏移量: %d-%d) 值: %d\n", 4, offset, offset+3, docTypeLen))
	offset += 4

	if offset+docTypeLen > len(data) {
		return result.String() + "数据包不完整，无法读取文档类型内容"
	}
	docType := string(data[offset : offset+docTypeLen])
	result.WriteString(fmt.Sprintf("文档类型内容: (偏移量: %d-%d) 值: '%s'\n\n", offset, offset+docTypeLen-1, docType))
	offset += docTypeLen

	// 8. 解析索引ID（再次写入）
	if offset+4 > len(data) {
		return result.String() + "数据包不完整，无法解析第二个索引ID长度"
	}
	idLen2 := int(binary.LittleEndian.Uint32(data[offset : offset+4]))
	result.WriteString(fmt.Sprintf("索引ID长度(再次): %d 字节 (偏移量: %d-%d) 值: %d\n", 4, offset, offset+3, idLen2))
	offset += 4

	if offset+idLen2 > len(data) {
		return result.String() + "数据包不完整，无法读取第二个索引ID内容"
	}
	indexId2 := string(data[offset : offset+idLen2])
	result.WriteString(fmt.Sprintf("索引ID内容(再次): (偏移量: %d-%d) 值: '%s'\n\n", offset, offset+idLen2-1, indexId2))
	offset += idLen2

	// 9. 解析查询参数（对应MongoDB的SQL值）
	if offset+4 > len(data) {
		return result.String() + "数据包不完整，无法解析查询参数"
	}
	queryParamValue := int32(binary.LittleEndian.Uint32(data[offset : offset+4]))
	result.WriteString(fmt.Sprintf("查询参数值: 4 字节 (偏移量: %d-%d) 值: %d\n\n", offset, offset+3, queryParamValue))
	offset += 4

	// 10. 解析字段数量
	if offset+4 > len(data) {
		return result.String() + "数据包不完整，无法解析字段数量"
	}
	fieldNum := int(binary.LittleEndian.Uint32(data[offset : offset+4]))
	result.WriteString(fmt.Sprintf("字段数量: %d 字节 (偏移量: %d-%d) 值: %d\n\n", 4, offset, offset+3, fieldNum))
	offset += 4

	// 11. 解析字段名
	fieldNames := make([]string, 0, fieldNum)
	result.WriteString("字段名信息:\n")
	for i := 0; i < fieldNum; i++ {
		if offset+4 > len(data) {
			return result.String() + fmt.Sprintf("数据包不完整，无法解析第%d个字段名长度", i+1)
		}
		fieldNameLen := int(binary.LittleEndian.Uint32(data[offset : offset+4]))
		result.WriteString(fmt.Sprintf("  第%d字段名长度: %d 字节 (偏移量: %d-%d) 值: %d\n", i+1, 4, offset, offset+3, fieldNameLen))
		offset += 4

		if offset+fieldNameLen > len(data) {
			return result.String() + fmt.Sprintf("数据包不完整，无法读取第%d个字段名内容", i+1)
		}
		fieldName := string(data[offset : offset+fieldNameLen])
		fieldNames = append(fieldNames, fieldName)
		result.WriteString(fmt.Sprintf("  第%d字段名内容: (偏移量: %d-%d) 值: '%s'\n", i+1, offset, offset+fieldNameLen-1, fieldName))
		offset += fieldNameLen
	}
	result.WriteString("\n")

	// 12. 解析字段属性
	result.WriteString("字段属性信息:\n")
	for i := 0; i < fieldNum; i++ {
		if offset+4 > len(data) {
			return result.String() + fmt.Sprintf("数据包不完整，无法解析第%d个字段属性", i+1)
		}
		fieldAttr := binary.LittleEndian.Uint32(data[offset : offset+4])
		result.WriteString(fmt.Sprintf("  第%d字段属性: %d 字节 (偏移量: %d-%d) 值: %d\n", i+1, 4, offset, offset+3, fieldAttr))
		offset += 4
	}
	result.WriteString("\n")

	// 13. 解析文档数量
	if offset+4 > len(data) {
		return result.String() + "数据包不完整，无法解析文档数量"
	}
	docNum := int(binary.LittleEndian.Uint32(data[offset : offset+4]))
	result.WriteString(fmt.Sprintf("文档数量: %d 字节 (偏移量: %d-%d) 值: %d\n\n", 4, offset, offset+3, docNum))
	offset += 4

	// 14. 解析文档数据
	result.WriteString("文档数据:\n")
	for i := 0; i < docNum; i++ {
		result.WriteString(fmt.Sprintf("  第%d个文档:\n", i+1))
		for j := 0; j < fieldNum; j++ {
			if offset+4 > len(data) {
				return result.String() + fmt.Sprintf("数据包不完整，无法解析第%d个文档第%d个字段的长度", i+1, j+1)
			}
			valueLen := int(binary.LittleEndian.Uint32(data[offset : offset+4]))
			result.WriteString(fmt.Sprintf("    字段'%s'值长度: %d 字节 (偏移量: %d-%d) 值: %d\n",
				fieldNames[j], 4, offset, offset+3, valueLen))
			offset += 4

			if offset+valueLen > len(data) {
				return result.String() + fmt.Sprintf("数据包不完整，无法读取第%d个文档第%d个字段的内容", i+1, j+1)
			}
			value := string(data[offset : offset+valueLen])
			result.WriteString(fmt.Sprintf("    字段'%s'值内容: (偏移量: %d-%d) 值: '%s'\n",
				fieldNames[j], offset, offset+valueLen-1, value))
			offset += valueLen
		}
		result.WriteString("\n")
	}

	// 15. 解析旧值（对于UPDATE操作）
	if dataType == 14 && offset < len(data) {
		result.WriteString("旧值数据 (UPDATE操作):\n")
		for i := 0; i < docNum; i++ {
			result.WriteString(fmt.Sprintf("  第%d个文档旧值:\n", i+1))
			for j := 0; j < fieldNum; j++ {
				if offset+4 > len(data) {
					return result.String() + fmt.Sprintf("数据包不完整，无法解析旧值第%d个文档第%d个字段的长度", i+1, j+1)
				}
				valueLen := int(binary.LittleEndian.Uint32(data[offset : offset+4]))
				result.WriteString(fmt.Sprintf("    字段'%s'旧值长度: %d 字节 (偏移量: %d-%d) 值: %d\n",
					fieldNames[j], 4, offset, offset+3, valueLen))
				offset += 4

				if offset+valueLen > len(data) {
					return result.String() + fmt.Sprintf("数据包不完整，无法读取旧值第%d个文档第%d个字段的内容", i+1, j+1)
				}
				value := string(data[offset : offset+valueLen])
				result.WriteString(fmt.Sprintf("    字段'%s'旧值内容: (偏移量: %d-%d) 值: '%s'\n",
					fieldNames[j], offset, offset+valueLen-1, value))
				offset += valueLen
			}
			result.WriteString("\n")
		}
	}

	// 添加剩余未解析的字节大小提示
	if offset < len(data) {
		result.WriteString(fmt.Sprintf("注意: 数据包中还有 %d 字节未解析\n", len(data)-offset))
	}

	return result.String()
}

// parseESStreamFormat 解析ES数据流的简化格式
func parseESStreamFormat(data []byte) {
	if len(data) < 10 {
		fmt.Println("数据太短，无法解析")
		return
	}

	// 首先偏移量从0开始
	offset := 0

	// 解析消息类型
	messageType := data[offset]
	fmt.Printf("消息类型: %d\n", messageType)
	offset++

	// 解析行数量
	if offset+4 > len(data) {
		fmt.Println("数据不完整，无法解析行数量")
		return
	}
	rowCount := binary.LittleEndian.Uint32(data[offset : offset+4])
	fmt.Printf("行数量: %d\n", rowCount)
	offset += 4

	// 解析索引ID
	if offset+4 > len(data) {
		fmt.Println("数据不完整，无法解析索引ID长度")
		return
	}
	idLen := int(binary.LittleEndian.Uint32(data[offset : offset+4]))
	offset += 4
	if offset+idLen > len(data) {
		fmt.Println("数据不完整，无法读取索引ID")
		return
	}
	indexId := string(data[offset : offset+idLen])
	offset += idLen

	// 解析版本号（ES中用于替代MongoDB的binlog文件）
	if offset+4 > len(data) {
		fmt.Println("数据不完整，无法解析版本号")
		return
	}
	versionNum := int32(binary.LittleEndian.Uint32(data[offset : offset+4]))
	offset += 4

	// 解析数据类型
	if offset >= len(data) {
		fmt.Println("数据不完整，无法解析数据类型")
		return
	}
	dataType := int8(data[offset])
	var typeStr string
	switch dataType {
	case 13:
		typeStr = "插入(13)"
	case 14:
		typeStr = "更新(14)"
	case 15:
		typeStr = "删除(15)"
	default:
		typeStr = fmt.Sprintf("未知(%d)", dataType)
	}
	offset++

	// 解析索引名
	if offset+4 > len(data) {
		fmt.Println("数据不完整，无法解析索引名长度")
		return
	}
	indexNameLen := int(binary.LittleEndian.Uint32(data[offset : offset+4]))
	offset += 4
	if offset+indexNameLen > len(data) {
		fmt.Println("数据不完整，无法读取索引名")
		return
	}
	indexName := string(data[offset : offset+indexNameLen])
	offset += indexNameLen

	// 解析文档类型
	if offset+4 > len(data) {
		fmt.Println("数据不完整，无法解析文档类型长度")
		return
	}
	docTypeLen := int(binary.LittleEndian.Uint32(data[offset : offset+4]))
	offset += 4
	if offset+docTypeLen > len(data) {
		fmt.Println("数据不完整，无法读取文档类型")
		return
	}
	docType := string(data[offset : offset+docTypeLen])

	fmt.Printf("解析结果 - 索引: %s, 文档类型: %s, 索引ID: %s, 操作类型: %s, 版本号: %d\n",
		indexName, docType, indexId, typeStr, versionNum)
}

// TestESParseStreamFormat 测试ES字符串流格式解析功能
func TestESParseStreamFormat(t *testing.T) {
	// 创建一个模拟的ES数据流
	indexName := "users"
	docType := "_doc"
	indexId := "user_123456"

	// 构建模拟数据
	data := make([]byte, 0)

	// 1. 添加消息类型
	messageType := byte(protocol.MRsprowmsg) // 使用正确的消息类型
	data = append(data, messageType)

	// 2. 添加行数
	rowCount := uint32(1)
	lenBytes := make([]byte, 4)
	binary.LittleEndian.PutUint32(lenBytes, rowCount)
	data = append(data, lenBytes...)

	// 3. 添加索引ID长度和索引ID
	indexIdBytes := []byte(indexId)
	indexIdLen := uint32(len(indexIdBytes))
	lenBytes = make([]byte, 4)
	binary.LittleEndian.PutUint32(lenBytes, indexIdLen)
	data = append(data, lenBytes...)
	data = append(data, indexIdBytes...)

	// 4. 添加版本号
	versionNum := int32(1)
	lenBytes = make([]byte, 4)
	binary.LittleEndian.PutUint32(lenBytes, uint32(versionNum))
	data = append(data, lenBytes...)

	// 5. 添加数据类型
	dataType := int8(13) // 13表示插入操作
	data = append(data, byte(dataType))

	// 6. 添加索引名长度和索引名
	indexNameBytes := []byte(indexName)
	indexNameLen := uint32(len(indexNameBytes))
	lenBytes = make([]byte, 4)
	binary.LittleEndian.PutUint32(lenBytes, indexNameLen)
	data = append(data, lenBytes...)
	data = append(data, indexNameBytes...)

	// 7. 添加文档类型长度和文档类型
	docTypeBytes := []byte(docType)
	docTypeLen := uint32(len(docTypeBytes))
	lenBytes = make([]byte, 4)
	binary.LittleEndian.PutUint32(lenBytes, docTypeLen)
	data = append(data, lenBytes...)
	data = append(data, docTypeBytes...)

	// 8. 再次添加索引ID长度和索引ID
	lenBytes = make([]byte, 4)
	binary.LittleEndian.PutUint32(lenBytes, indexIdLen)
	data = append(data, lenBytes...)
	data = append(data, indexIdBytes...)

	// 9. 添加查询参数（ES中为0）
	queryValue := int32(0)
	lenBytes = make([]byte, 4)
	binary.LittleEndian.PutUint32(lenBytes, uint32(queryValue))
	data = append(data, lenBytes...)

	// 10. 添加字段数量
	fieldNum := uint32(4) // ES文档有4个字段
	lenBytes = make([]byte, 4)
	binary.LittleEndian.PutUint32(lenBytes, fieldNum)
	data = append(data, lenBytes...)

	// 11. 添加字段名
	fieldNames := []string{"_id", "name", "email", "age"}
	for _, fieldName := range fieldNames {
		fieldNameBytes := []byte(fieldName)
		fieldNameLen := uint32(len(fieldNameBytes))
		lenBytes = make([]byte, 4)
		binary.LittleEndian.PutUint32(lenBytes, fieldNameLen)
		data = append(data, lenBytes...)
		data = append(data, fieldNameBytes...)
	}

	// 12. 添加字段属性
	for i := 0; i < int(fieldNum); i++ {
		attr := uint32(0)
		lenBytes = make([]byte, 4)
		binary.LittleEndian.PutUint32(lenBytes, attr)
		data = append(data, lenBytes...)
	}

	// 13. 添加文档数
	docNum := uint32(1)
	lenBytes = make([]byte, 4)
	binary.LittleEndian.PutUint32(lenBytes, docNum)
	data = append(data, lenBytes...)

	// 14. 添加文档数据
	docData := []string{indexId, "张三", "zhangsan@example.com", "30"}
	for _, value := range docData {
		valueBytes := []byte(value)
		valueLen := uint32(len(valueBytes))
		lenBytes = make([]byte, 4)
		binary.LittleEndian.PutUint32(lenBytes, valueLen)
		data = append(data, lenBytes...)
		data = append(data, valueBytes...)
	}

	// 测试parseESStreamFormat函数
	fmt.Println("测试parseESStreamFormat函数:")
	parseESStreamFormat(data)

	// 详细解析数据包
	fmt.Println("\n详细解析ES数据包:")
	analysis := analyzeESPacket(data)
	fmt.Println(analysis)

	// 保存测试数据包
	saveESDataPacket(data, "test_sample")
}

// parseESDataPacket 按顺序解析ES数据包的所有字段
func parseESDataPacket(data []byte) {
	if len(data) < 10 {
		fmt.Println("数据包太短，无法解析")
		return
	}

	offset := 0
	fmt.Println("\nElasticsearch数据包字段解析:")
	fmt.Println("============================")

	// 1. 消息类型 (byte)
	messageType := data[offset]
	fmt.Printf("1. 消息类型: byte (偏移量:%d) = %d (protocol.MRsprowmsg)\n", offset, messageType)
	offset++

	// 2. 行数量 (uint32)
	if offset+4 > len(data) {
		fmt.Println("数据包不完整，无法解析行数量")
		return
	}
	rowCount := binary.LittleEndian.Uint32(data[offset : offset+4])
	fmt.Printf("2. 行数量: uint32 (偏移量:%d-%d) = %d\n", offset, offset+3, rowCount)
	offset += 4

	// 3. 索引ID (len+str)
	if offset+4 > len(data) {
		fmt.Println("数据包不完整，无法解析索引ID长度")
		return
	}
	idLen := int(binary.LittleEndian.Uint32(data[offset : offset+4]))
	fmt.Printf("3. 索引ID长度: uint32 (偏移量:%d-%d) = %d\n", offset, offset+3, idLen)
	offset += 4
	if offset+idLen > len(data) {
		fmt.Println("数据包不完整，无法读取索引ID内容")
		return
	}
	indexId := string(data[offset : offset+idLen])
	fmt.Printf("   索引ID内容: string (偏移量:%d-%d) = '%s'\n", offset, offset+idLen-1, indexId)
	offset += idLen

	// 4. 版本号 (int32)
	if offset+4 > len(data) {
		fmt.Println("数据包不完整，无法解析版本号")
		return
	}
	versionNum := int32(binary.LittleEndian.Uint32(data[offset : offset+4]))
	fmt.Printf("4. 版本号: int32 (偏移量:%d-%d) = %d\n", offset, offset+3, versionNum)
	offset += 4

	// 5. 数据类型 (int8)
	if offset >= len(data) {
		fmt.Println("数据包不完整，无法解析数据类型")
		return
	}
	dataType := int8(data[offset])
	var typeStr string
	switch dataType {
	case 13:
		typeStr = "插入(insert)"
	case 14:
		typeStr = "更新(update)"
	case 15:
		typeStr = "删除(delete)"
	default:
		typeStr = fmt.Sprintf("未知(%d)", dataType)
	}
	fmt.Printf("5. 数据类型: int8 (偏移量:%d) = %d (%s)\n", offset, dataType, typeStr)
	offset++

	// 6. 索引名 (len+str)
	if offset+4 > len(data) {
		fmt.Println("数据包不完整，无法解析索引名长度")
		return
	}
	indexNameLen := int(binary.LittleEndian.Uint32(data[offset : offset+4]))
	fmt.Printf("6. 索引名长度: uint32 (偏移量:%d-%d) = %d\n", offset, offset+3, indexNameLen)
	offset += 4
	if offset+indexNameLen > len(data) {
		fmt.Println("数据包不完整，无法读取索引名内容")
		return
	}
	indexName := string(data[offset : offset+indexNameLen])
	fmt.Printf("   索引名内容: string (偏移量:%d-%d) = '%s'\n", offset, offset+indexNameLen-1, indexName)
	offset += indexNameLen

	// 7. 文档类型 (len+str)
	if offset+4 > len(data) {
		fmt.Println("数据包不完整，无法解析文档类型长度")
		return
	}
	docTypeLen := int(binary.LittleEndian.Uint32(data[offset : offset+4]))
	fmt.Printf("7. 文档类型长度: uint32 (偏移量:%d-%d) = %d\n", offset, offset+3, docTypeLen)
	offset += 4
	if offset+docTypeLen > len(data) {
		fmt.Println("数据包不完整，无法读取文档类型内容")
		return
	}
	docType := string(data[offset : offset+docTypeLen])
	fmt.Printf("   文档类型内容: string (偏移量:%d-%d) = '%s'\n", offset, offset+docTypeLen-1, docType)
	offset += docTypeLen

	// 剩余字段继续解析
	// ...省略后续代码，与analyzeESPacket相同...
	fmt.Printf("剩余数据解析详情请查看分析报告\n")
}

// TestESStreamFormat 主测试函数，测试与ES的连接和数据传输
func TestESStreamFormat(t *testing.T) {
	// 定义并解析命令行参数
	serverUnixFile := flag.String("serverUnixFile", "/Users/helay/Documents/GitHub/Kunlun-XPanel/KunlunMonitor/go/consistent_sql/socket.sock", "Unix socket 文件路径")
	workType := flag.String("workType", "create", "任务类型：create 或 status")
	jobId := flag.String("jobId", fmt.Sprintf("es_test_%d", time.Now().Unix()), "任务ID")
	esURL := flag.String("esURL", "http://192.168.0.16:9200", "Elasticsearch URL")
	esIndex := flag.String("esIndex", "kunlun2", "Elasticsearch 索引名")
	esType := flag.String("esType", "_doc", "Elasticsearch 文档类型")
	flag.Parse()

	fmt.Printf("连接到 Unix socket: %s\n", *serverUnixFile)
	fmt.Printf("任务ID: %s, 操作类型: %s\n", *jobId, *workType)
	fmt.Println("测试将持续运行直到被手动中断 (按 Ctrl+C 终止)")

	// 连接到 Unix socket
	conn, err := net.Dial("unix", *serverUnixFile)
	if err != nil {
		t.Fatalf("连接到Source失败: %v", err)
	}
	defer conn.Close()

	// 构建ES任务请求
	esTask := map[string]interface{}{
		"job_id":       *jobId,
		"url":          *esURL,
		"db_user":      "",
		"db_passwd":    "",
		"db_database":  *esIndex,
		"db_table":     *esType,
		"dump_db_type": "es", // 保持使用"es"作为数据库类型
		"gtid":         "{\"id\":\"\",\"seq_no\":0,\"scroll_id\":null,\"timestamp\":1745913513}",
		"event_type":   *workType,
		"index_type":   "_doc", // 添加额外的索引类型参数
		"batch_size":   1,      // 添加批处理大小
		"scroll_time":  "1m",   // 添加滚动时间参数
	}

	// 如果是status请求，只需要提供job_id
	if *workType == "status" {
		esTask = map[string]interface{}{
			"job_id":       *jobId,
			"dump_db_type": "es", // 使用与create请求相同的值
			"event_type":   "status",
		}
	}

	taskData, _ := json.Marshal(esTask)
	fmt.Printf("发送任务数据: %s\n", string(taskData))

	// 构建消息
	requestMsg := &protocol.Message{
		Type: protocol.MReqdumpjob,
		Data: string(taskData),
	}

	marshal, err := json.Marshal(requestMsg)
	if err != nil {
		t.Fatalf("序列化消息失败: %v", err)
	}

	// 发送请求
	if err := protocol.SendMsg(conn, marshal); err != nil {
		t.Fatalf("发送数据到Source失败: %v", err)
	}

	fmt.Println("已发送请求，等待响应...")

	// 循环读取响应
	messageCount := 0
	reconnectAttempts := 0
	maxReconnectAttempts := 5
	reconnectDelay := 5 * time.Second

	for {
		if err := readAndProcessESMsg(conn); err != nil {
			// 检查是否为连接断开或网络错误
			isConnectionError := false
			if err == io.EOF || strings.Contains(err.Error(), "use of closed network connection") {
				isConnectionError = true
				fmt.Printf("连接断开: %v，准备重新连接...\n", err)
			} else if ne, ok := err.(net.Error); ok && (ne.Timeout() || ne.Temporary()) {
				isConnectionError = true
				fmt.Printf("网络错误: %v，准备重新连接...\n", err)
			} else {
				fmt.Printf("处理消息出错: %v\n", err)
			}

			// 仅在首次连接出错且不是连接问题时终止测试
			if messageCount == 0 && !isConnectionError {
				t.Fatalf("首次处理消息出错(非连接问题): %v", err)
			}

			if isConnectionError {
				// 检查重连次数限制
				reconnectAttempts++
				if reconnectAttempts > maxReconnectAttempts {
					fmt.Printf("重连次数达到上限(%d次)，测试终止\n", maxReconnectAttempts)
					t.Fatalf("重连失败次数过多: %v", err)
				}

				// 关闭旧连接
				conn.Close()

				fmt.Printf("第%d次尝试重新连接...\n", reconnectAttempts)
				fmt.Printf("等待%v后重试...\n", reconnectDelay)
				time.Sleep(reconnectDelay)

				// 尝试重新连接
				newConn, reconnErr := net.Dial("unix", *serverUnixFile)
				if reconnErr != nil {
					fmt.Printf("重新连接失败: %v，将在%v后重试\n", reconnErr, reconnectDelay)
					continue
				}

				conn = newConn
				defer conn.Close()
				fmt.Printf("重新连接成功，地址: %s\n", conn.RemoteAddr().String())

				// 重新发送任务请求
				if err := protocol.SendMsg(conn, marshal); err != nil {
					fmt.Printf("重新发送任务请求失败: %v，将在%v后重试\n", err, reconnectDelay)
					continue
				}

				fmt.Println("已重新发送请求，继续监听...")
				// 重置连接尝试计数
				reconnectAttempts = 0
			}
		} else {
			// 成功处理消息，重置重连计数
			reconnectAttempts = 0
			messageCount++
			fmt.Printf("已成功接收 %d 条消息\n", messageCount)
		}

		// 给系统一点时间处理，避免CPU占用过高
		time.Sleep(100 * time.Millisecond)
	}
}

// readAndProcessESMsg 负责读取和处理从 Unix socket 接收到的ES消息
func readAndProcessESMsg(conn net.Conn) error {
	// 设置读取超时
	conn.SetReadDeadline(time.Now().Add(10 * time.Minute)) // 增加超时时间到10分钟
	defer conn.SetReadDeadline(time.Time{})                // 重置为不超时

	// 读取消息长度的4个字节
	lengthBytes := make([]byte, 4)
	bytesRead, err := io.ReadFull(conn, lengthBytes)
	if err != nil {
		// 检查是否为网络错误
		if ne, ok := err.(net.Error); ok && ne.Timeout() {
			return fmt.Errorf("读取消息长度超时: %w", err)
		}

		// 检查连接是否已关闭
		if err == io.EOF || strings.Contains(err.Error(), "use of closed network connection") {
			fmt.Printf("客户端连接已断开: %v，客户端地址: %s\n", err, conn.RemoteAddr().String())
			return fmt.Errorf("连接已关闭或远程主机断开连接: %w", err)
		}

		return fmt.Errorf("读取消息长度失败: %w", err)
	}

	fmt.Printf("读取到长度字节: %d 字节 [%X %X %X %X]\n", bytesRead,
		lengthBytes[0], lengthBytes[1], lengthBytes[2], lengthBytes[3])

	// 解析消息长度
	length := binary.LittleEndian.Uint32(lengthBytes)
	fmt.Printf("消息长度解析为: %d 字节\n", length)

	// 检查消息长度是否合理（增加到50MB）
	maxSize := uint32(50 * 1024 * 1024) // 50MB
	if length > maxSize {
		return fmt.Errorf("消息长度异常: %d 字节 (超过最大限制 %d 字节)", length, maxSize)
	}

	if length == 0 {
		fmt.Printf("警告: 解析到的消息长度为0，可能是协议不匹配或客户端已断开\n")
		// 尝试读取一些额外字节看看有什么
		extraBytes := make([]byte, 16)
		n, extraErr := conn.Read(extraBytes)
		if extraErr == nil && n > 0 {
			fmt.Printf("尝试读取的额外字节: %X\n", extraBytes[:n])
		} else if extraErr != nil {
			if extraErr == io.EOF || strings.Contains(extraErr.Error(), "use of closed network connection") {
				fmt.Printf("客户端连接已关闭: %v\n", extraErr)
			} else {
				fmt.Printf("读取额外字节出错: %v\n", extraErr)
			}
		}
		return fmt.Errorf("消息长度为0，无法处理")
	}

	// 分配缓冲区
	data := make([]byte, length)
	fmt.Printf("准备读取 %d 字节的消息数据...\n", length)

	// 使用 io.ReadFull 确保读取完整的数据
	actualRead, err := io.ReadFull(conn, data)
	if err != nil {
		// 检查是否为网络错误
		if ne, ok := err.(net.Error); ok && ne.Timeout() {
			return fmt.Errorf("读取消息数据超时: %w", err)
		}

		// 检查连接是否已关闭
		if err == io.EOF || strings.Contains(err.Error(), "use of closed network connection") {
			fmt.Printf("读取数据时客户端连接已断开: %v，客户端地址: %s\n", err, conn.RemoteAddr().String())
			return fmt.Errorf("连接已关闭或远程主机断开连接: %w", err)
		}

		fmt.Printf("读取出错，已读取 %d/%d 字节\n", actualRead, length)
		if actualRead > 0 {
			// 避免使用min函数
			previewLen := actualRead
			if previewLen > 32 {
				previewLen = 32
			}
			fmt.Printf("已读取部分数据: %X\n", data[:previewLen])
		}
		return fmt.Errorf("读取消息数据失败: %w", err)
	}

	fmt.Printf("成功读取 %d/%d 字节的消息数据\n", actualRead, length)

	// 打印前几个字节的十六进制表示，帮助调试
	previewSize := len(data)
	if previewSize > 32 {
		previewSize = 32
	}
	fmt.Printf("消息数据前 %d 字节: %X\n", previewSize, data[:previewSize])

	// 检查消息的第一个字节，确定消息类型
	if len(data) > 0 {
		messageTypeByte := data[0]
		fmt.Printf("消息类型字节: %d (0x%X)\n", messageTypeByte, messageTypeByte)

		// 检查是否为二进制协议消息
		if messageTypeByte == byte(protocol.MReqrowmsg) {
			// 这是一个二进制数据行消息
			fmt.Printf("收到MReqrowmsg (二进制数据行消息)，长度: %d 字节\n", len(data))

			// 保存完整的数据流
			if err := saveESDataPacket(data, "binary"); err != nil {
				fmt.Printf("保存数据包失败: %v\n", err)
			}

			// 创建并保存详细的字段解析
			analysis := analyzeESPacket(data)
			analysisFilename := filepath.Join("elasticsearch_data_packets",
				fmt.Sprintf("es_packet_analysis_%s.txt", time.Now().Format("20060102_150405.000")))
			if err := ioutil.WriteFile(analysisFilename, []byte(analysis), 0644); err != nil {
				fmt.Printf("保存字段分析失败: %v\n", err)
			} else {
				fmt.Printf("数据包字段分析已保存到: %s\n", analysisFilename)
			}

			// 在控制台显示前500个字符的分析结果
			analysisPreview := analysis
			if len(analysisPreview) > 500 {
				analysisPreview = analysisPreview[:500] + "...(更多内容见文件)"
			}
			fmt.Println("\n数据包分析预览:")
			fmt.Println(analysisPreview)

			// 确认消息采用SendResponse帮助函数确保格式一致
			fmt.Println("发送确认消息...")
			if err := protocol.SendResponse(conn, protocol.MRsprowmsg, "received"); err != nil {
				if err == io.EOF || strings.Contains(err.Error(), "use of closed network connection") {
					fmt.Printf("发送确认消息时客户端已断开连接: %v，客户端地址: %s\n", err, conn.RemoteAddr().String())
				} else {
					fmt.Printf("发送确认消息失败: %v\n", err)
				}
			} else {
				fmt.Println("确认消息已发送，等待下一包数据...")
			}

			return nil
		} else {
			// 如果不是MReqrowmsg，记录接收到的消息类型
			fmt.Printf("收到非MReqrowmsg消息，消息类型: %d (0x%X)\n", messageTypeByte, messageTypeByte)

			// 尝试匹配已知的消息类型
			typeName := "未知"
			switch messageTypeByte {
			case byte(protocol.MReqdumpjob):
				typeName = "MReqdumpjob"
			case byte(protocol.MRspdumpjob):
				typeName = "MRspdumpjob"
			case byte(protocol.MRspstatemsg):
				typeName = "MRspstatemsg"
			case byte(protocol.MReqrowmsg):
				typeName = "MReqrowmsg"
			case byte(protocol.MRsprowmsg):
				typeName = "MRsprowmsg"
			}
			fmt.Printf("消息类型对应: %s\n", typeName)
		}
	}

	// 如果不是二进制消息，尝试解析为JSON
	fmt.Println("尝试解析为JSON消息...")

	// 先查看数据是否可以打印为字符串
	printableData := string(data)
	if len(printableData) > 200 {
		printableData = printableData[:200] + "...(更多字符省略)"
	}
	fmt.Println("收到消息:", printableData)

	// 检查是否为JSON格式
	if !testutils.IsValidJSON(data) {
		fmt.Printf("收到非JSON格式数据，长度: %d 字节\n", len(data))
		// 保存数据以便后续分析
		if err := saveESDataPacket(data, "unknown"); err != nil {
			fmt.Printf("保存数据包失败: %v\n", err)
		}
		return nil
	}

	// 解析JSON消息
	var message protocol.Message
	if err := json.Unmarshal(data, &message); err != nil {
		fmt.Printf("无法解析的消息内容: %s\n", string(data))
		return fmt.Errorf("解析消息失败: %w", err)
	}

	// 处理不同类型的消息
	fmt.Printf("成功解析为JSON消息，类型: %d\n", message.Type)
	switch message.Type {
	case protocol.MRspdumpjob:
		fmt.Println("收到任务响应:", message.Data)
		saveESDataPacket([]byte(message.Data), "response")
	case protocol.MRspstatemsg:
		fmt.Println("收到状态响应:", message.Data)
		saveESDataPacket([]byte(message.Data), "status")
	case protocol.MReqrowmsg:
		fmt.Println("收到MReqrowmsg JSON消息:", message.Data)

		// 确认消息采用SendResponse帮助函数确保格式一致
		if err := protocol.SendResponse(conn, protocol.MRsprowmsg, "received"); err != nil {
			if err == io.EOF || strings.Contains(err.Error(), "use of closed network connection") {
				fmt.Printf("发送确认消息时客户端已断开连接: %v，客户端地址: %s\n", err, conn.RemoteAddr().String())
			} else {
				fmt.Printf("发送确认消息失败: %v\n", err)
			}
		} else {
			fmt.Println("确认消息已发送，等待下一包数据...")
		}

		saveESDataPacket([]byte(message.Data), "row_message")
	default:
		fmt.Println("收到未知类型JSON消息:", message.Type)
		saveESDataPacket([]byte(message.Data), "unknown_json")
	}

	return nil
}
