package model

import (
	"time"

	"go.mongodb.org/mongo-driver/bson/primitive"
)

// ESGTID Elasticsearch同步的GTID结构
type ESGTID struct {
	ID        string `json:"id"`        // 文档ID
	SeqNo     int64  `json:"seq_no"`    // 序列号
	ScrollID  string `json:"scroll_id"` // scroll ID
	Timestamp int64  `json:"timestamp"` // 时间戳
}

type SyncResult struct {
	LastID primitive.ObjectID
	Data   []MongoSourceDBData `json:"data"`
}

type TDSourceDBData struct {
	Ts     int64       `json:"ts"`     // 时间戳
	Data   interface{} `json:"data"`   // 数据JSON字符串
	Optype string      `json:"optype"` // 操作类型
}

// SyncTdResult 同步结果
type SyncTdResult struct {
	LastTS int64            `json:"last_ts"` // 最后同步的时间戳
	Data   []TDSourceDBData `json:"data"`    // 同步的数据
}

type MongoSourceDBData struct {
	Id     string `json:"__sys_obj_id__"`
	Addr   string `json:"doc"`
	Optype string `json:"optype"`
}

type RecvData struct {
	JobId       string `json:"job_id"`
	URL         string `json:"url"`
	DbUser      string `json:"db_user"`
	DbPasswd    string `json:"db_passwd"`
	DbDatabase  string `json:"db_database"`
	DbTable     string `json:"db_table"`
	DumpDbType  string `json:"dump_db_type"`
	Gtid        string `json:"gtid"` // GTID作为断点记忆的关键字段
	EventType   string `json:"event_type"`
	PluginParam string `json:"plugin_param"`

	// ES相关字段
	ESIndex      string `json:"es_index"`       // ES索引名称
	ESQuery      string `json:"es_query"`       // ES查询DSL
	ESScrollTime string `json:"es_scroll_time"` // scroll保持时间，如"1m"
	ESBatchSize  int    `json:"es_batch_size"`  // 每批获取数量
	ESTimeField  string `json:"es_time_field"`  // 时间字段名

	// PB级数据同步相关字段
	UseStreamMode     bool   `json:"use_stream_mode"`    // 是否使用流模式
	ChunkSize         int64  `json:"chunk_size"`         // 分块大小(MB)
	ParallelStreams   int    `json:"parallel_streams"`   // 并行数据流数量
	EnableCompression bool   `json:"enable_compression"` // 是否启用压缩
	CompressionLevel  int    `json:"compression_level"`  // 压缩级别
	CheckpointEnabled bool   `json:"checkpoint_enabled"` // 是否启用检查点
	CheckpointPath    string `json:"checkpoint_path"`    // 检查点文件路径
}

type PluginParam struct {
	Hostaddr string `json:"hostaddr"`
	Port     string `json:"port"`
	User     string `json:"user"`
	Password string `json:"password"`
	LogPath  string `json:"log_path"`
}

// ESSourceDBData ES数据结构
type ESSourceDBData struct {
	ID        string      `json:"_id"`       // 文档ID
	Index     string      `json:"_index"`    // 索引名称
	Data      interface{} `json:"data"`      // 文档数据
	Timestamp int64       `json:"timestamp"` // 时间戳
	Optype    string      `json:"optype"`    // 操作类型(index/update/delete)
	Gtid      string      `json:"gtid"`      // 添加GTID字段
}

// SyncESResult ES同步结果
type SyncESResult struct {
	LastScrollID  string           `json:"last_scroll_id"` // 最后的scroll ID
	LastTimestamp int64            `json:"last_timestamp"` // 最后同步的时间戳
	Data          []ESSourceDBData `json:"data"`           // 同步的数据
	LastGtid      string           `json:"last_gtid"`      // 添加最后处理的GTID
}

// DataChunk 分块数据结构
type DataChunk struct {
	ChunkID     string    `json:"chunk_id"`     // 分块ID
	SequenceNum int64     `json:"sequence_num"` // 序列号
	Data        []byte    `json:"data"`         // 二进制数据
	Compressed  bool      `json:"compressed"`   // 是否压缩
	CreateTime  time.Time `json:"create_time"`  // 创建时间
	Size        int64     `json:"size"`         // 原始大小(字节)
	JobID       string    `json:"job_id"`       // 关联的任务ID
	Gtid        string    `json:"gtid"`         // 添加GTID字段
}

// StreamProgress 数据流进度
type StreamProgress struct {
	JobID           string    `json:"job_id"`           // 任务ID
	TotalChunks     int64     `json:"total_chunks"`     // 总分块数
	ProcessedChunks int64     `json:"processed_chunks"` // 已处理分块数
	StartTime       time.Time `json:"start_time"`       // 开始时间
	LastUpdateTime  time.Time `json:"last_update_time"` // 最后更新时间
	BytesProcessed  int64     `json:"bytes_processed"`  // 已处理字节数
	Status          string    `json:"status"`           // 状态(running/paused/completed/failed)
	ErrorMsg        string    `json:"error_msg"`        // 错误信息
	LastGtid        string    `json:"last_gtid"`        // 添加最后处理的GTID
}

// Checkpoint 检查点信息
type Checkpoint struct {
	JobID           string    `json:"job_id"`            // 任务ID
	LastSequenceNum int64     `json:"last_sequence_num"` // 最后处理的序列号
	LastUpdateTime  time.Time `json:"last_update_time"`  // 最后更新时间
	BytesProcessed  int64     `json:"bytes_processed"`   // 已处理字节数
	Metadata        string    `json:"metadata"`          // 元数据(JSON字符串)
	Gtid            string    `json:"gtid"`              // 添加GTID字段，作为断点记忆的关键
}

// SyncStats 同步统计信息
type SyncStats struct {
	JobID            string    `json:"job_id"`             // 任务ID
	StartTime        time.Time `json:"start_time"`         // 开始时间
	EndTime          time.Time `json:"end_time"`           // 结束时间
	TotalBytes       int64     `json:"total_bytes"`        // 总字节数
	ProcessedBytes   int64     `json:"processed_bytes"`    // 已处理字节数
	TotalItems       int64     `json:"total_items"`        // 总项目数
	ProcessedItems   int64     `json:"processed_items"`    // 已处理项目数
	FailedItems      int64     `json:"failed_items"`       // 失败项目数
	AverageSpeed     float64   `json:"average_speed"`      // 平均速度(MB/s)
	CurrentSpeed     float64   `json:"current_speed"`      // 当前速度(MB/s)
	EstimateTimeLeft int64     `json:"estimate_time_left"` // 估计剩余时间(秒)
	Status           string    `json:"status"`             // 状态
	LastGtid         string    `json:"last_gtid"`          // 添加最后处理的GTID
}
