#!/bin/bash

# MongoDB Test Script
# Connection info: host=192.168.0.19, port=27018, username/password=admin

echo "=== MongoDB Test Script Started ==="
echo "Connecting to MongoDB: 192.168.0.19:27018"
echo "Database: test"
echo "Collection: test2"
echo ""

# MongoDB connection parameters
MONGO_HOST="192.168.0.19"
MONGO_PORT="27018"
MONGO_USER="admin"
MONGO_PASS="admin"
MONGO_DB="test"
MONGO_COLLECTION="test2"

# Check if mongo command exists
if ! command -v mongo &> /dev/null; then
    echo "Error: mongo command not found, please ensure MongoDB client is installed"
    exit 1
fi

echo "1. Inserting 5 records..."
mongo --host="$MONGO_HOST" --port="$MONGO_PORT" -u "$MONGO_USER" -p "$MONGO_PASS" --authenticationDatabase="admin" "$MONGO_DB" --eval '
db.test2.insertMany([
  {
    name: "test_record_1",
    ip: {
      ipv4: "192.168.1.100",
      ipv6: "2001:db8::1"
    },
    status: "active",
    created_at: new Date()
  },
  {
    name: "test_record_2",
    ip: {
      ipv4: "192.168.1.101",
      ipv6: "2001:db8::2"
    },
    status: "inactive",
    created_at: new Date()
  },
  {
    name: "test_record_3",
    ip: {
      ipv4: "192.168.1.102",
      ipv6: "2001:db8::3"
    },
    status: "pending",
    created_at: new Date()
  },
  {
    name: "test_record_4",
    ip: {
      ipv4: "192.168.1.103",
      ipv6: "2001:db8::4"
    },
    status: "active",
    created_at: new Date()
  },
  {
    name: "test_record_5",
    ip: {
      ipv4: "192.168.1.104",
      ipv6: "2001:db8::5"
    },
    status: "inactive",
    created_at: new Date()
  }
]);
print("✓ Insert operation completed");
'

if [ $? -eq 0 ]; then
    echo "✓ Successfully inserted 5 records"
else
    echo "✗ Failed to insert records"
fi

echo ""
echo "2. Updating 5 records..."

# Find and update 5 records
mongo --host="$MONGO_HOST" --port="$MONGO_PORT" -u "$MONGO_USER" -p "$MONGO_PASS" --authenticationDatabase="admin" "$MONGO_DB" --eval '
// Find 5 records to update
var recordsToUpdate = db.test2.find().limit(5).toArray();
var updateCount = 0;

recordsToUpdate.forEach(function(record) {
  var result = db.test2.updateOne(
    { _id: record._id },
    { $set: { 
        "status": "updated",
        "ip.ipv4": "192.168.2." + (100 + updateCount),
        "updated_at": new Date()
      }
    }
  );
  updateCount += result.modifiedCount;
});

print("✓ Update operation completed: " + updateCount + " records updated");
'

if [ $? -eq 0 ]; then
    echo "✓ Successfully updated 5 records"
else
    echo "✗ Failed to update records"
fi

echo ""
echo "3. Viewing records after update..."
mongo --host="$MONGO_HOST" --port="$MONGO_PORT" -u "$MONGO_USER" -p "$MONGO_PASS" --authenticationDatabase="admin" "$MONGO_DB" --eval '
print("Current records in test2 collection:");
db.test2.find().forEach(function(doc) {
    printjson(doc);
});
'

echo ""
echo "4. Deleting 5 records..."

# Find and delete 5 records
mongo --host="$MONGO_HOST" --port="$MONGO_PORT" -u "$MONGO_USER" -p "$MONGO_PASS" --authenticationDatabase="admin" "$MONGO_DB" --eval '
// Find 5 records to delete
var recordsToDelete = db.test2.find().limit(5).toArray();
var deleteCount = 0;

recordsToDelete.forEach(function(record) {
  var result = db.test2.deleteOne({ _id: record._id });
  deleteCount += result.deletedCount;
});

print("✓ Delete operation completed: " + deleteCount + " records deleted");
'

if [ $? -eq 0 ]; then
    echo "✓ Successfully deleted 5 records"
else
    echo "✗ Failed to delete records"
fi

echo ""
echo "5. Verifying final results..."
mongo --host="$MONGO_HOST" --port="$MONGO_PORT" -u "$MONGO_USER" -p "$MONGO_PASS" --authenticationDatabase="admin" "$MONGO_DB" --eval '
var count = db.test2.count();
print("Remaining records in test2 collection: " + count);
if (count > 0) {
    print("Remaining records:");
    db.test2.find().forEach(function(doc) {
        printjson(doc);
    });
}
'

echo ""
echo "=== MongoDB Test Script Completed ==="
echo "Test operations included:"
echo "  ✓ Inserted 5 records"
echo "  ✓ Updated 5 records"
echo "  ✓ Deleted 5 records"
echo "  ✓ Verified operation results"