<?php
defined('BASEPATH') or exit('No direct script access allowed');

class AlarmRecord extends MY_Controller
{
	public $Cluster_model;
	public $Login_model;
	public $Db_model;
	public $db;
	public $AlarmCategory_model;
	protected $alarm_service;
	protected $error_messages = [
		'token_empty' => 'Token不能为空',
		'token_invalid' => 'Token验证失败',
		'user_not_found' => '用户不存在',
		'missing_params' => '缺少必需参数',
		'update_failed' => '更新失败',
		'invalid_request' => '无效的请求数据'
	];

	/**
	 * 统一的JSON响应方法
	 */
	protected function jsonResponse($data = [], $code = 200, $message = 'Success')
	{
		$response = [
			'code' => $code,
			'message' => $message
		];

		if (!empty($data)) {
			$response['data'] = $data;
		}

		header('Content-Type: application/json; charset=utf-8');
		echo json_encode($response);
		return;
	}

	/**
	 * Token验证方法
	 */
	protected function validateToken($token)
	{
		if (empty($token)) {
			throw new Exception($this->error_messages['token_empty']);
		}

		$res_token = $this->Login_model->getToken($token, 'D', config_item('key'));
		if (empty($res_token)) {
			throw new Exception($this->error_messages['token_invalid']);
		}

		// 验证用户是否存在
		$sql = "SELECT count(id) as count FROM kunlun_user WHERE name = ?";
		$res = $this->Login_model->getList($sql, [$res_token]);
		if (empty($res) || $res[0]['count'] == 0) {
			throw new Exception($this->error_messages['user_not_found']);
		}

		return $res_token;
	}

	/**
	 * 获取并验证请求头中的Token
	 */
	protected function getRequestToken()
	{
		$headers = $_SERVER;

		if (isset($headers['HTTP_AUTHORIZATION'])) {
			return trim(str_replace('Bearer', '', $headers['HTTP_AUTHORIZATION']));
		} elseif (isset($headers['HTTP_TOKEN'])) {
			return $headers['HTTP_TOKEN'];
		} elseif (isset($_GET['token'])) {
			return $_GET['token'];
		}

		return null;
	}

	/**
	 * 获取POST请求数据
	 */
	protected function getPostData()
	{
		$rawInput = file_get_contents('php://input');
		return json_decode($rawInput, true);
	}

	public function __construct()
	{
		parent::__construct();

		header('Access-Control-Allow-Origin:*'); // *代表允许任何网址请求
		header('Access-Control-Allow-Headers: Content-Type,Content-Length,Accept-Encoding,X-Requested-with, Origin'); // 设置允许自定义请求头的字段
		header('Access-Control-Allow-Methods:POST,GET,OPTIONS,DELETE'); // 允许请求的类型
		header('Access-Control-Allow-Headers:x-requested-with,content-type,Token'); //允许接受token
		header('Content-Type: application/json;charset=utf-8');
		//header('Access-Control-Allow-Credentials: true'); // 设置是否允许发送 cookies

		// Load config and models
		$this->load->config('myconfig');
		$this->load->model('Cluster_model');
		$this->load->model('Login_model');
		$this->load->model('Db_model');
		$this->db = $this->load->database('default', true);
		$this->load->model('AlarmCategory_model');
		$this->load->library('AlarmService');
		$this->alarm_service = new AlarmService();
	}

	/**
	 * 获取告警记录列表
	 */
	public function getAlarmRecordList()
	{
		try {
			// 添加调试日志 - 记录请求开始
			log_message('debug', '开始处理告警记录列表请求: ' . json_encode($_GET));

			// 获取并验证分页参数
			$pageNo = max(1, intval($_GET['pageNo'] ?? 1));
			$pageSize = max(1, min(100, intval($_GET['pageSize'] ?? 10))); // 限制每页最大条数
			log_message('debug', '分页参数: pageNo=' . $pageNo . ', pageSize=' . $pageSize);

			// 构建查询条件
			$where = array_filter([
				'alarm_type' => $_GET['job_type'] ?? null,
				'status' => $_GET['status'] ?? null,
				'alarm_level' => $_GET['alarm_level'] ?? null
			]);
			log_message('debug', '查询条件: ' . json_encode($where));

			// 获取总记录数和分页数据
			try {
				$total = $this->Cluster_model->getAlarmRecordCount($where);
				log_message('debug', '总记录数: ' . $total);

				$offset = ($pageNo - 1) * $pageSize;
				log_message('debug', '查询偏移量: ' . $offset);

				// 记录SQL查询前的状态
				log_message('debug', '开始执行告警记录列表查询');
				$list = $this->Cluster_model->getAlarmRecordList($where, $offset, $pageSize);
				log_message('debug', '查询结果条数: ' . count($list));

				// 记录查询结果的第一条记录（如果存在）
				if (!empty($list)) {
					log_message('debug', '第一条记录示例: ' . json_encode(array_slice($list[0], 0, 5)));
				}
			} catch (Exception $e) {
				log_message('error', '数据库查询失败: ' . $e->getMessage());
				return $this->errorResponse('数据库查询失败: ' . $e->getMessage());
			}

			// 处理告警记录
			if ($list) {
				try {
					foreach ($list as &$item) {
						if($item['status'] == 'handled'){
							continue;
						}
						$this->processAlarmRecord($item);
					}
					log_message('debug', '告警记录处理完成');
				} catch (Exception $e) {
					log_message('error', '处理告警记录失败: ' . $e->getMessage());
					return $this->errorResponse('处理告警记录失败: ' . $e->getMessage());
				}
			}

			log_message('debug', '告警记录列表请求处理完成，准备返回数据');
			return $this->paginateResponse($list ?: [], $total, $pageNo, $pageSize);
		} catch (Exception $e) {
			log_message('error', 'Get alarm record list failed: ' . $e->getMessage() . "\n" . $e->getTraceAsString());
			return $this->errorResponse($e->getMessage());
		}
	}

	/**
	 * 更新告警记录状态
	 */
	public function update()
	{
		try {
			// 验证Token
			$token = $this->getRequestToken();
			$this->validateToken($token);

			// 获取POST数据
			$postData = $this->getPostData();

			// 验证请求数据
			if (!$this->validateRequest('AlarmRecordValidator', 'validateUpdate', $postData)) {
				return;
			}

			// 更新告警状态
			$sql = "UPDATE cluster_alarm_info SET handle_time = NOW(), handler_name = ?, status = ? WHERE id = ?";
			$params = [
				$postData['user_name'],
				$postData['status'],
				$postData['id']
			];

			$result = $this->Cluster_model->updateList($sql, $params);
			if ($result !== 1) {
				throw new Exception($this->error_messages['update_failed']);
			}

			return $this->jsonResponse([], 200, '更新成功');
		} catch (Exception $e) {
			log_message('error', 'Update alarm record failed: ' . $e->getMessage());
			return $this->errorResponse($e->getMessage());
		}
	}

	/**
	 * 删除告警记录
	 */
	public function deleteAlarmItem()
	{
		try {
			// 验证Token
			$token = $this->getRequestToken();
			$this->validateToken($token);

			// 获取POST数据
			$postData = $this->getPostData();

			// 验证请求数据
			if (!$this->validateRequest('AlarmRecordValidator', 'validateDelete', $postData)) {
				return;
			}

			// 删除告警记录
			$sql = "DELETE FROM cluster_alarm_user WHERE id = ?";
			$result = $this->Cluster_model->updateList($sql, [$postData['id']]);

			if (!$result) {
				throw new Exception('删除失败');
			}

			return $this->jsonResponse([], 200, '删除成功');
		} catch (Exception $e) {
			log_message('error', 'Delete alarm record failed: ' . $e->getMessage());
			return $this->errorResponse($e->getMessage());
		}
	}

	/**
	 * 处理单条告警记录
	 */
	private function processAlarmRecord(&$item)
	{
		try {
			log_message('debug', '开始处理告警记录: ' . json_encode(array_slice($item, 0, 5)));

			// 处理job_info
			if (!empty($item['job_info'])) {
				log_message('debug', '处理job_info: ' . substr($item['job_info'], 0, 100) . '...');
				try {
					$jobInfo = json_decode($item['job_info'], true);
					if (json_last_error() !== JSON_ERROR_NONE) {
						log_message('error', 'JSON解析错误: ' . json_last_error_msg() . ', 原始数据: ' . substr($item['job_info'], 0, 100));
						$jobInfo = [];
					}
					$item['object'] = $this->formatAlarmObject($jobInfo);
					$item['info'] = $this->formatAlarmInfo($jobInfo);
					log_message('debug', '处理job_info完成，object: ' . $item['object']);
				} catch (Exception $e) {
					log_message('error', '处理job_info异常: ' . $e->getMessage());
					$item['object'] = '处理失败';
					$item['info'] = '处理失败';
				}
			} else {
				log_message('debug', '告警记录没有job_info字段');
			}

			// 格式化时间
			if (!empty($item['occur_timestamp']) && $item['occur_timestamp'] != '0000-00-00 00:00:00.000000') {
				try {
					$item['occur_timestamp'] = date('Y-m-d H:i:s', strtotime($item['occur_timestamp']));
					log_message('debug', '格式化occur_timestamp完成: ' . $item['occur_timestamp']);
				} catch (Exception $e) {
					log_message('error', '格式化occur_timestamp异常: ' . $e->getMessage() . ', 原始值: ' . $item['occur_timestamp']);
				}
			}

			if (!empty($item['handle_time'])) {
				try {
					$item['handle_time'] = date('Y-m-d H:i:s', strtotime($item['handle_time']));
					log_message('debug', '格式化handle_time完成: ' . $item['handle_time']);
				} catch (Exception $e) {
					log_message('error', '格式化handle_time异常: ' . $e->getMessage() . ', 原始值: ' . $item['handle_time']);
				}
			}

			// 映射告警类型
			$item['job_type'] = $item['alarm_type'];
			log_message('debug', '映射告警类型: ' . $item['job_type']);

			// 添加告警分类信息
			if (!empty($item['alarm_type'])) {
				try {
					$category = $this->AlarmCategory_model->getCategoryByType($item['alarm_type']);
					$item['alarm_category'] = $category['id'];
					$item['alarm_category_label'] = $category['label'];
					log_message('debug', '添加告警分类信息完成: ' . $category['id'] . ' - ' . $category['label']);

					// 获取告警配置信息
					$sql = "SELECT accept_user, push_type FROM kunlun_metadata_db.cluster_alarm_user WHERE id = ? AND status = 'enabled'";
					$alarmConfig = $this->Cluster_model->getList($sql, [$item['alarm_type']]);
					log_message('debug', '获取告警配置: ' . json_encode($alarmConfig));

					if (!empty($alarmConfig)) {
						$config = $alarmConfig[0];

						// 获取接收用户ID列表
						$acceptUsers = !empty($config['accept_user']) ? explode(',', trim($config['accept_user'])) : [];

						// 获取推送方式列表
						$pushTypes = !empty($config['push_type']) ? explode(',', trim($config['push_type'])) : [];

						// 如果有配置接收用户和推送方式，则进行推送
						if (!empty($acceptUsers) && !empty($pushTypes)) {
							log_message('debug', '开始推送告警，接收用户: ' . implode(',', $acceptUsers) . ', 推送方式: ' . implode(',', $pushTypes));

							// 构造告警内容
							$alarmContent = "【告警通知】\n\n";
							$alarmContent .= "告警时间: " . $item['occur_timestamp'] . "\n";
							$alarmContent .= "告警类型: " . $item['alarm_type'] . "\n";
							$alarmContent .= "告警级别: " . $item['alarm_level'] . "\n";
							$alarmContent .= "告警对象: " . $item['object'] . "\n";
							$alarmContent .= "\n告警详情:\n" . $item['info'];

							// 遍历每个推送方式进行推送
							foreach ($pushTypes as $pushType) {
								foreach ($acceptUsers as $userId) {
									try {
										// 获取用户信息（包括邮箱地址）
										$userSql = "SELECT `name`, `email` FROM kunlun_user WHERE id = ?";
										$userInfo = $this->Login_model->getList($userSql, [$userId]);

										if (empty($userInfo)) {
											log_message('error', "未找到用户信息 - 用户ID: $userId");
											continue;
										}

										$receiver = '';

										// 根据不同的推送类型设置接收者信息
										switch ($pushType) {
											case 'email':
												// 使用用户的邮箱地址
												$receiver = $userInfo[0]['email'];
												if (empty($receiver)) {
													log_message('error', "用户未设置邮箱地址 - 用户ID: $userId");
													continue;
												}
												break;

											case 'proxy':
												// 获取代理配置
												$proxyConfig = $this->getProxyConfig();
												if (!$proxyConfig) {
													log_message('error', "未找到有效的代理配置");
													continue;
												}
												$receiver = $proxyConfig['server_address'];
												break;

											case 'phone_message':
												// 使用用户的手机号
												$phoneSql = "SELECT phone FROM kunlun_user WHERE id = ?";
												$phoneInfo = $this->Cluster_model->getList($phoneSql, [$userId]);
												$receiver = $phoneInfo[0]['phone'] ?? '';
												if (empty($receiver)) {
													log_message('error', "用户未设置手机号 - 用户ID: $userId");
													continue;
												}
												break;

											case 'wechat':
												// 使用用户的企业微信ID
												$receiver = $userInfo[0]['username'];
												break;

											default:
												$receiver = $userInfo[0]['username'];
										}

										if (empty($receiver)) {
											log_message('error', "接收者信息为空 - 用户ID: $userId, 推送类型: $pushType");
											continue;
										}

										log_message('debug', "开始推送告警 - 用户ID: $userId, 方式: $pushType, 接收者: $receiver");

										// 发送告警通知
										$result = $this->sendAlarmNotification($item, $pushType, $receiver, $alarmContent);

										// 记录推送历史
										$historyData = [
											'alarm_id' => $item['id'],
											'push_type' => $pushType,
											'receiver' => $receiver,
											'content' => $alarmContent,
											'status' => $result ? 'success' : 'failed',
											'push_time' => date('Y-m-d H:i:s'),
											'error_message' => $result ? '' : '推送失败',
											'is_manual' => 0
										];

										$historySql = "INSERT INTO kunlun_metadata_db.cluster_alarm_push_history 
                        (alarm_id, push_type, receiver, content, status, push_time, error_message, is_manual) 
                        VALUES (?, ?, ?, ?, ?, ?, ?, ?)";

										$this->Cluster_model->updateList($historySql, array_values($historyData));
										
										// 如果推送成功，更新告警状态为已处理
										if ($result) {
											$updateAlarmSql = "UPDATE kunlun_metadata_db.cluster_alarm_info SET status = 'send' WHERE id = ?";
											$this->Cluster_model->updateList($updateAlarmSql, [$item['id']]);
											log_message('debug', "告警状态已更新为已处理 - 告警ID: " . $item['id']);
										}

										log_message('debug', "告警推送完成 - 用户: $userId, 方式: $pushType, 接收者: $receiver, 结果: " . ($result ? '成功' : '失败'));
									} catch (Exception $e) {
										log_message('error', "告警推送失败 - 用户: $userId, 方式: $pushType, 错误: " . $e->getMessage());
									}
								}
							}
						}
					}
				} catch (Exception $e) {
					log_message('error', '添加告警分类信息异常: ' . $e->getMessage() . ', 告警类型: ' . $item['alarm_type']);
					$item['alarm_category'] = 'unknown';
					$item['alarm_category_label'] = '未知分类';
				}
			} else {
				log_message('debug', '告警记录没有alarm_type字段');
			}

			log_message('debug', '告警记录处理完成');
		} catch (Exception $e) {
			log_message('error', '处理告警记录异常: ' . $e->getMessage() . "\n" . $e->getTraceAsString());
		}
	}

	/**
	 * 格式化告警对象信息
	 */
	private function formatAlarmObject($jobInfo)
	{
		try {
			log_message('debug', '开始格式化告警对象信息: ' . json_encode(array_slice($jobInfo, 0, 5)));

			$object = '';
			if (isset($jobInfo['ip'])) {
				$object .= 'IP:' . $jobInfo['ip'];
				if (isset($jobInfo['port'])) {
					$object .= ':' . $jobInfo['port'];
				}
				log_message('debug', '添加IP和端口信息: ' . $object);
			}

			if (isset($jobInfo['cluster_id'])) {
				$object .= ' 集群ID:' . $jobInfo['cluster_id'];
				log_message('debug', '添加集群信息: ' . $object);
			}

			if (isset($jobInfo['shardid'])) {
				$object .= ' 分片ID:' . $jobInfo['shardid'];
				log_message('debug', '添加分片信息: ' . $object);
			}

			log_message('debug', '格式化告警对象信息完成: ' . $object);
			return $object ?: '未知对象';
		} catch (Exception $e) {
			log_message('error', '格式化告警对象信息异常: ' . $e->getMessage());
			return '格式化失败';
		}
	}

	/**
	 * 格式化告警详细信息
	 */
	private function formatAlarmInfo($jobInfo)
	{
		try {
			log_message('debug', '开始格式化告警详细信息');

			if (isset($jobInfo['message'])) {
				log_message('debug', '使用message字段作为告警信息');
				return $jobInfo['message'];
			}

			$info = isset($jobInfo['job_type']) ? '任务类型: ' . $jobInfo['job_type'] : '未知错误';
			log_message('debug', '格式化告警详细信息完成: ' . $info);

			return $info;
		} catch (Exception $e) {
			log_message('error', '格式化告警详细信息异常: ' . $e->getMessage());
			return '格式化失败';
		}
	}

	public function getAlarmSetList()
	{
		//GET请求
		$serve = $_SERVER['QUERY_STRING'];
		$string = preg_split('/[=&]/', $serve);
		$arr = array();
		for ($i = 0; $i < count($string); $i += 2) {
			$arr[$string[$i]] = $string[$i + 1];
		}
		$pageNo = $arr['pageNo'];
		$pageSize = $arr['pageSize'];
		$username = $arr['job_type'];
		$status = $arr['status'];
		$super_dba = $arr['user_name'];
		$start = ($pageNo - 1) * $pageSize;
		$this->load->model('Cluster_model');
		if ($super_dba == "super_dba") {
			$sqldalay = "select * from cluster_alarm_user order by id asc ";
		} else {
			$sqldalay = "select * from cluster_alarm_user order by id asc ";
		}
		$res = $this->Cluster_model->getList($sqldalay);
		$data['code'] = 200;
		$data['list'] = $res;
		$data['total'] = 0;
		print_r(json_encode($data));
	}


	public function getAlarmConfig()
	{

		$this->load->model('Cluster_model');
		$sqldalay = "select * from kunlun_metadata_db.cluster_alarm_message_config ";
		$res = $this->Cluster_model->getList($sqldalay);
		$data['code'] = 200;
		$data['list'] = $res;
		$data['total'] = 0;
		print_r(json_encode($data));
	}


	public function updateAlarmSet()
	{
		log_message('debug', '=== updateAlarmSet() Start ===');

		try {
			// 验证Token
			$token = $this->getRequestToken();
			$this->validateToken($token);

			// 获取POST数据
			$post_data = $this->getPostData();
			log_message('debug', 'Post data: ' . json_encode($post_data));

			// 验证数据量
			if (empty($post_data)) {
				throw new Exception('缺少必需参数');
			}

			// 限制单次更新的数据量
			if (count($post_data) > 100) {
				throw new Exception('单次更新数据不能超过100条');
			}

			// 开始事务
			$this->db->trans_begin();

			try {
				foreach ($post_data as $item) {
					// 验证必需字段
					if (empty($item['id']) || empty($item['label'])) {
						throw new Exception('告警ID和名称不能为空');
					}

					// 格式化数据
					$accept_user = is_array($item['accept_user']) ? implode(',', $item['accept_user']) : $item['accept_user'];
					$push_type = is_array($item['push_type']) ? implode(',', $item['push_type']) : $item['push_type'];

					// 使用REPLACE INTO代替检查+更新/插入
					$sql = "REPLACE INTO kunlun_metadata_db.cluster_alarm_user 
						(id, label, level, threshold, accept_user, push_type, status, create_at) 
						VALUES (?, ?, ?, ?, ?, ?, ?, ?)";

					$params = [
						$item['id'],
						$item['label'],
						$item['level'] ?? 'WARNING',
						$item['threshold'] ?? 0,
						$accept_user,
						$push_type,
						$item['status'] ?? 'enabled',
						time()
					];

					// 使用CI的查询构建器
					$this->db->query($sql, $params);
					if ($this->db->affected_rows() < 0) {
						throw new Exception('更新告警配置失败');
					}
				}

				if ($this->db->trans_status() === FALSE) {
					throw new Exception('事务执行失败');
				}

				$this->db->trans_commit();
				log_message('debug', '=== updateAlarmSet() End ===');
				return $this->jsonResponse([], 200, '更新成功');
			} catch (Exception $e) {
				$this->db->trans_rollback();
				throw $e;
			}
		} catch (Exception $e) {
			log_message('error', 'Update alarm set failed: ' . $e->getMessage());
			return $this->jsonResponse([], 500, $e->getMessage());
		}
	}


	public function updateAlarmConfig()
	{
		log_message('debug', '=== updateAlarmConfig() Start ===');

		try {
			// 验证Token
			$token = $this->getRequestToken();
			$this->validateToken($token);

			// 获取POST数据
			$post_data = $this->getPostData();
			log_message('debug', 'Post data: ' . json_encode($post_data));

			// 验证必需字段
			if (empty($post_data['type'])) {
				throw new Exception('告警类型不能为空');
			}

			$current_time = time();
			$message_json = json_encode($post_data, JSON_UNESCAPED_UNICODE);

			// 使用预处理语句防止SQL注入
			$checkSql = "SELECT id FROM kunlun_metadata_db.cluster_alarm_message_config WHERE type = ?";
			$existingConfig = $this->Cluster_model->getList($checkSql, [$post_data['type']]);

			if (!empty($existingConfig)) {
				// 更新配置
				$sql = "UPDATE cluster_alarm_message_config SET 
					`message` = ?, 
					`update_at` = ? 
					WHERE `type` = ?";
				$params = [$message_json, $current_time, $post_data['type']];
			} else {
				// 新增配置
				$sql = "INSERT INTO cluster_alarm_message_config 
					(`message`, `create_at`, `type`) 
					VALUES (?, ?, ?)";
				$params = [$message_json, $current_time, $post_data['type']];
			}

			$result = $this->Cluster_model->updateList($sql, $params);
			if (!$result) {
				throw new Exception('配置更新失败');
			}

			log_message('debug', '=== updateAlarmConfig() End ===');
			return $this->jsonResponse([], 200, '配置更新成功');
		} catch (Exception $e) {
			log_message('error', 'Update alarm config failed: ' . $e->getMessage());
			return $this->jsonResponse([], 500, $e->getMessage());
		}
	}

	public function getClusterName($id)
	{
		$sql = "select name,nick_name from db_clusters where id='$id'";
		$this->load->model('Cluster_model');
		$res = $this->Cluster_model->getList($sql);
		if (!empty($res)) {
			return $res[0]['nick_name'];
		} else {
			return '';
		}
	}

	public function getShardName($db_cluster_id, $id)
	{
		$sql = "select name from shards where id='$id' and db_cluster_id='$db_cluster_id'";
		$this->load->model('Cluster_model');
		$res = $this->Cluster_model->getList($sql);
		if (!empty($res)) {
			return $res[0]['name'];
		} else {
			return '';
		}
	}

	public function getCompName($db_cluster_id, $id)
	{
		$sql = "select name from comp_nodes where id='$id' and db_cluster_id='$db_cluster_id'";
		$this->load->model('Cluster_model');
		$res = $this->Cluster_model->getList($sql);
		if (!empty($res)) {
			return $res[0]['name'];
		} else {
			return '';
		}
	}

	public function getThisRCRList($id)
	{
		$sql = "select * from cluster_rcr_infos where id='$id' ";
		$this->load->model('Cluster_model');
		$res = $this->Cluster_model->getList($sql);
		return $res;
	}

	public function updateProxyConfig()
	{
		log_message('debug', '=== updateProxyConfig() Start ===');

		try {
			// 验证token
			$arr = apache_request_headers();
			$token = $arr["Token"];
			if (empty($token)) {
				throw new Exception('Token不能为空');
			}

			$this->load->model('Login_model');
			$this->load->model('Cluster_model');
			$res_token = $this->Login_model->getToken($token, 'D', config_item('key'));
			if (empty($res_token)) {
				throw new Exception('Token验证失败');
			}

			// 检查用户是否存在
			$sql = "SELECT count(id) as count FROM kunlun_user WHERE name='$res_token';";
			$res = $this->Login_model->getList($sql);
			if (empty($res) || $res[0]['count'] == 0) {
				throw new Exception('用户不存在或Token错误');
			}

			// 获取POST数据
			$post_data = json_decode(file_get_contents('php://input'), true);
			log_message('debug', 'Post data: ' . json_encode($post_data));

			// 验证必填字段
			if (empty($post_data['serverAddress']) || empty($post_data['port'])) {
				throw new Exception('服务器地址和端口不能为空');
			}

			// 检查是否存在配置
			$checkSql = "SELECT id FROM cluster_alarm_proxy_config LIMIT 1";
			$existingConfig = $this->Cluster_model->getList($checkSql);

			if (!empty($existingConfig)) {
				// 更新配置
				$updateSql = "UPDATE cluster_alarm_proxy_config SET 
					server_address = ?, 
					port = ?, 
					username = ?, 
					password = ?, 
					update_at = ?";
				$res_update = $this->Cluster_model->updateList($updateSql, [
					$post_data['serverAddress'],
					$post_data['port'],
					$post_data['username'] ?? '',
					$post_data['password'] ?? '',
					time()
				]);
			} else {
				// 新增配置
				$addSql = "INSERT INTO cluster_alarm_proxy_config 
					(server_address, port, username, password, create_at) 
					VALUES (?, ?, ?, ?, ?)";
				$res_update = $this->Cluster_model->updateList($addSql, [
					$post_data['serverAddress'],
					$post_data['port'],
					$post_data['username'] ?? '',
					$post_data['password'] ?? '',
					time()
				]);
			}

			if ($res_update) {
				$data = [
					'code' => 200,
					'message' => '代理服务器配置更新成功'
				];
			} else {
				throw new Exception('配置更新失败');
			}
		} catch (Exception $e) {
			log_message('error', 'Update proxy config failed: ' . $e->getMessage());
			$data = [
				'code' => 500,
				'message' => $e->getMessage()
			];
		}

		log_message('debug', 'Response: ' . json_encode($data));
		log_message('debug', '=== updateProxyConfig() End ===');

		echo json_encode($data);
	}

	public function testProxyConnection()
	{
		log_message('debug', '=== testProxyConnection() Start ===');

		try {
			// 验证Token
			$token = $this->getRequestToken();
			$this->validateToken($token);

			// 获取POST数据
			$post_data = $this->getPostData();
			log_message('debug', 'Post data: ' . json_encode($post_data));

			// 验证必需参数
			if (empty($post_data['serverAddress']) || empty($post_data['port'])) {
				throw new Exception('服务器地址和端口不能为空');
			}

			// 测试连接
			$ch = curl_init();
			$options = [
				CURLOPT_URL => "http://{$post_data['serverAddress']}:{$post_data['port']}",
				CURLOPT_TIMEOUT => 5,
				CURLOPT_RETURNTRANSFER => true,
				CURLOPT_CONNECTTIMEOUT => 3,
				CURLOPT_SSL_VERIFYPEER => false,
				CURLOPT_SSL_VERIFYHOST => false
			];

			// 添加认证信息
			if (!empty($post_data['username']) && !empty($post_data['password'])) {
				$options[CURLOPT_HTTPAUTH] = CURLAUTH_BASIC;
				$options[CURLOPT_USERPWD] = "{$post_data['username']}:{$post_data['password']}";
			}

			curl_setopt_array($ch, $options);
			$result = curl_exec($ch);
			$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
			$error = curl_error($ch);
			curl_close($ch);

			if ($httpCode >= 200 && $httpCode < 300) {
				log_message('debug', '=== testProxyConnection() End: Success ===');
				return $this->jsonResponse([], 200, '代理服务器连接成功');
			} else {
				throw new Exception('代理服务器连接失败: ' . ($error ?: "HTTP状态码 {$httpCode}"));
			}
		} catch (Exception $e) {
			log_message('error', 'Test proxy connection failed: ' . $e->getMessage());
			return $this->jsonResponse([], 500, $e->getMessage());
		}
	}

	/**
	 * 获取代理配置
	 * @return array|null 代理配置信息
	 */
	private function getProxyConfig() {
		$sql = "SELECT server_address, template_id FROM kunlun_metadata_db.cluster_alarm_proxy_config WHERE status = 'enabled' ORDER BY id DESC LIMIT 1";
		$config = $this->Cluster_model->getList($sql);
		return !empty($config) ? $config[0] : null;
	}

	/**
	 * 获取告警分类列表
	 */
	public function getAlarmCategories()
	{
		try {
			// 验证Token
			$token = $this->getRequestToken();
			$this->validateToken($token);

			// 获取告警分类
			$categories = $this->AlarmCategory_model->getCategories();

			return $this->jsonResponse([
				'list' => $categories,
				'total' => count($categories)
			]);
		} catch (Exception $e) {
			log_message('error', 'Get alarm categories failed: ' . $e->getMessage());
			return $this->jsonResponse([], 500, $e->getMessage());
		}
	}

	/**
	 * 获取告警推送历史记录
	 */
	public function getPushHistory()
	{
		try {
			// 验证Token
			$token = $this->getRequestToken();
			$username = $this->validateToken($token);

			// 获取查询参数
			$alarm_id = isset($_GET['alarm_id']) ? $_GET['alarm_id'] : null;
			$page = isset($_GET['page']) ? intval($_GET['page']) : 1;
			$limit = isset($_GET['limit']) ? intval($_GET['limit']) : 10;
			$offset = ($page - 1) * $limit;

			// 构建查询条件
			$where = [];
			$params = [];

			if (!empty($alarm_id)) {
				$where[] = "alarm_id = ?";
				$params[] = $alarm_id;
			}

			$whereClause = empty($where) ? "" : "WHERE " . implode(" AND ", $where);

			// 查询总数
			$countSql = "SELECT COUNT(*) as total FROM kunlun_alarm_push_history $whereClause";
			$countResult = $this->Db_model->getList($countSql, $params);
			$total = isset($countResult[0]['total']) ? intval($countResult[0]['total']) : 0;

			// 查询列表
			$sql = "SELECT * FROM kunlun_alarm_push_history $whereClause ORDER BY push_time DESC LIMIT ?, ?";
			$params[] = $offset;
			$params[] = $limit;

			$list = $this->Db_model->getList($sql, $params);

			// 处理结果
			$result = [
				'total' => $total,
				'list' => $list ?: []
			];

			$this->jsonResponse($result);
		} catch (Exception $e) {
			$this->jsonResponse([], 500, $e->getMessage());
		}
	}

	/**
	 * 重试推送告警
	 */
	public function retryPush()
	{
		try {
			// 验证Token
			$token = $this->getRequestToken();
			$username = $this->validateToken($token);

			// 获取POST数据
			$data = $this->getPostData();

			// 验证必要参数
			if (empty($data['push_id'])) {
				throw new Exception($this->error_messages['missing_params'] . ': push_id');
			}

			// 获取推送记录
			$sql = "SELECT * FROM kunlun_alarm_push_history WHERE id = ?";
			$pushRecord = $this->Db_model->getList($sql, [$data['push_id']]);

			if (empty($pushRecord)) {
				throw new Exception('推送记录不存在');
			}

			$pushRecord = $pushRecord[0];

			// 获取告警记录
			$sql = "SELECT * FROM kunlun_alarm_record WHERE id = ?";
			$alarmRecord = $this->Db_model->getList($sql, [$pushRecord['alarm_id']]);

			if (empty($alarmRecord)) {
				throw new Exception('告警记录不存在');
			}

			$alarmRecord = $alarmRecord[0];

			// 重新推送告警
			$pushResult = $this->sendAlarmNotification(
				$alarmRecord,
				$pushRecord['push_type'],
				$pushRecord['receiver'],
				$pushRecord['content']
			);

			// 更新推送记录
			$updateData = [
				'status' => $pushResult ? 'success' : 'failed',
				'push_time' => date('Y-m-d H:i:s'),
				'error_message' => $pushResult ? '' : '重试推送失败'
			];

			$this->Db_model->update('kunlun_alarm_push_history', $updateData, ['id' => $data['push_id']]);

			$this->jsonResponse(['success' => $pushResult]);
		} catch (Exception $e) {
			$this->jsonResponse([], 500, $e->getMessage());
		}
	}

	/**
	 * 手动推送告警
	 */
	public function manualPush()
	{
		log_message('debug', '=== manualPush() Start ===');
		try {
			// 验证Token
			$token = $this->getRequestToken();
			log_message('debug', 'Token received: ' . substr($token, 0, 10) . '...');

			$username = $this->validateToken($token);
			log_message('debug', 'Token validated for user: ' . $username);

			// 获取POST数据
			$data = $this->getPostData();
			log_message('debug', 'Received POST data: ' . json_encode($data));

			// 验证必要参数
			if (empty($data['alarm_id'])) {
				throw new Exception($this->error_messages['missing_params'] . ': alarm_id');
			}

			if (empty($data['push_type'])) {
				throw new Exception($this->error_messages['missing_params'] . ': push_type');
			}

			if (empty($data['receiver'])) {
				throw new Exception($this->error_messages['missing_params'] . ': receiver');
			}

			if (empty($data['content'])) {
				throw new Exception($this->error_messages['missing_params'] . ': content');
			}

			// 获取告警记录
			$sql = "SELECT * FROM cluster_alarm_info WHERE id = ?";
			$params = [intval($data['alarm_id'])];

			log_message('debug', 'Fetching alarm record with SQL: ' . $sql . ', params: ' . json_encode($params));

			$alarmRecord = $this->Cluster_model->getList($sql, $params);

			if (empty($alarmRecord)) {
				log_message('error', 'Alarm record not found for ID: ' . $data['alarm_id']);
				throw new Exception('告警记录不存在');
			}

			$alarmRecord = $alarmRecord[0];
			log_message('debug', 'Found alarm record: ' . json_encode($alarmRecord));

			// 发送告警通知
			log_message('debug', 'Attempting to send notification...');
			$pushResult = $this->sendAlarmNotification(
				$alarmRecord,
				$data['push_type'],
				$data['receiver'],
				$data['content']
			);
			log_message('debug', 'Push notification result: ' . ($pushResult ? 'success' : 'failed'));

			// 记录推送历史
			$pushHistoryData = [
				'alarm_id' => $data['alarm_id'],
				'push_type' => $data['push_type'],
				'receiver' => $data['receiver'],
				'content' => $data['content'],
				'status' => $pushResult ? 'success' : 'failed',
				'push_time' => date('Y-m-d H:i:s'),
				'error_message' => $pushResult ? '' : '手动推送失败',
				'is_manual' => 1
			];

			log_message('debug', 'Recording push history: ' . json_encode($pushHistoryData));

			$sql = "INSERT INTO kunlun_metadata_db.cluster_alarm_push_history (alarm_id, push_type, receiver, content, status, push_time, error_message, is_manual) 
					VALUES (?, ?, ?, ?, ?, ?, ?, ?)";

			$params = [
				$pushHistoryData['alarm_id'],
				$pushHistoryData['push_type'],
				$pushHistoryData['receiver'],
				$pushHistoryData['content'],
				$pushHistoryData['status'],
				$pushHistoryData['push_time'],
				$pushHistoryData['error_message'],
				$pushHistoryData['is_manual']
			];

			log_message('debug', 'Executing SQL: ' . $sql);
			log_message('debug', 'With parameters: ' . json_encode($params));

			$result = $this->Cluster_model->updateList($sql, $params);
			log_message('debug', 'Insert result: ' . json_encode($result));

			log_message('debug', '=== manualPush() End: Success ===');
			return $this->jsonResponse(['success' => $pushResult]);
		} catch (Exception $e) {
			log_message('error', '=== manualPush() End: Error ===');
			log_message('error', 'Error message: ' . $e->getMessage());
			log_message('error', 'Stack trace: ' . $e->getTraceAsString());
			return $this->jsonResponse([], 500, $e->getMessage());
		}
	}

	/**
	 * 生成告警邮件的HTML模板
	 */
	private function generateAlarmEmailTemplate($title, $alarmData)
	{
		return '
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>' . $title . '</title>
</head>
<body style="margin: 0; padding: 0; font-family: Arial, sans-serif; background-color: #f4f4f4;">
    <div style="max-width: 800px; margin: 0 auto; background-color: #ffffff; padding: 20px; border-radius: 8px; box-shadow: 0 2px 5px rgba(0,0,0,0.1); margin-top: 20px;">
        
        <!-- 基本信息部分 -->
        <div style="margin-bottom: 30px;">
            <h2 style="color: #17233d; font-size: 18px; margin-bottom: 15px; padding-bottom: 8px; border-bottom: 1px solid #e8eaec;">
                基本信息
            </h2>
            <table style="width: 100%; border-collapse: collapse;">
                <tr>
                    <td style="padding: 8px; width: 120px; color: #515a6e;">告警时间：</td>
                    <td style="padding: 8px; color: #17233d;">' . $alarmData['time'] . '</td>
                </tr>
                <tr>
                    <td style="padding: 8px; width: 120px; color: #515a6e;">告警类型：</td>
                    <td style="padding: 8px; color: #17233d;">' . $alarmData['type'] . '</td>
                </tr>
                <tr>
                    <td style="padding: 8px; width: 120px; color: #515a6e;">告警级别：</td>
                    <td style="padding: 8px;">
                        <span style="padding: 4px 8px; border-radius: 4px; font-size: 12px; ' .
			($alarmData['level'] === 'FATAL' ? 'background-color: #f56c6c; color: #fff;' : ($alarmData['level'] === 'ERROR' ? 'background-color: #e6a23c; color: #fff;' :
				'background-color: #909399; color: #fff;')) . '">
                            ' . $alarmData['level'] . '
                        </span>
                    </td>
                </tr>
                <tr>
                    <td style="padding: 8px; width: 120px; color: #515a6e;">告警对象：</td>
                    <td style="padding: 8px; color: #17233d;">' . $alarmData['object'] . '</td>
                </tr>
            </table>
        </div>

        <!-- 告警详情部分 -->
        <div style="margin-bottom: 30px;">
            <h2 style="color: #17233d; font-size: 18px; margin-bottom: 15px; padding-bottom: 8px; border-bottom: 1px solid #e8eaec;">
                告警详情
            </h2>
            <div style="background-color: #f8f8f9; border-radius: 4px; padding: 15px; color: #515a6e; line-height: 1.6; white-space: pre-wrap; font-family: monospace;">
                ' . htmlspecialchars($alarmData['detail']) . '
            </div>
        </div>
    </div>
</body>
</html>';
	}

	/**
	 * 发送告警通知
	 */
	private function sendAlarmNotification($alarm, $pushType, $receiver, $content)
	{
		log_message('debug', '=== sendAlarmNotification() Start ===');
		log_message('debug', 'Push type: ' . $pushType);
		log_message('debug', 'Receiver: ' . $receiver);

		if ($pushType == 'proxy') {
			//$receiver = 'http://http://message.klustron.com/api/v1/message/send:80';
		} else {
			// 获取告警配置
			$sql = "SELECT * FROM kunlun_metadata_db.cluster_alarm_message_config WHERE type = ? LIMIT 1";
			log_message('debug', 'Fetching alarm config with SQL: ' . $sql);

			$config = $this->Cluster_model->getList($sql, [$pushType]);

			if (empty($config)) {
				log_message('error', 'No alarm configuration found');
				return false;
			}

			$config = $config[0];
			log_message('debug', 'Raw config: ' . json_encode($config));

			// 解析配置数据
			try {
				$configData = json_decode($config['message'], true);
				if (json_last_error() !== JSON_ERROR_NONE) {
					log_message('error', 'Failed to parse config message: ' . json_last_error_msg());
					log_message('error', 'Raw message: ' . $config['message']);
					return false;
				}
				log_message('debug', 'Parsed config data: ' . json_encode($configData));
			} catch (Exception $e) {
				log_message('error', 'Exception while parsing config: ' . $e->getMessage());
				return false;
			}
		}

		// 根据推送类型发送通知
		try {
			$result = false;
			switch ($pushType) {
				case 'phone_message':
					log_message('debug', 'Sending SMS notification...');
					$result = $this->alarm_service->sendSmsNotification($configData, $receiver, $content);
					break;

				case 'email':
					log_message('debug', 'Sending email notification...');
					// 解析告警内容
					$contentLines = explode("\n", $content);
					$alarmData = [
						'time' => '',
						'type' => '',
						'level' => '',
						'object' => '',
						'detail' => ''
					];

					foreach ($contentLines as $line) {
						if (strpos($line, '告警时间:') !== false) {
							$alarmData['time'] = trim(str_replace('告警时间:', '', $line));
						} else if (strpos($line, '告警类型:') !== false) {
							$alarmData['type'] = trim(str_replace('告警类型:', '', $line));
						} else if (strpos($line, '告警级别:') !== false) {
							$alarmData['level'] = trim(str_replace('告警级别:', '', $line));
						} else if (strpos($line, '告警对象:') !== false) {
							$alarmData['object'] = trim(str_replace('告警对象:', '', $line));
						} else if (strpos($line, '告警详情:') !== false) {
							$alarmData['detail'] = trim(substr($content, strpos($content, '告警详情:') + strlen('告警详情:')));
						}
					}

					// 生成HTML内容
					$htmlContent = $this->generateAlarmEmailTemplate('Kunlun Monitor 告警通知', $alarmData);
					$result = $this->alarm_service->sendEmailNotification($configData, $receiver, $htmlContent, '告警通知: ' . $alarm['alarm_type']);
					break;

				case 'wechat':
					log_message('debug', 'Sending WeChat notification...');
					$result = $this->alarm_service->sendWechatNotification($configData, $alarm['alarm_type'], $content);
					break;

				case 'proxy':
					log_message('debug', 'Sending proxy notification...');
					// 获取代理配置
					$proxyConfig = $this->getProxyConfig();
					if (!$proxyConfig) {
						log_message('error', "未找到有效的代理配置");
						return false;
					}
					$templateId = $proxyConfig['template_id'] ?? '';
					log_message('debug', '使用代理服务器地址: ' . $proxyConfig['server_address']);
					log_message('debug', '使用模板ID: ' . $templateId);
					log_message('debug', '准备发送告警类型: ' . $alarm['alarm_type']);
					try {
						$result = $this->alarm_service->sendProxyNotification($proxyConfig['server_address'], $alarm['alarm_type'], $content, $templateId);
						log_message('debug', '代理通知发送结果: ' . ($result ? '成功' : '失败'));
					} catch (Exception $e) {
						log_message('error', '代理通知发送失败: ' . $e->getMessage());
						return false;
					}
					break;

				default:
					log_message('error', 'Unknown push type: ' . $pushType);
					return false;
			}

			log_message('debug', 'Notification result: ' . ($result ? 'success' : 'failed'));
			log_message('debug', '=== sendAlarmNotification() End ===');
			return $result;
		} catch (Exception $e) {
			log_message('error', 'Failed to send notification: ' . $e->getMessage());
			log_message('error', 'Stack trace: ' . $e->getTraceAsString());
			return false;
		}
	}

	/**
	 * 获取告警统计数据
	 * 用于生成告警报表和统计图表
	 */
	public function getAlarmStatistics()
	{
		try {
			// 验证Token
			$token = $this->getRequestToken();
			$this->validateToken($token);

			// 获取请求参数 - 不需要显式加载input库，它是CI核心组件
			$startDate = $this->input->get('start_time', TRUE);
			$endDate = $this->input->get('end_time', TRUE);
			$category = $this->input->get('category', TRUE);
			$alarm_type = $this->input->get('alarm_type', TRUE);

			// 记录日志以便调试
			log_message('debug', 'AlarmStatistics params: ' . json_encode([
				'start_time' => $startDate,
				'end_time' => $endDate,
				'category' => $category,
				'alarm_type' => $alarm_type
			]));

			// 构建基础SQL语句 - 按类型统计总数
			$sql = "SELECT alarm_type, COUNT(*) as count
				   FROM kunlun_metadata_db.cluster_alarm_info WHERE 1=1";
			
			// 构建查询条件
			$params = [];
			
			if (!empty($startDate)) {
				$sql .= " AND occur_timestamp >= ?";
				$params[] = $startDate;
			}
			
			if (!empty($endDate)) {
				$sql .= " AND occur_timestamp <= ?";
				$params[] = $endDate;
			}
			
			if (!empty($alarm_type)) {
				$sql .= " AND alarm_type = ?";
				$params[] = $alarm_type;
			}
			
			// 如有分类过滤
			if (!empty($category)) {
				// 首先查询该分类下的所有告警类型
				$sqlTypes = "SELECT alarm_type FROM kunlun_metadata_db.alarm_categories_mapping WHERE category_id = ?";
				$types = $this->Cluster_model->getList($sqlTypes, [$category]);
				
				if (!empty($types)) {
					$typeIds = array_column($types, 'alarm_type');
					$placeholders = implode(',', array_fill(0, count($typeIds), '?'));
					$sql .= " AND alarm_type IN ($placeholders)";
					$params = array_merge($params, $typeIds);
				}
			}
			
			$sql .= " GROUP BY alarm_type";
			log_message('debug', 'SQL type stats: ' . $sql . ' Params: ' . json_encode($params));
			$alarmsByType = $this->Cluster_model->getList($sql, $params);
			
			// 按告警级别统计
			$sqlLevel = "SELECT alarm_level, COUNT(*) as count
						FROM kunlun_metadata_db.cluster_alarm_info WHERE 1=1";
			
			// 复制参数数组，避免引用问题
			$levelParams = $params;
			
			// 添加相同的条件
			if (!empty($startDate)) {
				$sqlLevel .= " AND occur_timestamp >= ?";
				if (!in_array($startDate, $levelParams)) {
					$levelParams[] = $startDate;
				}
			}
			
			if (!empty($endDate)) {
				$sqlLevel .= " AND occur_timestamp <= ?";
				if (!in_array($endDate, $levelParams)) {
					$levelParams[] = $endDate;
				}
			}
			
			if (!empty($alarm_type)) {
				$sqlLevel .= " AND alarm_type = ?";
				if (!in_array($alarm_type, $levelParams)) {
					$levelParams[] = $alarm_type;
				}
			}
			
			$sqlLevel .= " GROUP BY alarm_level";
			log_message('debug', 'SQL level stats: ' . $sqlLevel . ' Params: ' . json_encode($levelParams));
			$alarmsByLevel = $this->Cluster_model->getList($sqlLevel, $levelParams);
			
			// 获取总告警数量 - 创建新的参数数组
			$totalParams = [];
			
			$sqlTotal = "SELECT 
				COUNT(*) as total,
				SUM(CASE WHEN status = 'unhandled' THEN 1 ELSE 0 END) as unhandled,
				SUM(CASE WHEN DATE(occur_timestamp) = CURDATE() THEN 1 ELSE 0 END) as today,
				SUM(CASE WHEN YEARWEEK(occur_timestamp) = YEARWEEK(NOW()) THEN 1 ELSE 0 END) as week
			  FROM kunlun_metadata_db.cluster_alarm_info WHERE 1=1";
			  
			// 添加相同的条件
			if (!empty($startDate)) {
				$sqlTotal .= " AND occur_timestamp >= ?";
				$totalParams[] = $startDate;
			}
			
			if (!empty($endDate)) {
				$sqlTotal .= " AND occur_timestamp <= ?";
				$totalParams[] = $endDate;
			}
			
			if (!empty($alarm_type)) {
				$sqlTotal .= " AND alarm_type = ?";
				$totalParams[] = $alarm_type;
			}
			
			// 如有分类过滤且已经查到了类型列表
			if (!empty($category) && isset($typeIds) && is_array($typeIds) && !empty($typeIds)) {
				$placeholders = implode(',', array_fill(0, count($typeIds), '?'));
				$sqlTotal .= " AND alarm_type IN ($placeholders)";
				$totalParams = array_merge($totalParams, $typeIds);
			}
			
			log_message('debug', 'SQL total: ' . $sqlTotal . ' Params: ' . json_encode($totalParams));
			$totalStats = $this->Cluster_model->getList($sqlTotal, $totalParams);
			
			// 安全地获取统计数据
			$total = 0;
			$unhandled = 0;
			$today = 0;
			$week = 0;
			
			if (!empty($totalStats) && isset($totalStats[0])) {
				$total = isset($totalStats[0]['total']) ? (int)$totalStats[0]['total'] : 0;
				$unhandled = isset($totalStats[0]['unhandled']) ? (int)$totalStats[0]['unhandled'] : 0;
				$today = isset($totalStats[0]['today']) ? (int)$totalStats[0]['today'] : 0;
				$week = isset($totalStats[0]['week']) ? (int)$totalStats[0]['week'] : 0;
			}
			
			// 返回结果
			$response = [
				'type_stats' => is_array($alarmsByType) ? $alarmsByType : [],
				'level_stats' => is_array($alarmsByLevel) ? $alarmsByLevel : [],
				'total' => $total,
				'unhandled' => $unhandled,
				'today' => $today,
				'week' => $week
			];
			
			log_message('debug', 'AlarmStatistics response: ' . json_encode($response));
			return $this->jsonResponse($response);
		} catch (Exception $e) {
			log_message('error', 'AlarmStatistics error: ' . $e->getMessage() . "\n" . $e->getTraceAsString());
			return $this->jsonResponse(['error' => $e->getMessage()], 500, 'Failed to get alarm statistics: ' . $e->getMessage());
		}
	}

	/**
	 * 获取告警类型的详细信息
	 * @param string $alarm_type 告警类型
	 * @return array|null 告警类型详细信息
	 */
	private function getAlarmTypeInfo($alarm_type)
	{
		try {
			// 查询alarm_categories表获取分类信息
			$sql = "SELECT cat.id, cat.label, alarm.label as alarm_label
				   FROM kunlun_metadata_db.alarm_categories cat
				   JOIN kunlun_metadata_db.alarm_types alarm ON alarm.category_id = cat.id
				   WHERE alarm.id = ?";
			
			$result = $this->Cluster_model->getList($sql, [$alarm_type]);
			
			if (empty($result)) {
				return null;
			}
			
			return [
				'id' => $alarm_type,
				'label' => $result[0]['alarm_label'],
				'category' => $result[0]['label'],
				'category_id' => $result[0]['id']
			];
		} catch (Exception $e) {
			log_message('error', 'Error getting alarm type info: ' . $e->getMessage());
			return null;
		}
	}
}
