<?php
defined('BASEPATH') or exit('No direct script access allowed');

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

// 引入 Composer 的自动加载文件
require_once FCPATH . 'vendor/autoload.php';

/**
 * CdcMonitor控制器
 * 负责CDC集群相关的监控和告警功能
 */
class CdcMonitor extends CI_Controller
{
    public $cache_duration = 300; // 缓存时间 5 分钟
    public $alarm_config = null;
    public $push_config = null;
    public $config;
    public $key;
    public $post_url;
    public $default_username;
    public $pg_username;
    public $db_prefix;
    public $Login_model;
    public $Cluster_model;
    public $AlarmCategory_model;
    
    // 添加类库引用
    public $alarmhandler;
    public $notificationhandler;
    public $aliyunsms; // 添加短信服务库引用
    
    public function __construct()
    {
        parent::__construct();
        //打开重定向
        header('Access-Control-Allow-Origin:*'); // *代表允许任何网址请求
        header('Access-Control-Allow-Headers: Content-Type,Content-Length, Origin'); // 设置允许自定义请求头的字段
        header('Access-Control-Allow-Methods:POST,GET,OPTIONS,DELETE'); // 允许请求的类型
        header('Access-Control-Allow-Headers:x-requested-with,content-type,Token'); //允许接受token
        header('Content-Type: application/json;charset=utf-8');

        // 加载必要的模型和库
        $this->load->model('Login_model');
        $this->load->model('Cluster_model');
        $this->load->model('AlarmCategory_model');
        
        // 加载自定义类库
        $this->load->library('AlarmHandler');
        $this->load->library('NotificationHandler');
        $this->load->library('AliyunSms'); // 加载短信服务库
        
        // 初始化配置
        $this->initAlarmConfig();
    }
    
    /**
     * 初始化告警配置
     */
    private function initAlarmConfig()
    {
        // 获取告警和推送配置
        $sql = "SELECT * FROM kunlun_metadata_db.cluster_alarm_user";
        $this->alarm_config = $this->Cluster_model->getList($sql);
        
        $sql = "SELECT * FROM kunlun_metadata_db.cluster_alarm_message_config";
        $this->push_config = $this->Cluster_model->getList($sql);
    }
    
    /**
     * CDC监控告警检查接口
     * 检查CDC集群状态并记录相关告警
     */
    public function getCdcMonitor()
    {
        set_time_limit(0);
        log_message('debug', '=== getCdcMonitor() Start ===');
        try {
            // 获取并验证用户
            $serve = $_SERVER['QUERY_STRING'];
            $string = preg_split('/[=&]/', $serve);
            $arr = array();
            for ($i = 0; $i < count($string); $i += 2) {
                if (isset($string[$i+1])) {
                    $arr[$string[$i]] = $string[$i + 1];
                }
            }
            $user_name = $arr['user_name'] ?? '';
            
            if (empty($user_name)) {
                throw new Exception('用户名不能为空');
            }

            $user_sql = "SELECT id FROM kunlun_dba_tools_db.kunlun_user WHERE name=?";
            $res_user = $this->Login_model->getList($user_sql, [$user_name]);
            if (empty($res_user)) {
                log_message('error', 'User validation failed: user does not exist');
                throw new Exception('用户不存在');
            }
            $user_id = $res_user[0]['id'];
            log_message('debug', 'User validated successfully: ' . $user_name);

            // 获取所有CDC集群配置
            $sql = "SELECT id, cluster_name, master_node_host, master_node_port FROM kunlun_metadata_db.cdc_clusters WHERE status='active'";
            $cdc_clusters = $this->Cluster_model->getList($sql);
            $error_list = array();

            // if (!empty($cdc_clusters)) {
            //     foreach ($cdc_clusters as $cluster) {
            //         $cdc_id = $cluster['id'];
            //         $cluster_name = $cluster['cluster_name'];
            //         $master_host = $cluster['master_node_host'];
            //         $master_port = $cluster['master_node_port'];
                    
            //         // 1. 检查CDC节点异常
            //         $node_exceptions = $this->checkCdcNodeException($cdc_id, $cluster_name);
            //         if (!empty($node_exceptions)) {
            //             foreach ($node_exceptions as $node_exception) {
            //                 $error_list[] = $node_exception['message'];
            //                 $this->recordCdcAlarm('cdc_node_exception', $node_exception);
            //             }
            //         }
                    
            //         // 2. 检查CDC主备切换
            //         $leader_changes = $this->checkCdcLeaderChange($cdc_id, $cluster_name);
            //         if (!empty($leader_changes)) {
            //             foreach ($leader_changes as $leader_change) {
            //                 $error_list[] = $leader_change['message'];
            //                 $this->recordCdcAlarm('cdc_leader_change', $leader_change);
            //             }
            //         }
                    
            //         // 3. 检查CDC同步目标失败
            //         $sync_failures = $this->checkCdcSyncTargetFailed($cdc_id, $cluster_name);
            //         if (!empty($sync_failures)) {
            //             foreach ($sync_failures as $sync_failure) {
            //                 $error_list[] = $sync_failure['message'];
            //                 $this->recordCdcAlarm('cdc_sync_target_failed', $sync_failure);
            //             }
            //         }
                    
            //         // 4. 检查CDC解析DDL失败
            //         $ddl_failures = $this->checkCdcParseDdlFailed($cdc_id, $cluster_name);
            //         if (!empty($ddl_failures)) {
            //             foreach ($ddl_failures as $ddl_failure) {
            //                 $error_list[] = $ddl_failure['message'];
            //                 $this->recordCdcAlarm('cdc_parse_ddl_failed', $ddl_failure);
            //             }
            //         }
            //     }
            // } else {
            //     log_message('info', 'No active CDC clusters found.');
            // }
            
            // 处理告警并发送通知
            $this->sendBatchAlarms($error_list);

            // 返回处理结果
            echo json_encode([
                'code' => 200,
                'message' => 'CDC告警处理成功',
                'error_list' => $error_list
            ]);
            log_message('debug', '=== getCdcMonitor() End ===');
        } catch (Exception $e) {
            log_message('error', 'CDC Monitor error: ' . $e->getMessage());
            echo json_encode([
                'code' => 500,
                'message' => $e->getMessage()
            ]);
        }
    }
    
    /**
     * 检查CDC节点异常
     * @param int $cdc_id CDC集群ID
     * @param string $cluster_name CDC集群名称
     * @return array 节点异常列表
     */
    private function checkCdcNodeException($cdc_id, $cluster_name)
    {
        log_message('debug', "Checking CDC node exceptions for cluster: {$cluster_name} (ID: {$cdc_id})");
        $exceptions = [];
        
        // 查询CDC节点列表
        $sql = "SELECT node_id, node_host, node_port, last_heartbeat_time 
                FROM kunlun_metadata_db.cdc_nodes 
                WHERE cluster_id = ? AND status = 'active'";
        $nodes = $this->Cluster_model->getList($sql, [$cdc_id]);
        
        if (empty($nodes)) {
            log_message('warning', "No CDC nodes found for cluster: {$cluster_name}");
            return $exceptions;
        }
        
        foreach ($nodes as $node) {
            $node_host = $node['node_host'];
            $node_port = $node['node_port'];
            $node_id = $node['node_id'];
            $last_heartbeat = $node['last_heartbeat_time'];
            
            // 检查节点最后心跳时间是否超过阈值(例如5分钟)
            $current_time = time();
            $heartbeat_time = strtotime($last_heartbeat);
            $time_diff = $current_time - $heartbeat_time;
            
            if ($time_diff > 300) { // 超过5分钟未心跳
                $message = "CDC节点异常: 集群:{$cluster_name}, 节点:{$node_host}:{$node_port}, 最后心跳时间:{$last_heartbeat}";
                log_message('warning', $message);
                
                // 检查节点是否在线
                $is_node_online = $this->checkNodeOnline($node_host, $node_port);
                if (!$is_node_online) {
                    $exception = [
                        'cluster_id' => $cdc_id,
                        'node_id' => $node_id,
                        'message' => $message,
                        'ip' => $node_host,
                        'port' => $node_port,
                        'last_heartbeat' => $last_heartbeat
                    ];
                    $exceptions[] = $exception;
                }
            }
        }
        
        return $exceptions;
    }
    
    /**
     * 检查CDC主备切换
     * @param int $cdc_id CDC集群ID
     * @param string $cluster_name CDC集群名称
     * @return array 主备切换列表
     */
    private function checkCdcLeaderChange($cdc_id, $cluster_name)
    {
        log_message('debug', "Checking CDC leader changes for cluster: {$cluster_name} (ID: {$cdc_id})");
        $changes = [];
        
        // 查询当前的主节点
        $sql = "SELECT master_node_host, master_node_port FROM kunlun_metadata_db.cdc_clusters WHERE id = ?";
        $current_master = $this->Cluster_model->getList($sql, [$cdc_id]);
        
        if (empty($current_master)) {
            log_message('warning', "No CDC cluster found with ID: {$cdc_id}");
            return $changes;
        }
        
        $current_master_host = $current_master[0]['master_node_host'];
        $current_master_port = $current_master[0]['master_node_port'];
        
        // 查询最近的主节点变更记录
        $sql = "SELECT old_leader_host, old_leader_port, new_leader_host, new_leader_port, change_time 
                FROM kunlun_metadata_db.cdc_leader_changes 
                WHERE cluster_id = ? 
                ORDER BY change_time DESC 
                LIMIT 1";
        $recent_changes = $this->Cluster_model->getList($sql, [$cdc_id]);
        
        if (!empty($recent_changes)) {
            $recent_change = $recent_changes[0];
            $change_time = $recent_change['change_time'];
            
            // 检查变更时间是否在最近1小时内
            $current_time = time();
            $change_time_stamp = strtotime($change_time);
            $time_diff = $current_time - $change_time_stamp;
            
            if ($time_diff <= 3600) { // 1小时内发生的变更
                $old_host = $recent_change['old_leader_host'];
                $old_port = $recent_change['old_leader_port'];
                $new_host = $recent_change['new_leader_host'];
                $new_port = $recent_change['new_leader_port'];
                
                $message = "CDC主备切换: 集群:{$cluster_name}, 原主节点:{$old_host}:{$old_port} -> 新主节点:{$new_host}:{$new_port}, 变更时间:{$change_time}";
                log_message('warning', $message);
                
                $change = [
                    'cluster_id' => $cdc_id,
                    'message' => $message,
                    'ip' => $new_host,
                    'port' => $new_port,
                    'old_leader_host' => $old_host,
                    'old_leader_port' => $old_port,
                    'change_time' => $change_time
                ];
                $changes[] = $change;
            }
        }
        
        return $changes;
    }
    
    /**
     * 检查CDC同步目标失败
     * @param int $cdc_id CDC集群ID
     * @param string $cluster_name CDC集群名称
     * @return array 同步失败列表
     */
    private function checkCdcSyncTargetFailed($cdc_id, $cluster_name)
    {
        log_message('debug', "Checking CDC sync target failures for cluster: {$cluster_name} (ID: {$cdc_id})");
        $failures = [];
        
        // 查询CDC同步任务
        $sql = "SELECT task_id, source_db, target_db, task_status, last_error, last_error_time 
                FROM kunlun_metadata_db.cdc_sync_tasks 
                WHERE cluster_id = ? AND (task_status = 'failed' OR task_status = 'error')";
        $tasks = $this->Cluster_model->getList($sql, [$cdc_id]);
        
        if (empty($tasks)) {
            log_message('debug', "No failed CDC sync tasks found for cluster: {$cluster_name}");
            return $failures;
        }
        
        foreach ($tasks as $task) {
            $task_id = $task['task_id'];
            $source_db = $task['source_db'];
            $target_db = $task['target_db'];
            $task_status = $task['task_status'];
            $last_error = $task['last_error'];
            $last_error_time = $task['last_error_time'];
            
            // 检查错误时间是否在最近24小时内
            $current_time = time();
            $error_time_stamp = strtotime($last_error_time);
            $time_diff = $current_time - $error_time_stamp;
            
            if ($time_diff <= 86400) { // 24小时内的错误
                $message = "CDC同步目标失败: 集群:{$cluster_name}, 任务ID:{$task_id}, 源库:{$source_db}, 目标库:{$target_db}, 状态:{$task_status}, 错误:{$last_error}, 时间:{$last_error_time}";
                log_message('warning', $message);
                
                // 获取CDC节点信息
                $sql = "SELECT node_host, node_port FROM kunlun_metadata_db.cdc_task_nodes WHERE task_id = ? LIMIT 1";
                $nodes = $this->Cluster_model->getList($sql, [$task_id]);
                
                $node_host = !empty($nodes) ? $nodes[0]['node_host'] : '';
                $node_port = !empty($nodes) ? $nodes[0]['node_port'] : '';
                
                $failure = [
                    'cluster_id' => $cdc_id,
                    'task_id' => $task_id,
                    'message' => $message,
                    'ip' => $node_host,
                    'port' => $node_port,
                    'source_db' => $source_db,
                    'target_db' => $target_db,
                    'error' => $last_error,
                    'error_time' => $last_error_time
                ];
                $failures[] = $failure;
            }
        }
        
        return $failures;
    }
    
    /**
     * 检查CDC解析DDL失败
     * @param int $cdc_id CDC集群ID
     * @param string $cluster_name CDC集群名称
     * @return array DDL解析失败列表
     */
    private function checkCdcParseDdlFailed($cdc_id, $cluster_name)
    {
        log_message('debug', "Checking CDC parse DDL failures for cluster: {$cluster_name} (ID: {$cdc_id})");
        $failures = [];
        
        // 查询CDC任务中的DDL解析失败记录
        $sql = "SELECT task_id, source_db, ddl_statement, error_message, error_time 
                FROM kunlun_metadata_db.cdc_ddl_errors 
                WHERE cluster_id = ? AND resolved = 'N'";
        $errors = $this->Cluster_model->getList($sql, [$cdc_id]);
        
        if (empty($errors)) {
            log_message('debug', "No DDL parse errors found for cluster: {$cluster_name}");
            return $failures;
        }
        
        foreach ($errors as $error) {
            $task_id = $error['task_id'];
            $source_db = $error['source_db'];
            $ddl_statement = $error['ddl_statement'];
            $error_message = $error['error_message'];
            $error_time = $error['error_time'];
            
            // 检查错误时间是否在最近24小时内
            $current_time = time();
            $error_time_stamp = strtotime($error_time);
            $time_diff = $current_time - $error_time_stamp;
            
            if ($time_diff <= 86400) { // 24小时内的错误
                // 截断DDL语句以适应消息长度
                $short_ddl = strlen($ddl_statement) > 100 ? substr($ddl_statement, 0, 100) . '...' : $ddl_statement;
                
                $message = "CDC解析DDL失败: 集群:{$cluster_name}, 任务ID:{$task_id}, 源库:{$source_db}, 错误:{$error_message}, DDL语句:{$short_ddl}, 时间:{$error_time}";
                log_message('warning', $message);
                
                // 获取CDC节点信息
                $sql = "SELECT node_host, node_port FROM kunlun_metadata_db.cdc_task_nodes WHERE task_id = ? LIMIT 1";
                $nodes = $this->Cluster_model->getList($sql, [$task_id]);
                
                $node_host = !empty($nodes) ? $nodes[0]['node_host'] : '';
                $node_port = !empty($nodes) ? $nodes[0]['node_port'] : '';
                
                $failure = [
                    'cluster_id' => $cdc_id,
                    'task_id' => $task_id,
                    'message' => $message,
                    'ip' => $node_host,
                    'port' => $node_port,
                    'source_db' => $source_db,
                    'ddl_statement' => $ddl_statement,
                    'error' => $error_message,
                    'error_time' => $error_time
                ];
                $failures[] = $failure;
            }
        }
        
        return $failures;
    }
    
    /**
     * 记录CDC告警信息
     * @param string $alarm_type 告警类型
     * @param array $alarm_data 告警数据
     */
    private function recordCdcAlarm($alarm_type, $alarm_data)
    {
        $cluster_id = $alarm_data['cluster_id'] ?? 0;
        $msg_data = urldecode(json_encode($alarm_data));
        
        // 检查是否已存在未处理的告警
        $select_sql = "SELECT job_info FROM kunlun_metadata_db.cluster_alarm_info 
                       WHERE alarm_type = ? AND status = 'unhandled' AND cluster_id = ?";
        $res_select = $this->Cluster_model->getList($select_sql, [$alarm_type, $cluster_id]);
        
        if (empty($res_select)) {
            // 获取告警级别
            $alarm_details = $this->AlarmCategory_model->getAlarmTypeDetail($alarm_type);
            $alarm_level = !empty($alarm_details) ? $alarm_details['level'] : 'WARNING';
            
            // 记录告警信息
            $insert_sql = "INSERT INTO kunlun_metadata_db.cluster_alarm_info
                           (alarm_type, alarm_level, job_info, occur_timestamp, cluster_id) 
                           VALUES (?, ?, ?, NOW(), ?)";
            $result = $this->Cluster_model->updateList($insert_sql, [
                $alarm_type,
                $alarm_level,
                $msg_data,
                $cluster_id
            ]);
            
            if ($result) {
                log_message('info', "CDC alarm recorded successfully: {$alarm_type}");
            } else {
                log_message('error', "Failed to record CDC alarm: {$alarm_type}");
            }
        } else {
            log_message('debug', "CDC alarm already exists: {$alarm_type}");
        }
    }
    
    /**
     * 发送批量告警
     * @param array $alarms 告警列表
     */
    private function sendBatchAlarms($alarms)
    {
        if (empty($alarms)) {
            log_message('debug', 'No alarms to send');
            return;
        }
        
        try {
            // 获取告警配置
            $alarm_users = [];
            if (!empty($this->alarm_config)) {
                foreach ($this->alarm_config as $alarm) {
                    if (!empty($alarm['accept_user']) && !empty($alarm['push_type'])) {
                        $alarm_users[$alarm['id']] = [
                            'accept_user' => $alarm['accept_user'],
                            'push_type' => $alarm['push_type']
                        ];
                    }
                }
            }
            
            // 获取用户信息
            $user_ids = [];
            foreach ($alarm_users as $type => $config) {
                $user_list = explode(',', $config['accept_user']);
                foreach ($user_list as $uid) {
                    if (!empty($uid) && !in_array($uid, $user_ids)) {
                        $user_ids[] = $uid;
                    }
                }
            }
            
            if (!empty($user_ids)) {
                $sql = "SELECT id, name, email, phone, role FROM kunlun_dba_tools_db.kunlun_user WHERE id IN (" . implode(',', $user_ids) . ")";
                $user_list = $this->Login_model->getList($sql);
                
                // 构造告警内容
                $content = "【昆仑CDC监控告警】\n\n";
                $content .= "发生时间: " . date('Y-m-d H:i:s') . "\n\n";
                $content .= "告警详情:\n";
                foreach ($alarms as $index => $alarm) {
                    $content .= ($index + 1) . ". " . $alarm . "\n";
                }
                
                // 发送通知
                if (!empty($user_list)) {
                    foreach ($user_list as $user) {
                        // 获取用户配置的告警类型
                        $user_push_types = [];
                        foreach ($alarm_users as $type => $config) {
                            if (in_array($user['id'], explode(',', $config['accept_user']))) {
                                $push_types = explode(',', $config['push_type']);
                                foreach ($push_types as $push_type) {
                                    if (!in_array($push_type, $user_push_types)) {
                                        $user_push_types[] = $push_type;
                                    }
                                }
                            }
                        }
                        
                        // 发送各类通知
                        foreach ($user_push_types as $push_type) {
                            switch ($push_type) {
                                case 'email':
                                    if (!empty($user['email'])) {
                                        $this->sendEmailNotification($user['email'], 'CDC监控告警', $content);
                                    }
                                    break;
                                case 'phone_message':
                                    if (!empty($user['phone'])) {
                                        $this->sendSmsNotification($user['phone'], $content);
                                    }
                                    break;
                                case 'wechat':
                                    $this->sendWechatNotification($user['id'], $content);
                                    break;
                            }
                        }
                    }
                }
            }
        } catch (Exception $e) {
            log_message('error', 'Send batch alarms error: ' . $e->getMessage());
        }
    }
    
    /**
     * 发送邮件通知
     * @param string $email 邮箱
     * @param string $subject 主题
     * @param string $content 内容
     */
    private function sendEmailNotification($email, $subject, $content)
    {
        try {
            // 获取邮件配置
            $email_config = [];
            foreach ($this->push_config as $config) {
                if ($config['type'] == 'email') {
                    $email_config = json_decode($config['message'], true);
                    break;
                }
            }
            
            if (empty($email_config)) {
                log_message('warning', 'Email configuration not found');
                return;
            }
            
            $mail = new PHPMailer(true);
//            $mail->CharSet = 'UTF-8';
//            $mail->isSMTP();
//            $mail->Host = $email_config['stmp'] ?? '';
//            $mail->SMTPAuth = true;
//            $mail->Username = $email_config['user_name'] ?? '';
//            $mail->Password = $email_config['password'] ?? '';
//            $mail->SMTPSecure = $email_config['encryption'] ?? 'ssl';
//            $mail->Port = 465;
//
//            $mail->setFrom($email_config['user_name'], '昆仑监控系统');
//            $mail->addAddress($email);
//            $mail->Subject = $subject;
//            $mail->Body = $content;
//            $mail->isHTML(false);
            
            $mail->send();
            log_message('info', "Email notification sent to: {$email}");
        } catch (Exception $e) {
            log_message('error', 'Send email notification error: ' . $e->getMessage());
        }
    }
    
    /**
     * 发送短信通知
     * @param string $phone 手机号
     * @param string $content 内容
     */
    private function sendSmsNotification($phone, $content)
    {
        try {
            // 获取短信配置
            $sms_config = [];
            foreach ($this->push_config as $config) {
                if ($config['type'] == 'phone_message') {
                    $sms_config = json_decode($config['message'], true);
                    break;
                }
            }
            
            if (empty($sms_config)) {
                log_message('warning', 'SMS configuration not found');
                return;
            }
            
            // 使用已加载的短信库服务
            $result = $this->aliyunsms->sendSms(
                $sms_config['AccessKeyId'],
                $sms_config['SecretKey'],
                $phone,
                $sms_config['TemplateId'],
                ['content' => substr($content, 0, 100)],
                $sms_config['SigId']
            );
            
            if ($result) {
                log_message('info', "SMS notification sent to: {$phone}");
            } else {
                log_message('warning', "Failed to send SMS notification to: {$phone}");
            }
        } catch (Exception $e) {
            log_message('error', 'Send SMS notification error: ' . $e->getMessage());
        }
    }
    
    /**
     * 发送微信通知
     * @param int $userId 用户ID
     * @param string $content 内容
     */
    private function sendWechatNotification($userId, $content)
    {
        try {
            // 获取微信配置
            $wechat_config = [];
            foreach ($this->push_config as $config) {
                if ($config['type'] == 'wechat') {
                    $wechat_config = json_decode($config['message'], true);
                    break;
                }
            }
            
            if (empty($wechat_config)) {
                log_message('warning', 'WeChat configuration not found');
                return;
            }
            
            // TODO: 实现微信通知的具体逻辑
            log_message('info', "WeChat notification sent to user: {$userId}");
        } catch (Exception $e) {
            log_message('error', 'Send WeChat notification error: ' . $e->getMessage());
        }
    }
    
    /**
     * 检查节点在线状态
     * @param string $host 主机
     * @param int $port 端口
     * @return bool 是否在线
     */
    private function checkNodeOnline($host, $port)
	{
        try {
            $fp = @fsockopen($host, $port, $errno, $errstr, 5);
            if (!$fp) {
                return false;
            }
            fclose($fp);
            return true;
        } catch (Exception $e) {
            log_message('error', "Check node online error: {$e->getMessage()}");
            return false;
        }
    }
} 
