<?php
defined('BASEPATH') or exit('No direct script access allowed');

class NotificationHandler {
    public $CI;

    public function __construct() {
        $this->CI =& get_instance();
        $this->CI->load->model('Cluster_model');
    }

    /**
     * 发送企业微信消息
     */
    public function sendWechatMessage($content, $type = 'text') {
        try {
            $config = $this->getWechatConfig();
            if (empty($config)) {
                log_message('error', 'WeChat configuration not found');
                return false;
            }

            $access_token = $this->getWechatAccessToken($config);
            if (!$access_token) {
                log_message('error', 'Failed to get WeChat access token');
                return false;
            }

            $url = "https://qyapi.weixin.qq.com/cgi-bin/message/send?access_token={$access_token}";
            $data = [
                'touser' => '@all',
                'msgtype' => $type,
                'agentid' => $config['agentid'],
                $type => ['content' => $content]
            ];

            $response = $this->sendHttpRequest($url, json_encode($data));
            $result = json_decode($response, true);

            return isset($result['errcode']) && $result['errcode'] == 0;
        } catch (Exception $e) {
            log_message('error', 'Failed to send WeChat message: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * 获取企业微信配置
     */
    private function getWechatConfig() {
        $sql = "SELECT message FROM cluster_alarm_message_config WHERE type = 'wechat'";
        $result = $this->CI->Cluster_model->getList($sql);
        $config = null;
        if (!empty($result)) {
            $config = json_decode($result[0]['message'], true);
        }
        return $config;
    }

    /**
     * 获取企业微信访问令牌
     */
    private function getWechatAccessToken($config) {
        $url = "https://qyapi.weixin.qq.com/cgi-bin/gettoken?corpid={$config['corpid']}&corpsecret={$config['secret']}";
        $response = $this->sendHttpRequest($url);
        $result = json_decode($response, true);
        if (isset($result['access_token'])) {
            return $result['access_token'];
        } else {
            throw new Exception('Failed to get WeChat access token');
        }
    }

    /**
     * 发送 HTTP 请求
     */
    private function sendHttpRequest($url, $postData = null) {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
        
        if ($postData) {
            curl_setopt($ch, CURLOPT_POST, 1);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $postData);
        }
        
        $proxy = $this->getProxyConfig();
        if ($proxy) {
            curl_setopt($ch, CURLOPT_PROXY, $proxy['host']);
            curl_setopt($ch, CURLOPT_PROXYPORT, $proxy['port']);
            if (!empty($proxy['user']) && !empty($proxy['password'])) {
                curl_setopt($ch, CURLOPT_PROXYUSERPWD, "{$proxy['user']}:{$proxy['password']}");
            }
        }
        
        $response = curl_exec($ch);
        $error = curl_error($ch);
        curl_close($ch);
        
        if ($error) {
            throw new Exception("cURL Error: {$error}");
        }
        
        return $response;
    }

    /**
     * 获取代理配置
     */
    private function getProxyConfig() {
        $sql = "SELECT message FROM cluster_alarm_message_config WHERE type = 'proxy'";
        $result = $this->CI->Cluster_model->getList($sql);
        $config = null;
        if (!empty($result)) {
            $config = json_decode($result[0]['message'], true);
        }
        return $config;
    }

    /**
     * 通过代理发送消息
     */
    public function sendThroughProxy($message, $type) {
        try {
            $proxy = $this->getProxyConfig();
            if (!$proxy['enabled']) {
                return false;
            }

            $data = [
                'type' => $type,
                'content' => $message,
                'timestamp' => time(),
                'sign' => $this->generateProxySign($message, $proxy['secret'])
            ];

            $response = $this->sendHttpRequest($proxy['url'], json_encode($data));
            $this->logPushAttempt('proxy', $message, $response);
            return $response;
        } catch (Exception $e) {
            log_message('error', 'Failed to send through proxy: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * 生成代理请求签名
     */
    private function generateProxySign($message, $secret) {
        return hash_hmac('sha256', $message . time(), $secret);
    }

    /**
     * 记录推送尝试
     */
    private function logPushAttempt($type, $content, $response) {
        $log_sql = "INSERT INTO cluster_alarm_push_log (alarm_type, push_type, content, content_res, create_at) VALUES (?, ?, ?, ?, ?)";
        $this->CI->Cluster_model->updateList($log_sql, [
            'notification',
            $type,
            $content,
            $response,
            time()
        ]);
    }
} 