<?php
defined('BASEPATH') or exit('No direct script access allowed');

class RequestValidator {
    protected $errors = [];
    protected $data = [];

    public function __construct($data = []) {
        $this->data = $data;
    }

    public function validate($rules) {
        foreach ($rules as $field => $rule) {
            if (!$this->validateField($field, $rule)) {
                return false;
            }
        }
        return true;
    }

    protected function validateField($field, $rules) {
        $value = isset($this->data[$field]) ? $this->data[$field] : null;
        $ruleArray = explode('|', $rules);

        foreach ($ruleArray as $rule) {
            $params = [];
            if (strpos($rule, ':') !== false) {
                list($rule, $param) = explode(':', $rule);
                $params = explode(',', $param);
            }

            $method = 'validate' . ucfirst($rule);
            if (method_exists($this, $method)) {
                if (!$this->$method($field, $value, $params)) {
                    return false;
                }
            }
        }

        return true;
    }

    protected function validateRequired($field, $value) {
        if ($value === null || $value === '') {
            $this->errors[$field] = "{$field} 不能为空";
            return false;
        }
        return true;
    }

    protected function validateNumeric($field, $value) {
        if ($value !== null && !is_numeric($value)) {
            $this->errors[$field] = "{$field} 必须是数字";
            return false;
        }
        return true;
    }

    protected function validateMin($field, $value, $params) {
        if ($value !== null && $value < $params[0]) {
            $this->errors[$field] = "{$field} 不能小于 {$params[0]}";
            return false;
        }
        return true;
    }

    protected function validateMax($field, $value, $params) {
        if ($value !== null && $value > $params[0]) {
            $this->errors[$field] = "{$field} 不能大于 {$params[0]}";
            return false;
        }
        return true;
    }

    protected function validateEmail($field, $value) {
        if ($value !== null && !filter_var($value, FILTER_VALIDATE_EMAIL)) {
            $this->errors[$field] = "{$field} 必须是有效的邮箱地址";
            return false;
        }
        return true;
    }

    protected function validateIp($field, $value) {
        if ($value !== null && !filter_var($value, FILTER_VALIDATE_IP)) {
            $this->errors[$field] = "{$field} 必须是有效的IP地址";
            return false;
        }
        return true;
    }

    protected function validatePort($field, $value) {
        if ($value !== null && (!is_numeric($value) || $value < 1 || $value > 65535)) {
            $this->errors[$field] = "{$field} 必须是有效的端口号(1-65535)";
            return false;
        }
        return true;
    }

    public function getErrors() {
        return $this->errors;
    }

    public function hasErrors() {
        return !empty($this->errors);
    }
} 