package protocol

import (
	"bytes"
	loggers "consistent_sql/logger"
	"encoding/binary"
	"encoding/json"
	"fmt"
	"io"
	"net"
	"strconv"
	"strings"
)

type MessageType uint32

const (
	MReqinitcfg MessageType = iota
	MReqdumpjob
	MRspdumpjob
	MReqrowmsg
	MRsprowmsg
	MRspstatemsg
)

type Message struct {
	Type MessageType `json:"type"`
	Data string      `json:"data"`
}

type SendMessage struct {
	Type MessageType `json:"type"`
	Data string      `json:"data"`
}

type SendMessageData struct {
	Type MessageType `json:"type"`
	Data string      `json:"data"`
}

func (m *Message) ToJSON() ([]byte, error) {
	return json.Marshal(m)
}

func (m MessageType) String() string {
	switch m {
	case MReqinitcfg:
		return "M_ReqInitCfg"
	case MReqdumpjob:
		return "M_ReqDumpJob"
	case MRspdumpjob:
		return "M_RspDumpJob"
	case MReqrowmsg:
		return "M_ReqRowMsg"
	case MRsprowmsg:
		return "M_RspRowMsg"
	case MRspstatemsg:
		return "M_RspStateMsg"
	default:
		return "Unknown"
	}
}

// formatBytesHex 将字节数组格式化为十六进制表示，便于日志打印
func formatBytesHex(data []byte) string {
	var hexStr strings.Builder
	for i := 0; i < len(data); i++ {
		hexStr.WriteString(fmt.Sprintf("%02X", data[i]))
		if i < len(data)-1 {
			hexStr.WriteString(" ")
		}
	}
	return hexStr.String()
}

func SendMsg(conn net.Conn, val []byte) error {
	// 计算长度
	dataLen := uint32(len(val))

	// 创建完整的数据包（长度字段+内容）
	buf := new(bytes.Buffer)

	// 写入长度（4字节小端序）
	if err := binary.Write(buf, binary.LittleEndian, dataLen); err != nil {
		loggers.Printf("写入长度字段失败: %v", err.Error())
		return fmt.Errorf("写入长度字段失败: %v", err)
	}

	// 写入数据内容
	if _, err := buf.Write(val); err != nil {
		loggers.Printf("写入数据内容失败: %v", err.Error())
		return fmt.Errorf("写入数据内容失败: %v", err)
	}

	// 获取完整数据包
	fullPacket := buf.Bytes()
	packetLen := len(fullPacket)

	// 记录日志
	loggers.Printf("发送数据包总长度: %d 字节", packetLen)

	// 打印数据包内容（限制最多100字节）
	// if packetLen > 0 {
	// 	if packetLen > 100 {
	// 		loggers.Printf("数据包内容(前100字节): %s", formatBytesHex(fullPacket[:100]))
	// 	} else {
	// 		loggers.Printf("数据包内容(全部%d字节): %s", packetLen, formatBytesHex(fullPacket))
	// 	}
	// }

	loggers.Printf("发送数据包内容: %s", formatBytesHex(fullPacket))
	loggers.Printf("发送数据包长度: %d", packetLen)
	loggers.Printf("================================================")

	// 发送数据包
	if _, err := conn.Write(fullPacket); err != nil {
		// 增强客户端中断日志记录
		if err == io.EOF || strings.Contains(err.Error(), "use of closed network connection") {
			loggers.Printf("发送消息时客户端断开连接: %v, 客户端地址: %s", err, conn.RemoteAddr().String())
		} else if ne, ok := err.(net.Error); ok {
			if ne.Timeout() {
				loggers.Printf("发送消息超时: %v, 客户端地址: %s", err, conn.RemoteAddr().String())
			} else if ne.Temporary() {
				loggers.Printf("发送消息临时网络错误: %v, 客户端地址: %s", err, conn.RemoteAddr().String())
			} else {
				loggers.Printf("发送消息致命网络错误: %v, 客户端地址: %s", err, conn.RemoteAddr().String())
			}
		} else {
			loggers.Printf("发送数据包失败: %v, 客户端地址: %s", err, conn.RemoteAddr().String())
		}
		return fmt.Errorf("发送数据包失败: %v", err)
	}

	return nil
}

// SendError 发送错误消息
func SendError(conn net.Conn, errorCode, errorMsg string, jobId string) {
	// RspDumpJob := map[string]interface{}{
	// 	"error_code": errorCode,
	// 	"error_msg":  errorMsg,
	// 	"jobId":      jobId,
	// }
	// marshal, _ := json.Marshal(RspDumpJob)

	// response := &SendMessage{
	// 	Type: MRspdumpjob,
	// 	Data: string(marshal),
	// }
	// respData, _ := json.Marshal(response)

	// 初始化消息buffer
	var message []byte

	// 1. 类型 (MRsprowmsg)
	messageType := byte(MRspdumpjob)
	message = append(message, messageType)
	//错误码
	errorCodeInt, _ := strconv.Atoi(errorCode)
	message = append(message, byte(int8(errorCodeInt)))
	//错误信息

	// 增加错误处理和日志记录
	if err := SendMsg(conn, message); err != nil {
		loggers.Printf("发送错误消息失败(JobID: %s): %v", jobId, err)
	}
}

// SendResponse 发送通用响应消息
func SendResponse(conn net.Conn, msgType MessageType, data string) error {
	response := &SendMessage{
		Type: msgType,
		Data: data,
	}
	respData, _ := json.Marshal(response)
	err := SendMsg(conn, respData)
	if err != nil {
		loggers.Printf("发送响应消息失败(类型: %s): %v", msgType.String(), err)
	}
	return err
}
