package main

import (
	"context"
	"fmt"
	"log"
	"time"

	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/bson/primitive"
	"go.mongodb.org/mongo-driver/mongo"
	"go.mongodb.org/mongo-driver/mongo/options"
)

// 测试 MongoDB changeStream 功能的独立脚本
func testChangeStream() {
	// MongoDB 连接配置 - 请根据实际情况修改
	mongoHost := "localhost:27017"
	username := "your_username" // 请修改为实际用户名
	password := "your_password" // 请修改为实际密码
	authDatabase := "admin"     // 认证数据库，通常是 admin
	dbName := "test_db"
	collectionName := "test_collection"

	// 构建带认证的 MongoDB URL
	var mongoURL string
	if username != "" && password != "" {
		mongoURL = fmt.Sprintf("mongodb://%s:%s@%s/%s", username, password, mongoHost, authDatabase)
	} else {
		mongoURL = fmt.Sprintf("mongodb://%s", mongoHost)
	}

	fmt.Println("=== MongoDB Change Stream 测试工具 ===")
	fmt.Printf("连接主机: %s\n", mongoHost)
	fmt.Printf("用户名: %s\n", username)
	fmt.Printf("认证数据库: %s\n", authDatabase)
	fmt.Printf("目标数据库: %s\n", dbName)
	fmt.Printf("集合: %s\n", collectionName)

	// 连接 MongoDB
	ctx := context.Background()
	client, err := mongo.Connect(ctx, options.Client().ApplyURI(mongoURL))
	if err != nil {
		log.Fatalf("连接 MongoDB 失败: %v", err)
	}
	defer client.Disconnect(ctx)

	// 测试连接
	err = client.Ping(ctx, nil)
	if err != nil {
		log.Fatalf("MongoDB ping 失败: %v", err)
	}
	fmt.Println("✓ MongoDB 连接成功")

	// 检查是否为副本集
	result := client.Database("admin").RunCommand(ctx, bson.D{{"isMaster", 1}})
	var isMasterResult bson.M
	if err := result.Decode(&isMasterResult); err != nil {
		log.Fatalf("检查副本集状态失败: %v", err)
	}

	if setName, exists := isMasterResult["setName"]; exists {
		fmt.Printf("✓ 检测到副本集: %v\n", setName)
	} else {
		fmt.Println("⚠️  警告: 未检测到副本集，Change Stream 可能无法工作")
		fmt.Println("   请确保 MongoDB 运行在副本集模式下")
	}

	// 获取集合
	collection := client.Database(dbName).Collection(collectionName)

	// 创建测试数据（如果集合为空）
	count, err := collection.CountDocuments(ctx, bson.M{})
	if err != nil {
		log.Fatalf("统计文档数量失败: %v", err)
	}

	if count == 0 {
		fmt.Println("集合为空，插入测试数据...")
		testDocs := []interface{}{
			bson.M{"name": "test1", "value": 100, "created": time.Now()},
			bson.M{"name": "test2", "value": 200, "created": time.Now()},
			bson.M{"name": "test3", "value": 300, "created": time.Now()},
		}
		_, err = collection.InsertMany(ctx, testDocs)
		if err != nil {
			log.Fatalf("插入测试数据失败: %v", err)
		}
		fmt.Printf("✓ 插入了 %d 条测试数据\n", len(testDocs))
	} else {
		fmt.Printf("✓ 集合中已有 %d 条文档\n", count)
	}

	// 设置 Change Stream 选项
	changeStreamOptions := options.ChangeStream()
	changeStreamOptions.SetFullDocument(options.UpdateLookup)

	// 设置监听 update 和 delete 操作的 pipeline
	pipeline := mongo.Pipeline{
		{{
			"$match", bson.D{{
				"operationType", bson.D{{
					"$in", bson.A{"update", "delete"},
				}},
			}},
		}},
	}

	fmt.Printf("Pipeline: %+v\n", pipeline)

	fmt.Println("\n=== 启动 Change Stream 监听 ===")
	fmt.Println("监听操作类型: update, delete")
	fmt.Println("按 Ctrl+C 停止监听")

	// 创建 Change Stream
	changeStream, err := collection.Watch(ctx, pipeline, changeStreamOptions)
	if err != nil {
		log.Fatalf("创建 Change Stream 失败: %v", err)
	}
	defer changeStream.Close(ctx)

	fmt.Println("✓ Change Stream 启动成功，等待变更事件...")

	// 启动一个 goroutine 来模拟数据变更
	go func() {
		time.Sleep(3 * time.Second)
		fmt.Println("\n--- 开始模拟数据变更 ---")

		// 模拟 update 操作
		filter := bson.M{"name": "test1"}
		update := bson.M{"$set": bson.M{"value": 150, "updated": time.Now()}}
		_, err := collection.UpdateOne(ctx, filter, update)
		if err != nil {
			fmt.Printf("更新操作失败: %v\n", err)
		} else {
			fmt.Println("✓ 执行了 update 操作")
		}

		time.Sleep(2 * time.Second)

		// 模拟 delete 操作
		deleteFilter := bson.M{"name": "test3"}
		_, err = collection.DeleteOne(ctx, deleteFilter)
		if err != nil {
			fmt.Printf("删除操作失败: %v\n", err)
		} else {
			fmt.Println("✓ 执行了 delete 操作")
		}

		time.Sleep(2 * time.Second)

		// 再次更新
		filter2 := bson.M{"name": "test2"}
		update2 := bson.M{"$set": bson.M{"value": 250, "updated": time.Now()}}
		_, err = collection.UpdateOne(ctx, filter2, update2)
		if err != nil {
			fmt.Printf("第二次更新操作失败: %v\n", err)
		} else {
			fmt.Println("✓ 执行了第二次 update 操作")
		}
	}()

	// 监听变更事件
	eventCount := 0
	timeout := time.After(15 * time.Second) // 15秒后超时

	for {
		select {
		case <-timeout:
			fmt.Println("\n=== 测试超时，结束监听 ===")
			if eventCount == 0 {
				fmt.Println("❌ 未收到任何 Change Stream 事件")
				fmt.Println("可能的原因:")
				fmt.Println("1. MongoDB 未运行在副本集模式")
				fmt.Println("2. 权限不足")
				fmt.Println("3. 网络连接问题")
			} else {
				fmt.Printf("✓ 总共收到 %d 个事件\n", eventCount)
			}
			return

		default:
			if changeStream.Next(ctx) {
				eventCount++
				var changeDoc bson.M
				if err := changeStream.Decode(&changeDoc); err != nil {
					fmt.Printf("❌ 解析变更事件失败: %v\n", err)
					continue
				}

				fmt.Printf("\n--- Change Stream 事件 #%d ---\n", eventCount)
				fmt.Printf("操作类型: %v\n", changeDoc["operationType"])

				if ns, ok := changeDoc["ns"].(bson.M); ok {
					fmt.Printf("数据库: %v\n", ns["db"])
					fmt.Printf("集合: %v\n", ns["coll"])
				}

				if documentKey, ok := changeDoc["documentKey"].(bson.M); ok {
					if id, exists := documentKey["_id"]; exists {
						if objectID, ok := id.(primitive.ObjectID); ok {
							fmt.Printf("文档ID: %s\n", objectID.Hex())
						} else {
							fmt.Printf("文档ID: %v\n", id)
						}
					}
				}

				if fullDocument, ok := changeDoc["fullDocument"]; ok && fullDocument != nil {
					fmt.Printf("完整文档: %v\n", fullDocument)
				}

				if resumeToken := changeStream.ResumeToken(); resumeToken != nil {
					fmt.Printf("Resume Token: %v\n", resumeToken)
				}

				fmt.Println("原始事件:", changeDoc)
			} else if err := changeStream.Err(); err != nil {
				fmt.Printf("❌ Change Stream 错误: %v\n", err)
				return
			}
		}
	}
}
