package database

import (
	"context"
	"encoding/json"
	"fmt"
	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/mongo"
	"go.mongodb.org/mongo-driver/mongo/options"
	"log"
	"sync"
)

type Document struct {
	ID     string `pg:"__sys_obj_id__,pk"`
	Doc    string `pg:"doc,type:jsonb"`
	OpType string `pg:"optype"`
}

func createPgSchemaAndTable(schemaName, tableName string) {
	// Create schema if not exists
	_, err := PgDB.Exec(fmt.Sprintf("CREATE SCHEMA IF NOT EXISTS %s", schemaName))
	if err != nil {
		log.Fatalf("Failed to create schema: %v", err)
	}

	createTableSQL := fmt.Sprintf(`
		CREATE TABLE IF NOT EXISTS %S.%S (
    __sys_obj_id__ VARCHAR PRIMARY KEY,
    doc JSONB,
    optype VARCHAR
);

	`, schemaName, tableName)

	_, err = PgDB.Exec(createTableSQL)
	if err != nil {
		return
	}

}

func upsertToPostgreSQL(schemaName, tableName, docID string, document bson.M, operationType string) {
	// 确保 PostgreSQL 中的 schema 和表已经创建
	createPgSchemaAndTable(schemaName, tableName)

	// 将 BSON 文档转换为 JSON 字符串
	docJSON, err := json.Marshal(document)
	if err != nil {
		log.Fatalf("Failed to marshal document to JSON: %v", err)
	}

	// 构建插入或更新的 SQL 语句，使用 fmt.Sprintf 动态设置 schemaName 和 tableName
	insertSQL := fmt.Sprintf(`
	INSERT INTO %s.%s (__sys_obj_id__, doc, optype)
	VALUES ($1, $2, $3)
	ON CONFLICT (__sys_obj_id__) 
	DO UPDATE SET doc = EXCLUDED.doc, optype = EXCLUDED.optype;
	`, schemaName, tableName)

	// 执行 SQL 语句
	_, err = PgDB.Exec(insertSQL, docID, string(docJSON), operationType)
	if err != nil {
		log.Fatalf("Failed to upsert into PostgreSQL: %v", err)
	}
}

func handleDeleteOperation(schemaName, tableName, docID string) {
	_, err := PgDB.Exec(fmt.Sprintf(
		"UPDATE %s.%s SET optype = ? WHERE __sys_obj_id__ = ?", schemaName, tableName),
		"delete", docID)
	if err != nil {
		log.Fatalf("Failed to handle delete operation: %v", err)
	}
	log.Printf("Delete operation updated in PostgreSQL: id = %s", docID)
}

func WatchMongoChanges(dbName, collectionName string, wg *sync.WaitGroup) {
	defer wg.Done()
	collection := MongoClient.Database(dbName).Collection(collectionName)

	csOptions := options.ChangeStream().SetFullDocument(options.UpdateLookup)
	cs, err := collection.Watch(context.TODO(), mongo.Pipeline{}, csOptions)
	if err != nil {
		log.Fatalf("Failed to watch collection: %v", err)
	}
	defer cs.Close(context.TODO())

	log.Printf("Watching MongoDB changes on %s.%s", dbName, collectionName)

	for cs.Next(context.TODO()) {
		var change bson.M
		if err := cs.Decode(&change); err != nil {
			log.Fatalf("Failed to decode change: %v", err)
		}

		operationType := change["operationType"].(string)
		docID := change["documentKey"].(bson.M)["_id"].(string)

		if operationType == "delete" {
			handleDeleteOperation(dbName, collectionName, docID)
		} else {
			fullDocument := change["fullDocument"].(bson.M)
			upsertToPostgreSQL(dbName, collectionName, docID, fullDocument, operationType)
		}
	}
}
