package routes

import (
	"context"
	"encoding/json"
	"fmt"
	"github.com/gin-gonic/gin"
	"go.mongodb.org/mongo-driver/bson"
	"log"
	"net/http"
	"streamServer/database"
	"strings"
	"sync"
)

func SetupConfigRouter() *gin.Engine {
	router := gin.Default()

	router.GET("/config/mongodb", func(c *gin.Context) {
		c.JSON(http.StatusOK, gin.H{
			"uri": "mongodb://192.168.0.19:27018/?replicaSet=rs0",
		})
	})

	router.GET("/config/postgresql", func(c *gin.Context) {
		c.JSON(http.StatusOK, gin.H{
			"dbname":   "postgres",
			"user":     "abc",
			"password": "abc",
			"host":     "192.168.0.20",
			"port":     "23001",
		})
	})

	router.GET("/config/mysql", func(c *gin.Context) {
		c.JSON(http.StatusOK, gin.H{
			"dbname":   "postgres.test",
			"user":     "abc",
			"password": "abc",
			"host":     "192.168.0.176",
			"port":     "47002",
		})
	})

	router.GET("/config/kafka", func(c *gin.Context) {
		c.JSON(http.StatusOK, gin.H{
			"bootstrap_servers": "192.168.0.128:59092",
			"topic_prefix":      "my",
		})
	})

	router.GET("/config/elasticsearch", func(c *gin.Context) {
		c.JSON(http.StatusOK, gin.H{
			"host": "http://192.168.0.19:9200",
		})
	})

	return router
}

var runningTasks = sync.Map{}

// 处理 MongoDB 任务的函数
func handleMongoDB(DataInfo database.SourceDataInfo) error {
	// 连接 MongoDB
	err := database.ConnectMongoDB(DataInfo.SrcInfo.RcrMeta)
	if err != nil || database.MongoClient.Ping(context.Background(), nil) != nil {
		return fmt.Errorf("无法连接到MongoDB: %v", err)
	}

	// 连接元数据库
	MetAddress := strings.Split(DataInfo.DestInfo.RcrMeta, ",")[0]
	MetAddr := strings.Split(MetAddress, ":")
	err = database.ConnectMysql(MetAddr[0], "pgx", "pgx_pwd", MetAddr[1], "kunlun_metadata_db")
	if err != nil || database.PgDB.Ping() != nil {
		return fmt.Errorf("无法连接到元数据库: %v", err)
	}

	var wg sync.WaitGroup
	dbNames, err := database.MongoClient.ListDatabaseNames(context.TODO(), bson.M{})
	if err != nil {
		return fmt.Errorf("Failed to list database names: %v", err)
	}

	for _, dbName := range dbNames {
		if dbName != "admin" && dbName != "local" && dbName != "config" {
			collectionNames, err := database.MongoClient.Database(dbName).ListCollectionNames(context.TODO(), bson.M{})
			if err != nil {
				log.Printf("Failed to list collection names for database %s: %v", dbName, err)
				continue
			}

			for _, collectionName := range collectionNames {
				wg.Add(1)
				go func(dbName, collectionName string) {
					defer wg.Done()
					database.WatchMongoChanges(dbName, collectionName, &wg)
				}(dbName, collectionName)
			}
		}
	}
	wg.Wait()

	// 关闭连接
	defer database.PgDB.Close()
	defer database.MongoClient.Disconnect(context.TODO())
	return nil
}

// 处理 Klustron 任务的函数
func handleKlustron(DataInfo database.SourceDataInfo) error {
	// Klustron 的处理逻辑
	return nil
}

func SetupSyncRouter() *gin.Engine {
	router := gin.Default()

	router.GET("/sync", func(c *gin.Context) {
		var db = database.DB
		var ClusterSteamTask []database.ClusterSteamTask

		err := db.Raw("SELECT task_id, status, source_data_info FROM cluster_steam_task").Scan(&ClusterSteamTask).Error
		if err != nil {
			c.JSON(http.StatusInternalServerError, gin.H{"error": "无法连接到数据库: " + err.Error()})
			return
		}

		for _, task := range ClusterSteamTask {
			if task.Status == 0 {
				// 检查任务是否已经存在
				if _, exists := runningTasks.Load(task.TaskID); exists {
					log.Printf("任务 %s 已经在运行中", task.TaskID)
					continue
				}

				var DataInfo database.SourceDataInfo
				err := json.Unmarshal([]byte(task.SourceDataInfo), &DataInfo)
				if err != nil {
					c.JSON(http.StatusInternalServerError, gin.H{"error": "sourceDataInfo解析错误: " + err.Error()})
					return
				}

				// 将任务标记为正在运行
				runningTasks.Store(task.TaskID, struct{}{})
				// 针对每个 case 启动独立的异步任务
				switch DataInfo.DbType {
				case "Mongodb":
					go func(taskID string, DataInfo database.SourceDataInfo) {
						defer runningTasks.Delete(taskID) // 任务完成后移除
						err := handleMongoDB(DataInfo)
						if err != nil {
							log.Printf("Mongodb 任务 %s 失败: %v", taskID, err)
						}
					}(task.TaskID, DataInfo)

				case "Klustron":
					go func(taskID string, DataInfo database.SourceDataInfo) {
						defer runningTasks.Delete(taskID) // 任务完成后移除
						err := handleKlustron(DataInfo)
						if err != nil {
							log.Printf("Klustron 任务 %s 失败: %v", taskID, err)
						}
					}(task.TaskID, DataInfo)
				}
			}
		}

		// 前端快速返回响应
		c.JSON(http.StatusOK, gin.H{"status": "数据同步任务已启动"})
	})

	return router
}
