<?php
defined('BASEPATH') or exit('No direct script access allowed');

class AlarmConfig extends MY_Controller
{
    public $Cluster_model;
    public $Login_model;
    public $db;
    protected $alarm_service;

    public function __construct()
    {
        parent::__construct();

        header('Access-Control-Allow-Origin:*');
        header('Access-Control-Allow-Headers: Content-Type,Content-Length,Accept-Encoding,X-Requested-with, Origin');
        header('Access-Control-Allow-Methods:POST,GET,OPTIONS,DELETE');
        header('Access-Control-Allow-Headers:x-requested-with,content-type,Token');
        header('Content-Type: application/json;charset=utf-8');

        // 加载模型和服务
        $this->load->model('Cluster_model');
        $this->load->model('Login_model');
        $this->db = $this->load->database('default', true);
        $this->load->library('AlarmService');
        $this->alarm_service = new AlarmService();
    }

    /**
     * 统一的JSON响应方法
     */
    protected function jsonResponse($data = [], $code = 200, $message = 'Success')
    {
        $response = [
            'code' => $code,
            'message' => $message
        ];
        
        if (!empty($data)) {
            $response['data'] = $data;
        }
        
        header('Content-Type: application/json; charset=utf-8');
        echo json_encode($response);
        return;
    }

    /**
     * Token验证方法
     */
    protected function validateToken($token)
    {
        if (empty($token)) {
            throw new Exception('Token不能为空');
        }

        $res_token = $this->Login_model->getToken($token, 'D', config_item('key'));
        if (empty($res_token)) {
            throw new Exception('Token验证失败');
        }

        // 验证用户是否存在
        $sql = "SELECT count(id) as count FROM kunlun_user WHERE name = ?";
        $res = $this->Login_model->getList($sql, [$res_token]);
        if (empty($res) || $res[0]['count'] == 0) {
            throw new Exception('用户不存在');
        }

        return $res_token;
    }

    /**
     * 获取并验证请求头中的Token
     */
    protected function getRequestToken()
    {
        $headers = apache_request_headers();
        return isset($headers['Token']) ? $headers['Token'] : '';
    }

    public function getConfig()
    {
        try {
            // 验证Token
            $token = $this->getRequestToken();
            $this->validateToken($token);

            $sql = "SELECT * FROM kunlun_metadata_db.cluster_alarm_message_config";
            $config = $this->Cluster_model->getList($sql);

            return $this->jsonResponse([
                'list' => $config ?: [],
                'total' => count($config)
            ]);

        } catch (Exception $e) {
            log_message('error', 'Get alarm config failed: ' . $e->getMessage());
            return $this->jsonResponse([], 500, $e->getMessage());
        }
    }

    public function updateConfig()
    {
        log_message('debug', '=== updateAlarmConfig() Start ===');

        try {
            // 验证Token
            $token = $this->getRequestToken();
            $this->validateToken($token);

            // 获取POST数据
            $post_data = $this->getPostData();
            log_message('debug', 'Post data: ' . json_encode($post_data));

            // 验证必需字段
            if (empty($post_data['type'])) {
                throw new Exception('告警类型不能为空');
            }

            $current_time = time();
            $message_json = json_encode($post_data, JSON_UNESCAPED_UNICODE);

            // 使用预处理语句防止SQL注入
            $checkSql = "SELECT id FROM cluster_alarm_message_config WHERE type = ?";
            $existingConfig = $this->Cluster_model->getList($checkSql, [$post_data['type']]);

            if (!empty($existingConfig)) {
                // 更新配置
                $sql = "UPDATE cluster_alarm_message_config SET 
                    `message` = ?, 
                    `update_at` = ? 
                    WHERE `type` = ?";
                $params = [$message_json, $current_time, $post_data['type']];
            } else {
                // 新增配置
                $sql = "INSERT INTO cluster_alarm_message_config 
                    (`message`, `create_at`, `type`) 
                    VALUES (?, ?, ?)";
                $params = [$message_json, $current_time, $post_data['type']];
            }

            $result = $this->Cluster_model->updateList($sql, $params);
            if (!$result) {
                throw new Exception('配置更新失败');
            }

            log_message('debug', '=== updateAlarmConfig() End ===');
            return $this->jsonResponse([], 200, '配置更新成功');

        } catch (Exception $e) {
            log_message('error', 'Update alarm config failed: ' . $e->getMessage());
            return $this->jsonResponse([], 500, $e->getMessage());

        }

    }

    /**
     * 获取代理配置
     */
    public function getProxyConfig()
    {
        log_message('debug', '=== getProxyConfig() Start ===');

        try {
            // 验证Token
            $token = $this->getRequestToken();
            $this->validateToken($token);

            // 获取代理配置
            $sql = "SELECT 
                    server_address, 
                    template_id,
                    create_at, 
                    update_at 
                FROM cluster_alarm_proxy_config 
                LIMIT 1";

            $config = $this->Cluster_model->getList($sql);
            
            if ($config) {
                // 格式化时间戳
                $config[0]['create_at'] = date('Y-m-d H:i:s', $config[0]['create_at']);
                if (!empty($config[0]['update_at'])) {
                    $config[0]['update_at'] = date('Y-m-d H:i:s', $config[0]['update_at']);
                }
            }

            log_message('debug', '=== getProxyConfig() End ===');
            return $this->jsonResponse($config ? $config[0] : null);

        } catch (Exception $e) {
            log_message('error', 'Get proxy config failed: ' . $e->getMessage());
            return $this->jsonResponse([], 500, $e->getMessage());
        }
    }

    /**
     * 更新代理配置
     */
    public function updateProxyConfig()
    {
        log_message('debug', '=== updateProxyConfig() Start ===');

        try {
            // 验证Token
            $token = $this->getRequestToken();
            $this->validateToken($token);

            // 获取POST数据
            $post_data = $this->getPostData();
            log_message('debug', 'Post data: ' . json_encode($post_data));

            // 验证必需字段
            if (empty($post_data['serverAddress'])) {
                throw new Exception('服务器地址不能为空');
            }

            // 检查是否存在配置
            $checkSql = "SELECT id FROM cluster_alarm_proxy_config LIMIT 1";
            $existingConfig = $this->Cluster_model->getList($checkSql);

            if (!empty($existingConfig)) {
                // 更新配置
                $sql = "UPDATE cluster_alarm_proxy_config SET 
                    server_address = ?, 
                    template_id = ?, 
                    update_at = ?";
                $params = [
                    $post_data['serverAddress'],
                    $post_data['templateId'] ?? '',
                    time()
                ];
            } else {
                // 新增配置
                $sql = "INSERT INTO cluster_alarm_proxy_config 
                    (server_address, template_id, create_at) 
                    VALUES (?, ?, ?)";
                $params = [
                    $post_data['serverAddress'],
                    $post_data['templateId'] ?? '',
                    time()
                ];
            }

            $result = $this->Cluster_model->updateList($sql, $params);
            if (!$result) {
                throw new Exception('代理配置更新失败');
            }

            return $this->jsonResponse([], 200, '代理配置更新成功');

        } catch (Exception $e) {
            log_message('error', 'Update proxy config failed: ' . $e->getMessage());
            return $this->jsonResponse([], 500, $e->getMessage());
        }
    }

    /**
     * 测试代理连接
     */
    public function testProxyConnection()
    {
        log_message('debug', '=== testProxyConnection() Start ===');

        try {
            // 验证Token
            $token = $this->getRequestToken();
            $this->validateToken($token);

            // 获取POST数据
            $post_data = $this->getPostData();
            log_message('debug', 'Post data: ' . json_encode($post_data));

            $result = $this->alarm_service->testProxyConnection(
                $post_data['serverAddress'],
                $post_data['templateId'] ?? ''
            );

            return $this->jsonResponse([], 200, $result['message']);

        } catch (Exception $e) {
            log_message('error', 'Test proxy connection failed: ' . $e->getMessage());
            return $this->jsonResponse([], 500, $e->getMessage());
        }
    }

    /**
     * 测试短信发送
     */
    public function testSmsAlarm()
    {
        log_message('debug', '=== testSmsAlarm() Start ===');

        try {
            // 验证Token
            $token = $this->getRequestToken();
            $this->validateToken($token);

            // 获取POST数据
            $post_data = $this->getPostData();
            log_message('debug', 'Post data: ' . json_encode($post_data));

            $result = $this->alarm_service->testSmsAlarm($post_data['phone']);
            return $this->jsonResponse([], 200, $result['message']);

        } catch (Exception $e) {
            log_message('error', 'Test SMS alarm failed: ' . $e->getMessage());
            return $this->jsonResponse([], 500, $e->getMessage());
        }
    }

    /**
     * 测试邮件发送
     */
    public function testEmailAlarm()
    {
        log_message('debug', '=== testEmailAlarm() Start ===');

        try {
            // 验证Token
            $token = $this->getRequestToken();
            $this->validateToken($token);

            // 获取POST数据
            $post_data = $this->getPostData();
            log_message('debug', 'Post data: ' . json_encode($post_data));

            $result = $this->alarm_service->testEmailAlarm($post_data['email']);
            return $this->jsonResponse([], 200, $result['message']);

        } catch (Exception $e) {
            log_message('error', 'Test email alarm failed: ' . $e->getMessage());
            return $this->jsonResponse([], 500, $e->getMessage());
        }
    }

    /**
     * 测试企业微信发送
     */
    public function testWechatAlarm()
    {
        log_message('debug', '=== testWechatAlarm() Start ===');

        try {
            // 验证Token
            $token = $this->getRequestToken();
            $this->validateToken($token);

            $result = $this->alarm_service->testWechatAlarm();
            return $this->jsonResponse([], 200, $result['message']);

        } catch (Exception $e) {
            log_message('error', 'Test wechat alarm failed: ' . $e->getMessage());
            return $this->jsonResponse([], 500, $e->getMessage());
        }
    }
} 