# 一致性SQL PB级数据同步方案

## 1. 方案概述

本方案旨在优化consistent_sql组件，使其能够支持PB（Petabyte）级别的数据同步。通过引入流式处理、数据分块、并行传输、压缩和检查点机制等技术手段，有效提升大规模数据同步的性能和可靠性。方案特别强化了GTID（全局事务ID）作为断点记忆的核心机制，确保数据同步的事务一致性。

## 2. 系统架构

### 2.1 整体架构

```
+------------------+     +-----------------+     +------------------+
|                  |     |                 |     |                  |
|  数据源          |     |  consistent_sql |     |  目标存储        |
|  (ES/MongoDB/TD) | --> |  (流式同步引擎) | --> |  (存储服务)      |
|                  |     |                 |     |                  |
+------------------+     +-----------------+     +------------------+
                              ^      |
                              |      v
                         +------------------+
                         |                  |
                         |  检查点存储      |
                         |  (断点续传)      |
                         |                  |
                         +------------------+
```

### 2.2 组件说明

1. **流式同步引擎**：基于Unix Socket的数据传输服务
2. **数据源适配器**：支持ES、MongoDB、TDengine等数据源
3. **任务管理器**：管理同步任务的创建、监控和终止
4. **检查点管理**：支持基于GTID的断点续传状态管理
5. **并行处理器**：多线程并行处理数据块
6. **压缩模块**：数据压缩与解压缩服务
7. **GTID追踪器**：跟踪全局事务ID，确保数据同步的准确性

## 3. 关键优化策略

### 3.1 数据分块处理

将PB级数据切分为可管理的小块进行处理：

- **块大小**：默认256MB，可根据内存限制动态调整
- **元数据追踪**：每个块包含唯一ID、序列号、大小等元数据
- **按需加载**：仅在需要时加载数据块，避免内存溢出

### 3.2 并行数据流

多线程并行处理数据，提高吞吐量：

- **动态线程池**：根据系统资源自动调整工作线程数
- **任务调度**：数据块分配到不同工作线程，避免资源争用
- **结果收集**：异步收集处理结果，减少等待时间

### 3.3 数据压缩

减少网络传输和存储开销：

- **自适应压缩级别**：根据CPU负载和网络情况调整压缩级别
- **选择性压缩**：对适合压缩的数据类型进行压缩
- **压缩算法**：使用gzip算法，平衡压缩率和性能

### 3.4 基于GTID的检查点与恢复机制

支持基于GTID的断点续传，确保数据同步的可靠性与一致性：

- **GTID跟踪**：使用全局事务ID跟踪数据处理状态
- **周期性检查点**：定期保存包含GTID的同步状态
- **原子性保存**：确保检查点数据的一致性
- **快速恢复**：从最近检查点的GTID位置恢复任务
- **事务一致性**：确保不会漏掉或重复处理事务

### 3.5 内存管理

防止内存溢出，优化资源利用：

- **内存使用上限**：设置最大内存使用量
- **缓冲区池化**：重用内存缓冲区，减少GC压力
- **渐进式处理**：流式处理数据，避免一次加载过多数据

### 3.6 错误处理与重试

提高系统稳定性：

- **自动重试**：遇到临时性错误时自动重试
- **退避策略**：指数退避，避免频繁重试
- **故障隔离**：单个任务失败不影响其他任务

## 4. 性能指标

| 指标            | 目标值                | 说明                                      |
|----------------|----------------------|------------------------------------------|
| 吞吐量          | >1TB/小时             | 单节点处理速度                             |
| 延迟            | <10秒                 | 从数据源到目标的端到端延迟                   |
| 内存使用         | <8GB                 | 单进程内存占用峰值                          |
| CPU使用率       | <80%                 | 平均CPU使用率                              |
| 错误率          | <0.001%              | 数据处理错误率                             |
| 恢复时间        | <30秒                 | 从故障中恢复的时间                          |

## 5. 部署方案

### 5.1 系统要求

- **操作系统**：Linux (CentOS 7+/Ubuntu 18.04+)
- **CPU**：8核以上，推荐16核
- **内存**：16GB以上，推荐32GB
- **磁盘**：SSD，至少100GB可用空间
- **网络**：千兆以上，推荐万兆网络

### 5.2 部署步骤

1. **编译构建**：
   ```bash
   cd KunlunMonitor/go/consistent_sql
   go build -o consistent_sql_pb
   ```

2. **配置**：
   - 创建配置文件目录
   - 设置检查点存储路径
   - 配置日志级别和路径

3. **启动服务**：
   ```bash
   ./consistent_sql_pb -workDir=/data/kunlun -logDir=/var/log/kunlun -serverUnixFile=/tmp/kunlun.sock -maxConcurrent=100 -chunkSize=256 -parallelStreams=8 -enableCompression=true
   ```

4. **监控配置**：
   - 设置服务监控和告警
   - 配置日志收集和分析

### 5.3 水平扩展

对于超大规模数据同步任务，可以部署多个节点：

- **分片策略**：按数据源、时间范围或ID范围分片
- **负载均衡**：请求分发到多个节点
- **结果合并**：合并多个节点的处理结果

## 6. 使用指南

### 6.1 通用连接与请求格式

所有数据同步操作都使用Unix Socket通信，以下是基本连接流程：

```go
// 连接到Unix socket
conn, err := net.Dial("unix", "/path/to/kunlun.sock")
if err != nil {
    log.Fatalf("连接失败: %v", err)
}
defer conn.Close()

// 构建请求消息
requestMsg := &protocol.Message{
    Type: protocol.MReqdumpjob,
    Data: string(taskJSON),
}

// 序列化并发送请求
marshal, _ := json.Marshal(requestMsg)
if err := protocol.SendMsg(conn, marshal); err != nil {
    log.Fatalf("发送请求失败: %v", err)
}

// 读取并处理响应
// ...
```

### 6.2 创建同步任务

#### 6.2.1 Elasticsearch同步任务

```json
{
  "job_id": "es_sync_job_1",          // 任务ID，必填
  "url": "http://elasticsearch:9200", // ES URL，必填
  "db_user": "elastic",               // ES用户名，可选
  "db_passwd": "password",            // ES密码，可选
  "db_database": "my_index",          // ES索引名，必填
  "db_table": "_doc",                 // 文档类型，通常为"_doc"，必填
  "dump_db_type": "es",               // 数据库类型，使用"es"而非"elasticsearch"，必填
  "gtid": "doc_id_12345",             // 文档ID作为GTID起始点，必填
  "event_type": "create",             // 事件类型，必填
  
  // ES特有可选参数
  "es_query": "{ \"match_all\": {} }", // ES查询DSL
  "es_scroll_time": "1m",             // scroll保持时间，默认"1m"
  "es_batch_size": 1000,              // 每批获取文档数，默认1000
  "index_type": "_doc",               // 索引类型，通常为"_doc"
  
  // PB级同步可选参数
  "use_stream_mode": true,            // 是否使用流模式
  "chunk_size": 256,                  // 分块大小(MB)
  "parallel_streams": 8,              // 并行数据流数量
  "enable_compression": true,         // 是否启用压缩
  "compression_level": 6              // 压缩级别(1-9)
}
```

#### 6.2.2 MongoDB同步任务

```json
{
  "job_id": "mongo_sync_job_1",       // 任务ID，必填
  "url": "mongodb://localhost:27017", // MongoDB连接URI，必填
  "db_user": "admin",                 // MongoDB用户名，必填
  "db_passwd": "password",            // MongoDB密码，必填
  "db_database": "my_database",       // 数据库名，必填
  "db_table": "my_collection",        // 集合名，必填
  "dump_db_type": "mongodb",          // 数据库类型，必填
  "gtid": "507f1f77bcf86cd799439011", // ObjectID作为GTID，必填
  "event_type": "create",             // 事件类型，必填
  
  // MongoDB特有可选参数
  "mongo_query": "{}",                // MongoDB查询过滤条件
  "mongo_batch_size": 1000,           // 每批获取文档数
  "mongo_use_gte": true,              // 查询时使用$gte而非$gt
  
  // PB级同步可选参数
  "use_stream_mode": true,            // 是否使用流模式
  "chunk_size": 256,                  // 分块大小(MB)
  "parallel_streams": 8               // 并行数据流数量
}
```

#### 6.2.3 TDengine同步任务

```json
{
  "job_id": "td_sync_job_1",          // 任务ID，必填
  "url": "http://tdengine:6041/rest", // TDengine REST接口URL，必填
  "db_user": "root",                  // TDengine用户名，必填
  "db_passwd": "taosdata",            // TDengine密码，必填
  "db_database": "my_database",       // 数据库名，必填
  "db_table": "my_stable",            // 超级表名，必填
  "dump_db_type": "tdengine",         // 数据库类型，必填
  "gtid": "1609459200000",            // 时间戳作为GTID，必填
  "event_type": "create",             // 事件类型，必填
  
  // TDengine特有可选参数
  "td_time_field": "ts",              // 时间戳字段名
  "td_batch_size": 1000,              // 每批获取数据条数
  "td_time_range": "1h",              // 每次查询的时间窗口
  
  // PB级同步可选参数
  "use_stream_mode": true             // 是否使用流模式
}
```

### 6.3 获取任务状态

```json
{
  "job_id": "es_sync_job_1",     // 任务ID，必填
  "event_type": "status",        // 事件类型，必填为"status"
  "dump_db_type": "es",          // 与创建任务时使用相同的数据库类型
  "use_stream_mode": true        // 如果是流模式任务则设为true
}
```

响应示例：
```json
{
  "error_code": "0",
  "error_msg": "",
  "jobId": "es_sync_job_1",
  "status": true,
  "progress": {
    "processed_chunks": 1024,
    "total_chunks": -1,
    "bytes_processed": 268435456,
    "last_gtid": "doc_id_67890"
  },
  "last_checkpoint": {
    "last_sequence_num": 1024,
    "bytes_processed": 268435456,
    "gtid": "doc_id_67890"
  }
}
```

### 6.4 任务管理命令

#### 6.4.1 停止任务

```json
{
  "job_id": "es_sync_job_1",      // 任务ID，必填
  "event_type": "stop",           // 事件类型，必填为"stop"
  "dump_db_type": "es",           // 与创建任务时使用相同的数据库类型
  "use_stream_mode": true         // 如果是流模式任务则设为true
}
```

#### 6.4.2 删除任务

```json
{
  "job_id": "es_sync_job_1",      // 任务ID，必填
  "event_type": "remove",         // 事件类型，必填为"remove"
  "dump_db_type": "es",           // 与创建任务时使用相同的数据库类型
  "use_stream_mode": true         // 如果是流模式任务则设为true
}
```

### 6.5 命令行测试工具

#### 6.5.1 Elasticsearch测试命令示例

```bash
# 创建ES同步任务
go test -v elasticsearch_test.go -run TestESStreamFormat \
  -serverUnixFile=/tmp/kunlun.sock \
  -workType=create \
  -jobId=es_test_job \
  -esURL=http://localhost:9200 \
  -esIndex=my_index \
  -esType=_doc

# 查询ES任务状态
go test -v elasticsearch_test.go -run TestESStreamFormat \
  -serverUnixFile=/tmp/kunlun.sock \
  -workType=status \
  -jobId=es_test_job
```

#### 6.5.2 MongoDB测试命令示例

```bash
# 创建MongoDB同步任务
go test -v mongodb_test.go -run TestMongoStreamFormat \
  -serverUnixFile=/tmp/kunlun.sock \
  -workType=create \
  -jobId=mongo_test_job \
  -mongoURL=mongodb://localhost:27017 \
  -mongoDb=my_database \
  -mongoColl=my_collection

# 查询MongoDB任务状态
go test -v mongodb_test.go -run TestMongoStreamFormat \
  -serverUnixFile=/tmp/kunlun.sock \
  -workType=status \
  -jobId=mongo_test_job
```

### 6.6 数据传输格式

数据传输采用二进制格式，以下是格式说明：

1. **消息类型** (1字节): 用于标识消息类型，例如`MReqrowmsg`(13)
2. **行数量** (4字节): 当前批次的总行数
3. **GTID** (长度+字符串): 数据源的标识符(ES文档ID、MongoDB ObjectID等)
4. **版本号** (4字节): 用于标识数据版本，ES中为0
5. **数据类型** (1字节): 操作类型(13=插入, 14=更新, 15=删除)
6. **数据库名** (长度+字符串): 数据库或索引名称
7. **表/集合名** (长度+字符串): 表、集合或文档类型名称
8. **GTID重复** (长度+字符串): 再次写入GTID(兼容性设计)
9. **查询参数** (4字节): ES和MongoDB中通常为0
10. **字段数量** (4字节): 数据中的字段数
11. **字段名数组**: 每个字段的名称(长度+字符串)
12. **字段属性数组**: 每个字段的属性(4字节整数，通常为0)
13. **文档数量** (4字节): 实际发送的文档数
14. **文档数据**: 每行数据的每个字段值(长度+字符串)

### 6.7 GTID获取方法

#### 6.7.1 Elasticsearch GTID获取

对于Elasticsearch，GTID通常是文档ID，可以通过以下方式获取：

```bash
# 获取索引中按ID排序的第一条记录
curl -X GET "http://your-es-host:9200/your-index/_search?size=1&sort=_id:asc"
```

从结果中提取`_id`字段值作为初始GTID。

#### 6.7.2 MongoDB GTID获取

对于MongoDB，GTID通常是ObjectID，可以通过以下方式获取：

```javascript
// 在MongoDB shell中执行
db.your_collection.find().sort({_id: 1}).limit(1).pretty()
```

从结果中提取`_id`字段值作为初始GTID。

#### 6.7.3 TDengine GTID获取

对于TDengine，GTID通常是时间戳，可以通过以下方式获取：

```sql
-- 在TDengine命令行中执行
SELECT _c0 FROM your_table ORDER BY _c0 ASC LIMIT 1;
```

从结果中提取时间戳值作为初始GTID。

### 6.8 实际使用案例

#### 6.8.1 从Elasticsearch同步数据到目标存储

```go
package main

import (
    "encoding/json"
    "fmt"
    "net"
    "os"
)

func main() {
    // 1. 连接到Unix socket
    conn, err := net.Dial("unix", "/tmp/kunlun.sock")
    if err != nil {
        fmt.Printf("连接失败: %v\n", err)
        os.Exit(1)
    }
    defer conn.Close()
    
    // 2. 构建ES同步任务
    esTask := map[string]interface{}{
        "job_id":       "es_prod_sync",
        "url":          "http://es-cluster:9200",
        "db_user":      "elastic",
        "db_passwd":    "secure_password",
        "db_database":  "logs-2023.06",
        "db_table":     "_doc",
        "dump_db_type": "es",
        "gtid":         "AWmSqOiCEtW-3YEP8M7Q",
        "event_type":   "create",
        "es_batch_size": 2000,
        "es_scroll_time": "2m",
        "use_stream_mode": true,
        "parallel_streams": 4
    }
    
    // 3. 序列化任务
    taskData, _ := json.Marshal(esTask)
    
    // 4. 构建请求消息
    requestMsg := map[string]interface{}{
        "Type": 1, // MReqdumpjob
        "Data": string(taskData),
    }
    
    // 5. 发送请求
    marshal, _ := json.Marshal(requestMsg)
    
    // 实现SendMsg函数
    // ...
    
    fmt.Println("ES同步任务已提交")
}
```

#### 6.8.2 监控同步进度并显示状态

```go
package main

import (
    "encoding/json"
    "fmt"
    "net"
    "time"
)

func main() {
    jobId := "es_prod_sync"
    
    for {
        // 1. 连接到Unix socket
        conn, _ := net.Dial("unix", "/tmp/kunlun.sock")
        
        // 2. 构建状态查询请求
        statusTask := map[string]interface{}{
            "job_id":         jobId,
            "event_type":     "status",
            "dump_db_type":   "es",
            "use_stream_mode": true
        }
        
        taskData, _ := json.Marshal(statusTask)
        requestMsg := map[string]interface{}{
            "Type": 1, // MReqdumpjob
            "Data": string(taskData),
        }
        
        marshal, _ := json.Marshal(requestMsg)
        // 发送请求
        // ...
        
        // 3. 接收并解析响应
        // 解析响应数据
        // ...
        
        fmt.Printf("同步进度: %d/%d 块, 已处理 %.2f GB, 最新GTID: %s\n", 
            progress.ProcessedChunks, 
            progress.TotalChunks,
            float64(progress.BytesProcessed)/(1024*1024*1024),
            progress.LastGtid)
            
        conn.Close()
        time.Sleep(30 * time.Second)
    }
}
```

## 7. GTID断点续传机制

### 7.1 GTID概念

GTID（Global Transaction ID，全局事务标识符）是确保数据同步一致性的关键机制。在本系统中，GTID作为断点记忆的核心标识，用于：

1. **定位同步位置**：精确标识事务在源数据库中的位置
2. **避免重复同步**：确保每个事务只被处理一次
3. **保证顺序性**：维持事务的先后顺序
4. **支持故障恢复**：在系统崩溃后从正确位置恢复

### 7.2 GTID格式

不同数据库系统的GTID格式有所不同：

- **MySQL**：`server_uuid:transaction_id`，例如 `3E11FA47-71CA-11E1-9E33-C80AA9429562:23`
- **MongoDB**：时间戳格式的ObjectId，例如 `507f1f77bcf86cd799439011`
- **TDengine**：整数时间戳，例如 `1609459200000`
- **Elasticsearch**：自定义格式，如 `index-name:sequence-number`，或直接使用文档ID

### 7.3 GTID处理流程

1. **初始化**：同步任务必须提供初始GTID
2. **跟踪**：系统持续跟踪并更新当前处理的GTID
3. **存储**：定期将最新GTID保存到检查点
4. **恢复**：任务重启时从保存的GTID位置继续处理
5. **验证**：确保GTID值的合法性和有效性

### 7.4 断点续传实现

1. **创建任务时**：必须提供有效的GTID值
2. **处理数据时**：持续更新最新处理的GTID
3. **保存检查点时**：将GTID与其他元数据一起保存
4. **恢复任务时**：优先使用检查点中的GTID
5. **查询状态时**：返回当前处理的GTID信息

### 7.5 Elasticsearch GTID获取方法

对于Elasticsearch数据源，获取第一条记录ID作为GTID的方法如下：

1. **自动获取机制**：
   - 系统现已支持在任务首次运行且GTID为空时，自动获取第一条记录ID作为起始GTID
   - 使用`getFirstESDocument`函数实现，基于排序查询获取最小ID

2. **手动获取方法**：
   - 使用以下命令获取索引中按ID排序的第一条记录:
   ```bash
   curl -X GET "http://your-es-host:9200/your-index/_search?size=1&sort=_id:asc"
   ```
   - 从结果中提取`_id`字段值，作为初始GTID

3. **在任务创建时使用**：
   ```json
   {
     "job_id": "es_sync_job_1",
     "url": "http://elasticsearch:9200",
     "es_index": "your-index",
     "gtid": "获取到的文档ID", // 填入获取到的ID
     // 其他参数...
   }
   ```

4. **空GTID处理**：
   - 当提供空GTID时，系统将尝试自动获取第一条记录ID
   - 如果自动获取失败，将记录警告日志并从头开始同步

### 7.6 同步机制优化

为确保不遗漏任何记录，系统进行了如下优化：

1. **MongoDB查询边界优化**：
   - 默认使用`$gt`（大于）操作符查询，现已更新为可选择使用`$gte`（大于等于）操作符
   - 避免因边界条件导致记录遗漏的问题

2. **Elasticsearch时间范围查询优化**：
   - 更新查询条件从`gt`（大于）到`gte`（大于等于）
   - 确保不会遗漏边界时间点的记录

3. **空结果处理优化**：
   - 即使查询返回空结果，同步过程也会继续，保持任务存活
   - 对无数据情况采用指数退避策略，减少无效查询

4. **数据批次统一**：
   - 无论获取到数据量是否达到批次大小，都会确保发送
   - 解决可能丢失最后几条不足一批的数据问题

## 8. 实施建议

### 8.1 阶段性部署

1. **概念验证**：
   - 在测试环境部署单节点服务
   - 同步小规模数据（~10GB）验证功能

2. **小规模试点**：
   - 在生产环境同步单个数据源的有限数据（~100GB）
   - 监控系统性能，调整参数

3. **规模化部署**：
   - 逐步扩大同步范围
   - 添加多个数据源和目标

4. **全面部署**：
   - 全量数据同步
   - 定期优化和调整

### 8.2 优化建议

- **网络优化**：调整TCP参数，使用长连接和窗口缩放
- **批处理**：根据数据特性调整批处理大小
- **资源隔离**：为重要任务分配专用资源
- **监控告警**：设置性能和错误阈值告警

## 9. 性能测试与调优

### 9.1 测试场景

1. **小数据量测试**：同步10GB数据，验证功能完整性
2. **中等数据量测试**：同步100GB数据，验证稳定性
3. **大数据量测试**：同步1TB数据，验证性能和资源使用
4. **持续负载测试**：连续运行24小时，验证稳定性
5. **故障恢复测试**：模拟各类故障，验证恢复能力
6. **GTID恢复测试**：验证基于GTID的断点续传功能

### 9.2 调优策略

- **批处理大小**：调整batch_size，寻找最佳平衡点
- **并行度**：调整parallel_streams，根据CPU核心数优化
- **内存分配**：调整内存使用上限，避免OOM
- **压缩级别**：调整压缩级别，平衡CPU和网络使用
- **网络参数**：优化TCP参数，提高网络吞吐量
- **GTID更新频率**：调整GTID检查点保存频率，平衡性能和可靠性

## 10. 注意事项和限制

- 初始同步可能需要较长时间，应在非高峰期进行
- 超大规模数据同步应考虑增加硬件资源
- 定期清理检查点和日志文件，避免磁盘空间不足
- 对业务关键数据进行额外备份
- 网络带宽是PB级数据同步的主要瓶颈，应确保充足的网络资源
- **GTID必须提供**：所有同步任务创建时必须提供有效的GTID
- **不同数据源GTID格式不同**：使用时需注意各数据源GTID的特点

## 11. 未来优化方向

- 支持更多数据源和目标存储系统
- 引入智能调度，自动优化任务参数
- 添加数据验证功能，确保同步数据的完整性
- 实现分布式协同处理，支持跨节点负载均衡
- 开发Web管理界面，简化任务管理和监控
- 支持数据转换和过滤，提供ETL能力
- **高级GTID管理**：实现自动GTID探测和管理
- **多源GTID协调**：支持多数据源之间的GTID映射和转换 

## 2.3 三种数据源同步流转图示


# 三种数据源同步流转函数细化

## 2.3.1 Elasticsearch数据同步流转与函数映射

```
+------------------+                  +---------------------+                 +------------------+
|                  |                  |                     |                 |                  |
|  Elasticsearch   |                  |   consistent_sql    |                 |   目标存储        |
|                  |                  |                     |                 |                  |
+--------+---------+                  +----------+----------+                 +--------+---------+
         |                                       |                                     |
         |                                       |                                     |
         |  1. 初始化请求                        |                                     |
         +-------------------------------------->+                                     |
         |                                       |  StartTaskES()                      |
         |  2. ES API调用                        |                                     |
         +<--------------------------------------+                                     |
         |                                       |                                     |
         |                                       |  3. syncESTask()                    |
         |                                       | +-----------------------+           |
         |                                       | | • 构建批处理参数       |           |
         |                                       | | • 构建滚动API查询     |           |
         |                                       | | • 记录LastScrollID   |           |           
         |                                       | +-----------------------+           |
         |                                       |                                     |
         |                                       |  4. sendSyncResult()                |
         |                                       +------------------------------------>+
         |                                       |                                     |
         |                                       |  5. 接收确认                        |
         |                                       +<------------------------------------+
         |                                       |                                     |
         |                                       |  6. updateTaskStatus()              |
         |                                       |                                     |
         |                     +----------------+|                                     |
         |                     |                ||                                     |
         |                     | ESTaskStatus   |<+                                     |
         |                     | LastScrollID   ||                                     |
         |                     |                |+                                     |
         |                     +----------------+                                      |
         |                                       |                                     |
```

## 2.3.2 MongoDB数据同步流转与函数映射

```
+------------------+                  +---------------------+                 +------------------+
|                  |                  |                     |                 |                  |
|    MongoDB       |                  |   consistent_sql    |                 |   目标存储        |
|                  |                  |                     |                 |                  |
+--------+---------+                  +----------+----------+                 +--------+---------+
         |                                       |                                     |
         |                                       |                                     |
         |  1. 连接初始化                        |                                     |
         +-------------------------------------->+                                     |
         |                                       |  StartTaskMongo()                   |
         |  2. 数据获取                          |                                     |
         +<--------------------------------------+                                     |
         |                                       |                                     |
         |                                       |  3. syncMongoTask()                 |
         |                                       | +------------------------+          |
         |                                       | | • 连接MongoDB         |          |
         |                                       | | • SyncMongoTask()     |          |
         |                                       | | • 跟踪ObjectId作为GTID |          |
         |                                       | +------------------------+          |
         |                                       |                                     |
         |                                       |  4. sendSyncResult()                |
         |                                       +------------------------------------>+
         |                                       |                                     |
         |                                       |  5. 响应接收                        |
         |                                       +<------------------------------------+
         |                                       |                                     |
         |                                       |  6. updateTaskStatus()              |
         |                                       |                                     |
         |                    +-----------------+|                                     |
         |                    |                 ||                                     |
         |                    | MongoTaskStatus |<+                                    |
         |                    | LastID (GTID)   ||                                    |
         |                    |                 |+                                     |
         |                    +-----------------+                                      |
         |                                       |                                     |
```

## 2.3.3 TDengine数据同步流转与函数映射

```
+------------------+                  +---------------------+                 +------------------+
|                  |                  |                     |                 |                  |
|    TDengine      |                  |   consistent_sql    |                 |   目标存储        |
|                  |                  |                     |                 |                  |
+--------+---------+                  +----------+----------+                 +--------+---------+
         |                                       |                                     |
         |                                       |                                     |
         |  1. 时间范围查询                      |                                     |
         +-------------------------------------->+                                     |
         |                                       |  StartTaskTD()                      |
         |  2. 返回时间窗口数据                  |                                     |
         +<--------------------------------------+                                     |
         |                                       |                                     |
         |                                       |  3. syncTDTask()                    |
         |                                       | +-------------------------+         |
         |                                       | | • getTDClient()        |         |
         |                                       | | • SyncTdTask()         |         |
         |                                       | | • 时间戳作为GTID        |         |
         |                                       | +-------------------------+         |
         |                                       |                                     |
         |                                       |  4. sendSyncResult()                |
         |                                       +------------------------------------>+
         |                                       |                                     |
         |                                       |  5. 响应接收                        |
         |                                       +<------------------------------------+
         |                                       |                                     |
         |                                       |  6. updateTaskStatus()              |
         |                                       |                                     |
         |                    +-----------------+|                                     |
         |                    |                 ||                                     |
         |                    | TdTaskStatus    |<+                                    |
         |                    | LastTS (时间戳)  ||                                    |
         |                    |                 |+                                     |
         |                    +-----------------+                                      |
         |                                       |                                     |
```

## 2.3.4 PB级流式同步流转与函数映射

```
+------------------+                  +---------------------+                 +------------------+
|                  |                  |                     |                 |                  |
|  数据源          |                  |   consistent_sql    |                 |   目标存储        |
|                  |                  |                     |                 |                  |
+--------+---------+                  +----------+----------+                 +--------+---------+
         |                                       |                                     |
         |                                       |                                     |
         |  1. 流任务请求                        |                                     |
         +-------------------------------------->+                                     |
         |                                       |  StreamTaskManager.StartStreamTask()|
         |  2. 数据获取                          |                                     |
         +<--------------------------------------+                                     |
         |                                       |                                     |
         |                                       |  3. 根据数据源类型选择处理函数       |
         |                                       | +-------------------------------+   |
         |                                       | | • processMongoStream()       |   |
         |                                       | | • processTDEngineStream()    |   |
         |                                       | | • processESStream()          |   |
         |                                       | +-------------------------------+   |
         |                                       |                                     |
         |                                       |  4. 数据发送                        |
         |                                       +------------------------------------>+
         |                                       |                                     |
         |                                       |  5. 响应接收                        |
         |                                       +<------------------------------------+
         |                                       |                                     |
         |                                       |  6. saveCheckpoint()                |
         |                                       |                                     |
         |                    +-----------------+|                                     |
         |                    |                 ||                                     |
         |                    | Checkpoint      |<+                                    |
         |                    | Gtid, Metadata  ||                                    |
         |                    |                 |+                                     |
         |                    +-----------------+                                      |
         |                                       |                                     |
```

## 2.4 函数说明与工作流程

### 2.4.1 Elasticsearch同步函数工作流程

1. `StartTaskES()`: 初始化ES同步任务，创建任务状态并启动工作协程
   - 设置FirstSend=true标记首次发送
   - 设置WaitingAck=false初始化等待确认状态
   - 创建context和取消函数进行任务管理
   
2. `syncESTask()`: 执行ES同步任务的主要流程
   - 获取或创建ES HTTP客户端
   - 构建批处理参数和滚动API查询
   - 迭代处理ES数据
   - 注册确认消息处理回调函数
   - **增量同步机制**：同步到最后一条数据后不退出，持续监听新数据
   - **消息确认机制**：
     - 首次发送数据(FirstSend=true)不等待确认，继续处理
     - 后续发送后(FirstSend=false)将WaitingAck设为true，等待确认
     - 收到确认后(MRsprowmsg)将WaitingAck设为false，继续处理
   - **断点记忆**：使用LastScrollID和LastTimestamp作为断点

3. `getESClient()`: 获取或创建优化过的ES HTTP客户端
   - 维护连接池，重用HTTP客户端
   - 优化连接参数，提高性能

4. `sendSyncResult()`: 将处理后的数据发送到目标系统
   - **消息格式**：采用与MongoDB相同的二进制协议格式
     - 消息类型(MReqrowmsg)：标识消息类型
     - 行数(uint32)：当前批次的总行数
     - GTID(长度+字符串)：ES文档ID作为GTID
     - binlog_file(int32)：ES无binlog，置为0
     - 数据类型(int8)：默认使用insert=13
     - 数据库名(长度+字符串)："elasticsearch"
     - 索引名(长度+字符串)：ES索引名
     - GTID再次写入(长度+字符串)：重复写入GTID
     - SQL(int32)：ES无SQL，置为0
     - 列数量(uint32)：根据实际字段数确定
     - 列名数组：每列的名称(长度+字符串)
     - 列属性数组：每列的属性(uint32，均为0)
     - 行数(uint32)：每次发送批量数据
     - 行内容：包含每列的值(长度+字符串)
   - **错误重试**：引入重试机制，失败后自动重试

5. `updateTaskStatusFull()`: 更新任务状态，包括LastScrollID、LastTimestamp、FirstSend和WaitingAck状态
   - 全面记录任务状态，支持断点续传和消息确认机制

6. `ESTaskStatus`: 包含IsRunning、LastScrollID、LastTimestamp、FirstSend和WaitingAck等任务状态信息
   - FirstSend：标记是否是首次发送数据
   - WaitingAck：标记是否正在等待确认

7. `registerAckHandler()` 和 `HandleAckMessage()`：
   - 注册和处理确认消息的回调函数
   - 支持不同消息类型的处理
   - 确保异步消息处理的线程安全

### 2.4.2 MongoDB同步函数工作流程

1. `StartTaskMongo()`: 初始化MongoDB同步任务，创建任务状态并启动工作协程
   - 设置FirstSend=true标记首次发送
   - 设置WaitingAck=false初始化等待确认状态
   - 创建context和取消函数进行任务管理

2. `syncMongoTask()`: 处理MongoDB同步任务的主要流程
   - 连接MongoDB数据库
   - 基于GTID查询数据
   - 迭代处理数据变更
   - 注册确认消息处理回调函数
   - **增量同步机制**：同步到最后一条数据后不退出，持续监听新数据
   - **消息确认机制**：
     - 首次发送数据(FirstSend=true)不等待确认，继续处理
     - 后续发送后(FirstSend=false)将WaitingAck设为true，等待确认
     - 收到确认后(MRsprowmsg)将WaitingAck设为false，继续处理
   - **断点记忆**：使用LastID(ObjectID)作为断点

3. `SyncMongoTask()`: 实际查询并同步MongoDB数据的函数
   - 构建ObjectID过滤查询(_id > lastID)
   - 设置批处理大小和排序
   - 返回文档批次和最后ObjectID

4. `sendSyncResult()`: 将处理后的数据发送到目标系统
   - **数据格式**：二进制消息包含以下字段
     - 消息类型(MReqrowmsg)：标识消息类型
     - 行数(uint32)：当前批次的总行数
     - GTID(长度+字符串)：MongoDB的ObjectId作为GTID
     - binlog_file(int32)：MongoDB无binlog，置为0
     - 数据类型(int8)：插入=13/更新=14/删除=15
     - 数据库名(长度+字符串)：MongoDB数据库名
     - 表名(长度+字符串)：MongoDB集合名
     - GTID再次写入(长度+字符串)：重复写入GTID
     - SQL(int32)：MongoDB无SQL，置为0
     - 列数量(uint32)：MongoDB使用3列(_id, data, optype)
     - 列名数组：每列的名称(长度+字符串)
     - 列属性数组：每列的属性(uint32，均为0)
     - 行数(uint32)：批量发送行数
     - 行内容：包含每列的值(长度+字符串)
     - 更新操作额外字段：旧值信息(更新操作时)
   - **错误重试**：引入重试机制，失败后自动重试
     
   - **消息发送逻辑**：
     - 第一次发送数据时(FirstSend=true)，不等待确认就继续处理
     - 后续发送数据后(FirstSend=false)，标记为等待确认状态(WaitingAck=true)
     - 收到确认消息后，才可以继续发送下一批数据

   - **消息接收逻辑**：
     - 使用回调函数注册机制接收和处理确认消息
     - 使用正确的二进制协议格式读取消息长度和内容
     - 当接收到MRsprowmsg消息时，通知主循环继续处理

5. `updateTaskStatus()`: 更新任务状态，包括LastID(GTID)、FirstSend和WaitingAck信息
   - 全面记录任务状态，支持断点续传和消息确认机制

6. `MongoTaskStatus`: 包含IsRunning、LastID、Error、FirstSend和WaitingAck等任务状态信息
   - FirstSend：标记是否是首次发送数据
   - WaitingAck：标记是否正在等待确认

7. `registerAckHandler()` 和 `HandleAckMessage()`：
   - 注册和处理确认消息的回调函数
   - 支持不同消息类型的处理
   - 确保异步消息处理的线程安全

### 2.4.3 数据源同步的一致性处理机制

1. **消息确认机制统一**：
   - MongoDB和Elasticsearch使用相同的消息确认机制
   - 首次发送时不等待确认，后续发送必须等待确认
   - 确保数据按顺序被处理，避免数据丢失

2. **数据格式统一**：
   - 两种数据源发送的消息格式完全一致
   - 使用相同的消息类型和字段结构
   - 便于客户端使用统一方式处理不同来源的数据

3. **持续增量同步**：
   - 同步到最后一条数据后不退出，持续监听新数据
   - 使用适当的等待间隔，避免频繁空查询
   - 当无新数据时，MongoDB清理资源并等待
   - 当无新数据时，Elasticsearch清理scroll并等待
   - 保证任务持续运行，捕获所有数据变更

4. **断点续传一致性**：
   - MongoDB使用ObjectID作为GTID
   - Elasticsearch使用ScrollID和LastTimestamp作为断点
   - 两者都支持任务重启后从断点继续
   - 确保不会遗漏或重复处理数据

通过以上机制，系统实现了不同数据源的统一处理流程，确保数据同步的一致性、可靠性和持续性。客户端看到的是统一的数据流，无需关心底层数据源的差异。

### 2.4.4 TDengine同步函数工作流程

1. `StartTaskTD()`: 初始化TDengine同步任务，解析起始时间并启动工作协程
2. `syncTDTask()`: 处理TDengine同步任务的主要流程
   - 获取TDengine客户端
   - 基于时间戳查询数据
   - 持续监控新数据
3. `getTDClient()`: 获取或创建TDengine连接
4. `SyncTdTask()`: 实际查询并同步TDengine数据的函数
5. `sendSyncResult()`: 将处理后的数据发送到目标系统
6. `updateTaskStatus()`: 更新任务状态，包括LastTS(时间戳)信息
7. `TdTaskStatus`: 包含IsRunning、LastTS和Error等任务状态信息

### 2.4.5 PB级流式同步函数工作流程

1. `StreamTaskManager.StartStreamTask()`: 启动流式同步任务，初始化进度和状态
2. `loadCheckpoint()`: 尝试加载现有检查点，用于断点续传
3. 根据数据源类型选择相应的处理函数：
   - `processMongoStream()`: 处理MongoDB流式同步
   - `processTDEngineStream()`: 处理TDengine流式同步
   - `processESStream()`: 处理ES流式同步
4. `saveCheckpoint()`: 定期保存包含GTID的检查点，确保断点续传
5. `StreamTaskStatus`: 包含Progress、LastCheckpoint等状态信息
6. `Checkpoint`: 包含LastSequenceNum、BytesProcessed和Gtid等断点信息

以上函数实现了三种不同数据源的高效同步，并通过各自适合的GTID机制实现断点续传，确保数据的完整性、一致性和可恢复性。






