package main

import (
	"context"
	"fmt"
	"log"
	"time"

	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/bson/primitive"
	"go.mongodb.org/mongo-driver/mongo"
	"go.mongodb.org/mongo-driver/mongo/options"
)

// 专门用于调试 changeStream 问题的工具
func main() {
	// MongoDB 连接配置 - 请根据实际情况修改
	mongoHost := "localhost:27017"
	username := ""                  // 请修改为实际用户名，如果没有认证则留空
	password := ""                  // 请修改为实际密码，如果没有认证则留空
	authDatabase := "admin"         // 认证数据库，通常是 admin
	dbName := "test"               // 要测试的数据库
	collectionName := "test2"      // 要测试的集合

	// 构建带认证的 MongoDB URL
	var mongoURL string
	if username != "" && password != "" {
		mongoURL = fmt.Sprintf("mongodb://%s:%s@%s/%s", username, password, mongoHost, authDatabase)
	} else {
		mongoURL = fmt.Sprintf("mongodb://%s", mongoHost)
	}

	fmt.Println("=== MongoDB Change Stream 调试工具 ===")
	fmt.Printf("连接主机: %s\n", mongoHost)
	fmt.Printf("目标数据库: %s\n", dbName)
	fmt.Printf("目标集合: %s\n", collectionName)
	fmt.Println()

	// 连接 MongoDB
	ctx := context.Background()
	client, err := mongo.Connect(ctx, options.Client().ApplyURI(mongoURL))
	if err != nil {
		log.Fatalf("连接 MongoDB 失败: %v", err)
	}
	defer client.Disconnect(ctx)

	// 测试连接
	err = client.Ping(ctx, nil)
	if err != nil {
		log.Fatalf("MongoDB ping 失败: %v", err)
	}
	fmt.Println("✓ MongoDB 连接成功")

	// 1. 检查副本集状态
	fmt.Println("\n=== 检查副本集状态 ===")
	result := client.Database("admin").RunCommand(ctx, bson.D{{"isMaster", 1}})
	var isMasterResult bson.M
	if err := result.Decode(&isMasterResult); err != nil {
		fmt.Printf("❌ 检查副本集状态失败: %v\n", err)
		fmt.Println("这是导致 Change Stream 不工作的最常见原因！")
		return
	}

	if setName, exists := isMasterResult["setName"]; exists {
		fmt.Printf("✓ 副本集名称: %v\n", setName)
		if isPrimary, ok := isMasterResult["ismaster"].(bool); ok && isPrimary {
			fmt.Println("✓ 当前节点是主节点")
		} else {
			fmt.Println("⚠️  当前节点不是主节点")
		}
	} else {
		fmt.Println("❌ 未检测到副本集配置")
		fmt.Println("这是导致 Change Stream 不工作的根本原因！")
		fmt.Println()
		fmt.Println("解决方案:")
		fmt.Println("1. 停止 MongoDB: sudo systemctl stop mongod")
		fmt.Println("2. 修改配置文件 /etc/mongod.conf 添加:")
		fmt.Println("   replication:")
		fmt.Println("     replSetName: \"rs0\"")
		fmt.Println("3. 启动 MongoDB: sudo systemctl start mongod")
		fmt.Println("4. 初始化副本集: mongo --eval \"rs.initiate()\"")
		fmt.Println("5. 检查状态: mongo --eval \"rs.status()\"")
		return
	}

	// 2. 检查 oplog
	fmt.Println("\n=== 检查 Oplog 状态 ===")
	oplogCollection := client.Database("local").Collection("oplog.rs")
	oplogCount, err := oplogCollection.CountDocuments(ctx, bson.M{})
	if err != nil {
		fmt.Printf("❌ 检查 oplog 失败: %v\n", err)
	} else if oplogCount > 0 {
		fmt.Printf("✓ Oplog 可用，包含 %d 条记录\n", oplogCount)
	} else {
		fmt.Println("❌ Oplog 不可用或为空")
		return
	}

	// 3. 测试目标集合
	fmt.Printf("\n=== 检查目标集合 %s.%s ===\n", dbName, collectionName)
	collection := client.Database(dbName).Collection(collectionName)
	
	// 检查集合是否存在
	count, err := collection.CountDocuments(ctx, bson.M{})
	if err != nil {
		fmt.Printf("❌ 访问集合失败: %v\n", err)
		return
	}
	fmt.Printf("✓ 集合存在，包含 %d 条文档\n", count)

	// 如果集合为空，插入测试数据
	if count == 0 {
		fmt.Println("集合为空，插入测试数据...")
		testDocs := []interface{}{
			bson.M{"name": "test1", "value": 100, "created": time.Now()},
			bson.M{"name": "test2", "value": 200, "created": time.Now()},
		}
		_, err = collection.InsertMany(ctx, testDocs)
		if err != nil {
			fmt.Printf("❌ 插入测试数据失败: %v\n", err)
			return
		}
		fmt.Printf("✓ 插入了 %d 条测试数据\n", len(testDocs))
	}

	// 4. 测试 Change Stream 创建
	fmt.Println("\n=== 测试 Change Stream 创建 ===")
	
	// 设置 Change Stream 选项
	changeStreamOptions := options.ChangeStream()
	changeStreamOptions.SetFullDocument(options.UpdateLookup)
	changeStreamOptions.SetBatchSize(10)

	// 设置监听 update 和 delete 操作的 pipeline（与您的代码完全一致）
	pipeline := mongo.Pipeline{
		{{
			"$match", bson.D{{
				"operationType", bson.D{{
					"$in", bson.A{"update", "delete"},
				}},
			}},
		}},
	}

	fmt.Printf("Pipeline: %+v\n", pipeline)
	fmt.Printf("Options: FullDocument=%v, BatchSize=%v\n", 
		options.UpdateLookup, 10)

	// 创建 Change Stream
	changeStream, err := collection.Watch(ctx, pipeline, changeStreamOptions)
	if err != nil {
		fmt.Printf("❌ 创建 Change Stream 失败: %v\n", err)
		return
	}
	defer changeStream.Close(ctx)

	fmt.Println("✓ Change Stream 创建成功")
	fmt.Println("\n=== 开始监听变更事件 ===")
	fmt.Println("监听操作类型: update, delete")
	fmt.Println("将在10秒后自动执行测试操作...")

	// 启动一个 goroutine 来模拟数据变更
	go func() {
		time.Sleep(3 * time.Second)
		fmt.Println("\n--- 开始执行测试操作 ---")

		// 执行 update 操作
		filter := bson.M{"name": "test1"}
		update := bson.M{"$set": bson.M{"value": 150, "updated": time.Now()}}
		result, err := collection.UpdateOne(ctx, filter, update)
		if err != nil {
			fmt.Printf("❌ 更新操作失败: %v\n", err)
		} else {
			fmt.Printf("✓ 执行了 update 操作，匹配 %d 条，修改 %d 条\n", 
				result.MatchedCount, result.ModifiedCount)
		}

		time.Sleep(2 * time.Second)

		// 执行 delete 操作
		deleteFilter := bson.M{"name": "test2"}
		deleteResult, err := collection.DeleteOne(ctx, deleteFilter)
		if err != nil {
			fmt.Printf("❌ 删除操作失败: %v\n", err)
		} else {
			fmt.Printf("✓ 执行了 delete 操作，删除 %d 条记录\n", deleteResult.DeletedCount)
		}
	}()

	// 监听变更事件
	eventCount := 0
	timeout := time.After(15 * time.Second) // 15秒后超时

	for {
		select {
		case <-timeout:
			fmt.Println("\n=== 测试结束 ===")
			if eventCount == 0 {
				fmt.Println("❌ 未收到任何 Change Stream 事件")
				fmt.Println("\n可能的原因:")
				fmt.Println("1. MongoDB 未运行在副本集模式（最常见）")
				fmt.Println("2. Pipeline 过滤条件过于严格")
				fmt.Println("3. 权限不足")
				fmt.Println("4. 网络连接问题")
			} else {
				fmt.Printf("✓ 总共收到 %d 个事件，Change Stream 工作正常\n", eventCount)
			}
			return

		default:
			if changeStream.Next(ctx) {
				eventCount++
				var changeDoc bson.M
				if err := changeStream.Decode(&changeDoc); err != nil {
					fmt.Printf("❌ 解析变更事件失败: %v\n", err)
					continue
				}

				fmt.Printf("\n🎉 收到 Change Stream 事件 #%d\n", eventCount)
				fmt.Printf("操作类型: %v\n", changeDoc["operationType"])
				
				if ns, ok := changeDoc["ns"].(bson.M); ok {
					fmt.Printf("数据库: %v\n", ns["db"])
					fmt.Printf("集合: %v\n", ns["coll"])
				}

				if documentKey, ok := changeDoc["documentKey"].(bson.M); ok {
					if id, exists := documentKey["_id"]; exists {
						if objectID, ok := id.(primitive.ObjectID); ok {
							fmt.Printf("文档ID: %s\n", objectID.Hex())
						} else {
							fmt.Printf("文档ID: %v\n", id)
						}
					}
				}

				if resumeToken := changeStream.ResumeToken(); resumeToken != nil {
					fmt.Printf("Resume Token: 存在 (长度: %d)\n", len(resumeToken))
				}

			} else if err := changeStream.Err(); err != nil {
				fmt.Printf("❌ Change Stream 错误: %v\n", err)
				return
			}
		}
	}
}
